-- -*- haskell -*- 
--
-- Simple syntax highlighting for Latex source files
--
-- This is not intended to be a lexical analyser for
-- latex, merely good enough to provide some syntax
-- highlighting for latex source files.
--

{
#define NO_ALEX_CONTEXTS
{-# OPTIONS -w  #-}
module Yi.Lexer.Latex ( initState, alexScanToken, Token(..), HlState, tokenToText ) where
import Yi.Lexer.Alex
import Yi.Style
}

$special   = [\[\]\{\}\$\\\%\(\)\ ]
$textChar = [~$special $white]
$idchar = [a-zA-Z 0-9_\-]

@reservedid = begin|end|newcommand
@ident = $idchar+
@text = $textChar+

haskell :-

 "%"\-*[^\n]*                                { c $ Comment }
 $special                                    { cs $ \(c:_) -> Special c }
 \\"begin{"@ident"}"                         { cs $ \s -> Begin (drop 6 s) }
 \\"end{"@ident"}"                           { cs $ \s -> End (drop 4 s) }
 \\$special                                  { cs $ \(_:cs) -> Command cs }
 \\newcommand                                { c $ NewCommand }
 \\@ident                                    { cs $ \(_:cs) -> Command cs }
 @text                                       { c $ Text }


{

data Token = Comment | Text | Special !Char | Command !String | Begin !String | End !String | NewCommand  
  deriving (Eq, Show, Ord)

type HlState = Int

{- See Haskell.x which uses this to say whether we are in a
   comment (perhaps a nested comment) or not.
-}
stateToInit x = 0

initState :: HlState
initState = 0

tokenToText (Command "alpha") = Just "α"
tokenToText (Command "beta") = Just "β"
tokenToText (Command "gamma") = Just "γ"
tokenToText (Command "delta") = Just "δ"
tokenToText (Command "epsilon") = Just "∊"
tokenToText (Command "varepsilon") = Just "ε"
tokenToText (Command "zeta") = Just "ζ"
tokenToText (Command "eta") = Just "η"
tokenToText (Command "theta") = Just "θ"
tokenToText (Command "vartheta") = Just "ϑ"
tokenToText (Command "iota") = Just "ι"
tokenToText (Command "kappa") = Just "κ"
tokenToText (Command "lambda") = Just "λ"
tokenToText (Command "mu") = Just "μ"
tokenToText (Command "nu") = Just "ν"
tokenToText (Command "xi") = Just "ξ"
tokenToText (Command "pi") = Just "π"
tokenToText (Command "varpi") = Just "ϖ"
tokenToText (Command "rho") = Just "ρ"
tokenToText (Command "varrho") = Just "ϱ"
tokenToText (Command "sigma") = Just "σ"
tokenToText (Command "varsigma") = Just "ς"
tokenToText (Command "tau") = Just "τ"
tokenToText (Command "upsilon") = Just "υ"
tokenToText (Command "phi") = Just "φ"
tokenToText (Command "varphi") = Just "ϕ"
tokenToText (Command "chi") = Just "χ"
tokenToText (Command "psi") = Just "ψ"
tokenToText (Command "omega") = Just "ω"
tokenToText (Command "Gamma") = Just "Γ"
tokenToText (Command "Delta") = Just "Δ"
tokenToText (Command "Theta") = Just "Θ"
tokenToText (Command "Lambda") = Just "Λ"
tokenToText (Command "Xi") = Just "Ξ"
tokenToText (Command "Pi") = Just "Π"
tokenToText (Command "Upsilon") = Just "Υ"
tokenToText (Command "Phi") = Just "Φ"
tokenToText (Command "Psi") = Just "Ψ"
tokenToText (Command "Omega") = Just "Ω"
tokenToText (Command "leq") = Just "≤"
tokenToText (Command "ll") = Just "≪"
tokenToText (Command "prec") = Just "≺"
tokenToText (Command "preceq") = Just "≼"
tokenToText (Command "subset") = Just "⊂"
tokenToText (Command "subseteq") = Just "⊆"
tokenToText (Command "sqsubset") = Just "⊏"
tokenToText (Command "sqsubseteq") = Just "⊑"
tokenToText (Command "in") = Just "∈"
tokenToText (Command "vdash") = Just "⊢"
tokenToText (Command "mid") = Just "∣"
tokenToText (Command "smile") = Just "⌣"
tokenToText (Command "geq") = Just "≥"
tokenToText (Command "gg") = Just "≫"
tokenToText (Command "succ") = Just "≻"
tokenToText (Command "succeq") = Just "≽"
tokenToText (Command "supset") = Just "⊃"
tokenToText (Command "supseteq") = Just "⊇"
tokenToText (Command "sqsupset") = Just "⊐"
tokenToText (Command "sqsupseteq") = Just "⊒"
tokenToText (Command "ni") = Just "∋"
tokenToText (Command "dashv") = Just "⊣"
tokenToText (Command "parallel") = Just "∥"
tokenToText (Command "frown") = Just "⌢"
tokenToText (Command "notin") = Just "∉"
tokenToText (Command "equiv") = Just "≡"
tokenToText (Command "doteq") = Just "≐"
tokenToText (Command "sim") = Just "∼"
tokenToText (Command "simeq") = Just "≃"
tokenToText (Command "approx") = Just "≈"
tokenToText (Command "cong") = Just "≅"
tokenToText (Command "Join") = Just "⋈"
tokenToText (Command "bowtie") = Just "⋈"
tokenToText (Command "propto") = Just "∝"
tokenToText (Command "models") = Just "⊨"
tokenToText (Command "perp") = Just "⊥"
tokenToText (Command "asymp") = Just "≍"
tokenToText (Command "neq") = Just "≠"
tokenToText (Command "pm") = Just "±"
tokenToText (Command "cdot") = Just "⋅"
tokenToText (Command "times") = Just "×"
tokenToText (Command "cup") = Just "∪"
tokenToText (Command "sqcup") = Just "⊔"
tokenToText (Command "vee") = Just "∨"
tokenToText (Command "oplus") = Just "⊕"
tokenToText (Command "odot") = Just "⊙"
tokenToText (Command "otimes") = Just "⊗"
tokenToText (Command "bigtriangleup") = Just "△"
tokenToText (Command "lhd") = Just "⊲"
tokenToText (Command "unlhd") = Just "⊴"
tokenToText (Command "mp") = Just "∓"
tokenToText (Command "div") = Just "÷"
tokenToText (Command "setminus") = Just "∖"
tokenToText (Command "cap") = Just "∩"
tokenToText (Command "sqcap") = Just "⊓"
tokenToText (Command "wedge") = Just "∧"
tokenToText (Command "ominus") = Just "⊖"
tokenToText (Command "oslash") = Just "⊘"
tokenToText (Command "bigcirc") = Just "○"
tokenToText (Command "bigtriangledown") = Just "▽"
tokenToText (Command "rhd") = Just "⊳"
tokenToText (Command "unrhd") = Just "⊵"
tokenToText (Command "triangleleft") = Just "◁"
tokenToText (Command "triangleright") = Just "▷"
tokenToText (Command "star") = Just "⋆"
tokenToText (Command "ast") = Just "∗"
tokenToText (Command "circ") = Just "∘"
tokenToText (Command "bullet") = Just "∙"
tokenToText (Command "diamond") = Just "⋄"
tokenToText (Command "uplus") = Just "⊎"
tokenToText (Command "dagger") = Just "†"
tokenToText (Command "ddagger") = Just "‡"
tokenToText (Command "wr") = Just "≀"
tokenToText (Command "sum") = Just "∑"
tokenToText (Command "prod") = Just "∏"
tokenToText (Command "coprod") = Just "∐"
tokenToText (Command "int") = Just "∫"
tokenToText (Command "bigcup") = Just "⋃"
tokenToText (Command "bigcap") = Just "⋂"
tokenToText (Command "bigsqcup") = Just "⊔"
tokenToText (Command "oint") = Just "∮"
tokenToText (Command "bigvee") = Just "⋁"
tokenToText (Command "bigwedge") = Just "⋀"
tokenToText (Command "bigoplus") = Just "⊕"
tokenToText (Command "bigotimes") = Just "⊗"
tokenToText (Command "bigodot") = Just "⊙"
tokenToText (Command "biguplus") = Just "⊎"
tokenToText (Command "leftarrow") = Just "←"
tokenToText (Command "rightarrow") = Just "→"
tokenToText (Command "leftrightarrow") = Just "↔"
tokenToText (Command "Leftarrow") = Just "⇐"
tokenToText (Command "Rightarrow") = Just "⇒"
tokenToText (Command "Leftrightarrow") = Just "⇔"
tokenToText (Command "mapsto") = Just "↦"
tokenToText (Command "hookleftarrow") = Just "↩"
tokenToText (Command "leftharpoonup") = Just "↼"
tokenToText (Command "leftharpoondown") = Just "↽"
tokenToText (Command "hookrightarrow") = Just "↪"
tokenToText (Command "rightharpoonup") = Just "⇀"
tokenToText (Command "rightharpoondown") = Just "⇁"
tokenToText (Command "longleftarrow") = Just "←"
tokenToText (Command "longrightarrow") = Just "→"
tokenToText (Command "longleftrightarrow") = Just "↔"
tokenToText (Command "Longleftarrow") = Just "⇐"
tokenToText (Command "Longrightarrow") = Just "⇒"
tokenToText (Command "Longleftrightarrow") = Just "⇔"
tokenToText (Command "longmapsto") = Just "⇖"
tokenToText (Command "uparrow") = Just "↑"
tokenToText (Command "downarrow") = Just "↓"
tokenToText (Command "updownarrow") = Just "↕"
tokenToText (Command "Uparrow") = Just "⇑"
tokenToText (Command "Downarrow") = Just "⇓"
tokenToText (Command "Updownarrow") = Just "⇕"
tokenToText (Command "nearrow") = Just "↗"
tokenToText (Command "searrow") = Just "↘"
tokenToText (Command "swarrow") = Just "↙"
tokenToText (Command "nwarrow") = Just "↖"
tokenToText (Command "leadsto") = Just "↝"
tokenToText (Command "dots") = Just "…"
tokenToText (Command "cdots") = Just "⋯"
tokenToText (Command "vdots") = Just "⋮"
tokenToText (Command "ddots") = Just "⋱"
tokenToText (Command "hbar") = Just "ℏ"
tokenToText (Command "ell") = Just "ℓ"
tokenToText (Command "Re") = Just "ℜ"
tokenToText (Command "Im") = Just "ℑ"
tokenToText (Command "aleph") = Just "א"
tokenToText (Command "wp") = Just "℘"
tokenToText (Command "forall") = Just "∀"
tokenToText (Command "exists") = Just "∃"
tokenToText (Command "mho") = Just "℧"
tokenToText (Command "partial") = Just "∂"
tokenToText (Command "prime") = Just "′"
tokenToText (Command "emptyset") = Just "∅"
tokenToText (Command "infty") = Just "∞"
tokenToText (Command "nabla") = Just "∇"
tokenToText (Command "triangle") = Just "△"
tokenToText (Command "Box") = Just "□"
tokenToText (Command "Diamond") = Just "◇"
tokenToText (Command "bot") = Just "⊥"
tokenToText (Command "top") = Just "⊤"
tokenToText (Command "angle") = Just "∠"
tokenToText (Command "surd") = Just "√"
tokenToText (Command "diamondsuit") = Just "♢"
tokenToText (Command "heartsuit") = Just "♡"
tokenToText (Command "clubsuit") = Just "♣"
tokenToText (Command "spadesuit") = Just "♠"
tokenToText (Command "neg") = Just "¬"
tokenToText (Command "flat") = Just "♭"
tokenToText (Command "natural") = Just "♮"
tokenToText (Command "sharp") = Just "♯"
tokenToText (Command "digamma") = Just "Ϝ"
tokenToText (Command "varkappa") = Just "ϰ"
tokenToText (Command "beth") = Just "ב"
tokenToText (Command "daleth") = Just "ד"
tokenToText (Command "gimel") = Just "ג"
tokenToText (Command "lessdot") = Just "⋖"
tokenToText (Command "leqslant") = Just "≤"
tokenToText (Command "leqq") = Just "≦"
tokenToText (Command "lll") = Just "⋘"
tokenToText (Command "lesssim") = Just "≲"
tokenToText (Command "lessgtr") = Just "≶"
tokenToText (Command "lesseqgtr") = Just "⋚"
tokenToText (Command "preccurlyeq") = Just "≼"
tokenToText (Command "curlyeqprec") = Just "⋞"
tokenToText (Command "precsim") = Just "≾"
tokenToText (Command "Subset") = Just "⋐"
tokenToText (Command "sqsubset") = Just "⊏"
tokenToText (Command "therefore") = Just "∴"
tokenToText (Command "smallsmile") = Just "⌣"
tokenToText (Command "vartriangleleft") = Just "⊲"
tokenToText (Command "trianglelefteq") = Just "⊴"
tokenToText (Command "gtrdot") = Just "⋗"
tokenToText (Command "geqq") = Just "≧"
tokenToText (Command "ggg") = Just "⋙"
tokenToText (Command "gtrsim") = Just "≳"
tokenToText (Command "gtrless") = Just "≷"
tokenToText (Command "gtreqless") = Just "⋛"
tokenToText (Command "succcurlyeq") = Just "≽"
tokenToText (Command "curlyeqsucc") = Just "⋟"
tokenToText (Command "succsim") = Just "≿"
tokenToText (Command "Supset") = Just "⋑"
tokenToText (Command "sqsupset") = Just "⊐"
tokenToText (Command "because") = Just "∵"
tokenToText (Command "shortparallel") = Just "∥"
tokenToText (Command "smallfrown") = Just "⌢"
tokenToText (Command "vartriangleright") = Just "⊳"
tokenToText (Command "trianglerighteq") = Just "⊵"
tokenToText (Command "doteqdot") = Just "≑"
tokenToText (Command "risingdotseq") = Just "≓"
tokenToText (Command "fallingdotseq") = Just "≒"
tokenToText (Command "eqcirc") = Just "≖"
tokenToText (Command "circeq") = Just "≗"
tokenToText (Command "triangleq") = Just "≜"
tokenToText (Command "bumpeq") = Just "≏"
tokenToText (Command "Bumpeq") = Just "≎"
tokenToText (Command "thicksim") = Just "∼"
tokenToText (Command "thickapprox") = Just "≈"
tokenToText (Command "approxeq") = Just "≊"
tokenToText (Command "backsim") = Just "∽"
tokenToText (Command "vDash") = Just "⊨"
tokenToText (Command "Vdash") = Just "⊩"
tokenToText (Command "Vvdash") = Just "⊪"
tokenToText (Command "backepsilon") = Just "∍"
tokenToText (Command "varpropto") = Just "∝"
tokenToText (Command "between") = Just "≬"
tokenToText (Command "pitchfork") = Just "⋔"
tokenToText (Command "blacktriangleleft") = Just "◀"
tokenToText (Command "blacktriangleright") = Just "▷"
tokenToText (Command "dashleftarrow") = Just "⇠"
tokenToText (Command "leftleftarrows") = Just "⇇"
tokenToText (Command "leftrightarrows") = Just "⇆"
tokenToText (Command "Lleftarrow") = Just "⇚"
tokenToText (Command "twoheadleftarrow") = Just "↞"
tokenToText (Command "leftarrowtail") = Just "↢"
tokenToText (Command "leftrightharpoons") = Just "⇋"
tokenToText (Command "Lsh") = Just "↰"
tokenToText (Command "looparrowleft") = Just "↫"
tokenToText (Command "curvearrowleft") = Just "↶"
tokenToText (Command "circlearrowleft") = Just "↺"
tokenToText (Command "dashrightarrow") = Just "⇢"
tokenToText (Command "rightrightarrows") = Just "⇉"
tokenToText (Command "rightleftarrows") = Just "⇄"
tokenToText (Command "Rrightarrow") = Just "⇛"
tokenToText (Command "twoheadrightarrow") = Just "↠"
tokenToText (Command "rightarrowtail") = Just "↣"
tokenToText (Command "rightleftharpoons") = Just "⇌"
tokenToText (Command "Rsh") = Just "↱"
tokenToText (Command "looparrowright") = Just "↬"
tokenToText (Command "curvearrowright") = Just "↷"
tokenToText (Command "circlearrowright") = Just "↻"
tokenToText (Command "multimap") = Just "⊸"
tokenToText (Command "upuparrows") = Just "⇈"
tokenToText (Command "downdownarrows") = Just "⇊"
tokenToText (Command "upharpoonleft") = Just "↿"
tokenToText (Command "upharpoonright") = Just "↾"
tokenToText (Command "downharpoonleft") = Just "⇃"
tokenToText (Command "downharpoonright") = Just "⇂"
tokenToText (Command "rightsquigarrow") = Just "⇝"
tokenToText (Command "leftrightsquigarrow") = Just "↭"
tokenToText (Command "dotplus") = Just "∔"
tokenToText (Command "ltimes") = Just "⋉"
tokenToText (Command "Cup") = Just "⋓"
tokenToText (Command "veebar") = Just "⊻"
tokenToText (Command "boxplus") = Just "⊞"
tokenToText (Command "boxtimes") = Just "⊠"
tokenToText (Command "leftthreetimes") = Just "⋋"
tokenToText (Command "curlyvee") = Just "⋎"
tokenToText (Command "centerdot") = Just "⋅"
tokenToText (Command "rtimes") = Just "⋈"
tokenToText (Command "Cap") = Just "⋒"
tokenToText (Command "barwedge") = Just "⊼"
tokenToText (Command "boxminus") = Just "⊟"
tokenToText (Command "boxdot") = Just "⊡"
tokenToText (Command "rightthreetimes") = Just "⋌"
tokenToText (Command "curlywedge") = Just "⋏"
tokenToText (Command "intercal") = Just "⊺"
tokenToText (Command "divideontimes") = Just "⋇"
tokenToText (Command "smallsetminus") = Just "∖"
tokenToText (Command "circleddash") = Just "⊝"
tokenToText (Command "circledcirc") = Just "⊚"
tokenToText (Command "circledast") = Just "⊛"
tokenToText (Command "hbar") = Just "ℏ"
tokenToText (Command "hslash") = Just "ℏ"
tokenToText (Command "square") = Just "□"
tokenToText (Command "blacksquare") = Just "■"
tokenToText (Command "circledS") = Just "Ⓢ"
tokenToText (Command "vartriangle") = Just "△"
tokenToText (Command "blacktriangle") = Just "▲"
tokenToText (Command "complement") = Just "∁"
tokenToText (Command "triangledown") = Just "▽"
tokenToText (Command "blacktriangledown") = Just "▼"
tokenToText (Command "lozenge") = Just "◊"
tokenToText (Command "blacklozenge") = Just "◆"
tokenToText (Command "bigstar") = Just "★"
tokenToText (Command "angle") = Just "∠"
tokenToText (Command "measuredangle") = Just "∡"
tokenToText (Command "sphericalangle") = Just "∢"
tokenToText (Command "backprime") = Just "‵"
tokenToText (Command "nexists") = Just "∄"
tokenToText (Command "Finv") = Just "Ⅎ"
tokenToText (Command "varnothing") = Just "∅"
tokenToText (Command "eth") = Just "ð"
tokenToText (Command "mho") = Just "℧"
tokenToText (Command "AA") = Just "Å"
tokenToText (Command "euro") = Just "€"
tokenToText (Command "degree") = Just "°"
tokenToText (Command "lnot") = Just "¬"
tokenToText (Command "textdegree") = Just "°"
tokenToText (Command "textlnot") = Just "¬"
tokenToText (Command "textsurd") = Just "√"
tokenToText (Command "textdiv") = Just "÷"
tokenToText (Command "textpm") = Just "±"
tokenToText _                  = Nothing

#include "common.hsinc"
}
