/************************************************************************/
/*                                                                      */
/*    vspline - a set of generic tools for creation and evaluation      */
/*              of uniform b-splines                                    */
/*                                                                      */
/*            Copyright 2015 - 2017 by Kay F. Jahnke                    */
/*                                                                      */
/*    The git repository for this software is at                        */
/*                                                                      */
/*    https://bitbucket.org/kfj/vspline                                 */
/*                                                                      */
/*    Please direct questions, bug reports, and contributions to        */
/*                                                                      */
/*    kfjahnke+vspline@gmail.com                                        */
/*                                                                      */
/*    Permission is hereby granted, free of charge, to any person       */
/*    obtaining a copy of this software and associated documentation    */
/*    files (the "Software"), to deal in the Software without           */
/*    restriction, including without limitation the rights to use,      */
/*    copy, modify, merge, publish, distribute, sublicense, and/or      */
/*    sell copies of the Software, and to permit persons to whom the    */
/*    Software is furnished to do so, subject to the following          */
/*    conditions:                                                       */
/*                                                                      */
/*    The above copyright notice and this permission notice shall be    */
/*    included in all copies or substantial portions of the             */
/*    Software.                                                         */
/*                                                                      */
/*    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND    */
/*    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES   */
/*    OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND          */
/*    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT       */
/*    HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,      */
/*    WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING      */
/*    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR     */
/*    OTHER DEALINGS IN THE SOFTWARE.                                   */
/*                                                                      */
/************************************************************************/

/*! \file common.h

    \brief definitions common to all files in this project, utility code
    
    This file contains
    
    - some common enums and strings

    - a traits class fixing the simdized types used for vectorized code
      and some additional type inference used mainly for unary functors
    
    - exceptions used throughout vspline
*/

#ifndef VSPLINE_COMMON
#define VSPLINE_COMMON

#include <vigra/multi_array.hxx>
#include <vigra/tinyvector.hxx>

#ifdef USE_VC

#include <Vc/Vc>

#define VECTOR_TYPE Vc::SimdArray
#define DEFAULT_RSIZE Vc::Vector < ET<T> > :: Size
#define DEFAULT_VSIZE 2 * DEFAULT_RSIZE

#else

#define DEFAULT_RSIZE 1
#define DEFAULT_VSIZE 1

#endif // #ifdef USE_VC

namespace vspline
{

// this enum will hold true or false, depending on whether the
// translation unit including this header was compiled with USE_VC
// defined or not.

enum { vc_in_use =
#ifdef USE_VC
  true
#else
  false
#endif
} ;
  
/// This enumeration is used for codes connected to boundary conditions. There are
/// two aspects to boundary conditions: During prefiltering, if the implicit scheme is used,
/// the initial causal and anticausal coefficients have to be calculated in a way specific to
/// the chosen boundary conditions. Bracing, both before prefiltering when using the explicit
/// scheme, and after prefiltering when using the implicit scheme, also needs these codes to
/// pick the appropriate extrapolation code to extend the knot point data/coefficients beyond
/// the core array.

typedef enum { 
  MIRROR ,    ///< mirror on the bounds, so that f(-x) == f(x)
  PERIODIC,   ///< periodic boundary conditions
  REFLECT ,   ///< reflect, so  that f(-1) == f(0) (mirror between bounds)
  NATURAL,    ///< natural boundary conditions, f(-x) + f(x) == 2 * f(0)
  CONSTANT ,  ///< clamp. used for framing, with explicit prefilter scheme
  ZEROPAD ,   ///< used for boundary condition, bracing
  IDENTITY ,  ///< used as solver argument, mostly internal use
  GUESS ,     ///< used with EXPLICIT scheme to keep margin errors low
  SPHERICAL , ///< use for spherical panoramas, y axis
} bc_code;

/// bc_name is for diagnostic output of bc codes

const std::string bc_name[] =
{
  "MIRROR   " ,
  "PERIODIC ",
  "REFLECT  " ,
  "NATURAL  ",
  "CONSTANT " ,
  "ZEROPAD  " ,
  "IDENTITY " ,
  "GUESS    " ,
  "SPHERICAL" ,
} ;

/// using definition for the 'elementary type' of a type via vigra's
/// ExpandElementResult mechanism.

template < class T >
using ET = typename vigra::ExpandElementResult < T > :: type ;

/// unwrapping 'anything' produces the argument unchanged

template < class in_type >
in_type unwrap ( const in_type & in )
{
  return in ;
}

/// but unwrapping a TinyVector with just one element produces the
/// contained object

template < class T >
T unwrap ( const vigra::TinyVector < T , 1 > & in )
{
  return in[0] ;
}

/// wrapping 'anything' packages 'in' in a TinyVector with one element

template < class T >
vigra::TinyVector < T , 1 > wrap ( const T & in )
{
  return vigra::TinyVector < T , 1 > ( in ) ;
}

/// but 'wrapping' a TinyVector produces the TinyVector itself, since
/// it is 'already wrapped'.

template < class T , int N >
vigra::TinyVector < T , N > wrap
  ( const vigra::TinyVector < T , N > & in )
{
  return in ;
}

// if Vc isn't used, we nevertheless define is_vectorizable and vector_traits
// to provide a common interface for enquiry. This way, we have a uniform
// interface for inquiry about vectorization, which simply collapses to
// providing the unvectorized types when queried without Vc in use. And,
// additionally, in code using Vc we can switch to fallback code on
// inspection of vsize, if we find it it 1: in these cases, we can route
// the code so that vector code is avoided.

template < typename T > class is_vectorizable : public std::false_type {} ;

template < typename VT > class is_simd_type : public std::false_type {} ;

#ifdef USE_VC

#ifdef HAVE_IS_SIMD_VECTOR

// this test yields std::true_type for any T which can be vectorized by Vc,
// std::false_type for other T. TODO: not commonly available!

template < class T > using is_vectorizable =
typename Vc::is_simd_vector < Vc::Vector < T > > :: type ;

#else

// is_simd_vector hopefully comes with future Vc versions,
// but for the time being, we have to be explicit.
// in Vc ML discussion mkretz states that the set of types Vc can vectorize
// (with 1.3) is consistent throughout all ABIs, so we can just
// list the acceptable types without having to take the ABI int account.

template<> class is_vectorizable<float>          : public std::true_type {} ;
template<> class is_vectorizable<double>         : public std::true_type {} ;
template<> class is_vectorizable<int>            : public std::true_type {} ;
template<> class is_vectorizable<unsigned int>   : public std::true_type {} ;
template<> class is_vectorizable<short>          : public std::true_type {} ;
template<> class is_vectorizable<unsigned short> : public std::true_type {} ;

template < class candidate >
using is_vcable =
typename std::conditional < is_vectorizable < candidate > :: value ,
                            std::true_type ,
                            std::false_type > :: type ;
                            
#endif // HAVE_IS_SIMD_VECTOR

// test if a given type VT is a Vc::Vector or Vc::SimdArray

template < class T >
class is_simd_type < Vc::Vector < T > >
  : public std::true_type {} ;

template < class T , int _vsize >
class is_simd_type < Vc::SimdArray < T , _vsize > >
  : public std::true_type {} ;

// to code vector_traits, we need a few helper types.
// note how I pass 'vectorizable', and SZ, below, as types, even though
// it would be natural to pass a bool/an int. This is due to a bug in
// g++, which produces a failure to specialize in these cases.
// TODO not really for public use. hide?

// default vectorized type. This is where an actual Vc type is used
// to produce a SIMD type for vectorizable T. This SIMD type is not
// actally used, we only use it's size                            
                            
template < typename T , typename vectorizable >
struct dvtype
{
  typedef T type ;
  enum { size = 1 } ;
} ;

template < typename T >
struct dvtype < T , std::true_type >
{
  typedef Vc::SimdArray < T , 2 * Vc::Vector < T > :: size() > type ;
  enum { size = type::size() } ;
} ;

// ditto, but inferring rsize, which is used in filtering code.

template < typename T , typename vectorizable = std::false_type >
struct drtype
{
  typedef T type ;
  enum { size = 1 } ;
} ;

template < typename T >
struct drtype < T , std::true_type >
{
  typedef Vc::Vector < T > type ;
  enum { size = type::size() } ;
} ;

// with a given vector width, we construct the appropriate SIMD type.
// with vsize==1 (below) this collopses to T itself

template < typename T , typename SZ >
struct vtype
{
  enum { size = SZ::value } ;
  typedef Vc::SimdArray < T , size > type ;
  typedef typename type::IndexType index_type ;
  static index_type IndexesFromZero() { return index_type::IndexesFromZero() ; }
} ;

template < typename T >
struct vtype < T , std::integral_constant<int,1> >
{
  typedef T type ;
  enum { size = 1 } ;
  typedef int index_type ;
  static int IndexesFromZero() { return 0 ; }
} ;

/// struct vector_traits is a traits class fixing the types used for
/// vectorized code in vspline.

template < typename T , int _vsize = 0 , int _rsize = 0 >
struct vector_traits
{
  // first, analyze T: how many channels/dimensions does it have,
  // and what is it's elementary type? We rely on vigra's ExpandElementResult
  // mechanism here, which allows us to uniformly handle all types known to
  // this mechanism - and since it is a traits class, it can be extended to
  // handle more types if necessary.
  
  enum { dimension = vigra::ExpandElementResult < T > :: size } ;

  typedef typename vigra::ExpandElementResult < T > :: type ele_type ;

  // now we take a look at the elementary type. Can it be vectorized?
  // is_vectorizable yields a type directly inheriting from either
  // std::true_type or std::false_type. We want a type here, since we
  // need one to specialize dvtype and drtype with a type rather than
  // with a boolean, which would be more natural - but g++ fails to
  // perform the specialization if we use a non-type template argument
  // here, so I'm working around a compiler bug.
  
  typedef is_vcable < ele_type > isv ;
  
  enum { size = isv::value
                ? _vsize == 0
                  ? dvtype < ele_type , isv > :: size
                  : _vsize
                : 1 ,
         rsize = isv::value
                 ? _vsize == 0
                   ? drtype < ele_type , isv > :: size
                   : _rsize
                 : 1 } ;

  // the same compiler bug keeps me from specializing vtype with an int
  // as second template argument. Again, I wrap the argument in a
  // class type template argument to assure correct specialization:

  typedef typename std::integral_constant < int , size > sz_t ;
  
  // now the type 'ele_v', the simdized elementary type, can be produced.
  // This may 'collapse' to ele_type itself, if size if 1
  
  typedef typename vtype < ele_type , sz_t > :: type ele_v ;

  // The next two types are TinyVectors of ele_type and ele_v,
  // respectively. This is 'nice to have' in case code needs TinyVectors
  // even if the data are 1D.

  typedef vigra::TinyVector < ele_type , dimension > nd_ele_type ;
  
  typedef vigra::TinyVector < ele_v , dimension > nd_ele_v ;
  
  // syn_type, which isn't (currently) used outside, is the 'synthetic'
  // vectorized type. It is derived from ele_v or nd_ele_v, which are
  // built from the elementary type - hence the term 'synthetic'.

  typedef typename std::conditional < std::is_same < T , ele_type > :: value ,
                                      ele_v ,
                                      nd_ele_v > :: type syn_type ;
       
  // finally, we look at a special case, namely size==1. This occurs if
  // the elementary type can't be vectorized, or if _vsize is passed as
  // 1 in the first place. If this is the case, we want to use T itself
  // as the 'final' type, rather than the synthetic type which we have
  // built above. Hence this last step:
                                      
  typedef typename std::conditional
    < size == 1 , T , syn_type > :: type type ;
  
  typedef typename std::conditional
    < size == 1 , nd_ele_type , nd_ele_v > :: type tv_type ;
  
  // Some code in vspline requires indexing of either single or SIMD values.
  // So we provide a type for indexing and a static function providing
  // canonical indices for this type.

  typedef typename vtype < ele_type , sz_t > :: index_type
          index_type ;
  
  typedef typename vigra::TinyVector < index_type , dimension > 
          nd_index_type ;

  static index_type IndexesFromZero()
    { return vtype < ele_type , sz_t > :: IndexesFromZero() ; }
} ;

#else // #ifdef USE_VC

template < typename T , int _vsize = 0 >
struct vector_traits
{
  enum { size = 1 , rsize = 1 } ;

  enum { dimension = vigra::ExpandElementResult < T > :: size } ;

  typedef typename vigra::ExpandElementResult < T > :: type ele_type ;

  typedef vigra::TinyVector < ele_type , dimension > nd_ele_type ;
  
  typedef ele_type ele_v ;
  
  typedef nd_ele_type nd_ele_v ;
  
  typedef T type ;
} ;

#endif

// TODO The exceptions need some work. My use of exceptions is a bit sketchy...

/// for interfaces which need specific implementations we use:

struct not_implemented
: std::invalid_argument
{
  not_implemented ( const char * msg )
  : std::invalid_argument ( msg ) { }  ;
} ;

/// dimension-mismatch is thrown if two arrays have different dimensions
/// which should have the same dimensions.

struct dimension_mismatch
: std::invalid_argument
{
  dimension_mismatch ( const char * msg )
  : std::invalid_argument ( msg ) { }  ;
} ;

/// shape mismatch is the exception which is thrown if the shapes of
/// an input array and an output array do not match.

struct shape_mismatch
: std::invalid_argument
{
  shape_mismatch  ( const char * msg )
  : std::invalid_argument ( msg ) { }  ;
} ;

/// exception which is thrown if an opertion is requested which vspline
/// does not support

struct not_supported
: std::invalid_argument
{
  not_supported  ( const char * msg )
  : std::invalid_argument ( msg ) { }  ;
} ;

/// out_of_bounds is thrown by mapping mode REJECT for out-of-bounds coordinates
/// this exception is left without a message, it only has a very specific application,
/// and there it may be thrown often, so we don't want anything slowing it down.

struct out_of_bounds
{
} ;

/// exception which is thrown when an assiging an rvalue which is larger than
/// what the lvalue can hold

struct numeric_overflow
: std::invalid_argument
{
  numeric_overflow  ( const char * msg )
  : std::invalid_argument ( msg ) { }  ;
} ;

} ; // end of namespace vspline

#endif // VSPLINE_COMMON
