/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

//
// Quit game procedure
//

#include "ags/shared/core/platform.h"
#include "ags/engine/ac/cd_audio.h"
#include "ags/engine/ac/game_setup.h"
#include "ags/shared/ac/game_setup_struct.h"
#include "ags/engine/ac/game_state.h"
#include "ags/engine/ac/room_status.h"
#include "ags/engine/ac/route_finder.h"
#include "ags/engine/ac/translation.h"
#include "ags/engine/debugging/ags_editor_debugger.h"
#include "ags/engine/debugging/debug_log.h"
#include "ags/engine/debugging/debugger.h"
#include "ags/shared/debugging/out.h"
#include "ags/shared/font/fonts.h"
#include "ags/engine/main/config.h"
#include "ags/engine/main/engine.h"
#include "ags/engine/main/main.h"
#include "ags/engine/main/quit.h"
#include "ags/shared/ac/sprite_cache.h"
#include "ags/engine/gfx/graphics_driver.h"
#include "ags/shared/gfx/bitmap.h"
#include "ags/shared/core/asset_manager.h"
#include "ags/engine/platform/base/ags_platform_driver.h"
#include "ags/plugins/plugin_engine.h"
#include "ags/engine/media/audio/audio_system.h"
#include "ags/globals.h"
#include "ags/ags.h"

namespace AGS3 {

using namespace AGS::Shared;
using namespace AGS::Engine;

void quit_tell_editor_debugger(const String &qmsg, QuitReason qreason) {
	if (_G(editor_debugging_initialized)) {
		if (qreason & kQuitKind_GameException)
			_G(handledErrorInEditor) = send_exception_to_editor(qmsg.GetCStr());
		send_message_to_editor("EXIT");
		_G(editor_debugger)->Shutdown();
	}
}

void quit_stop_cd() {
	if (_G(need_to_stop_cd))
		cd_manager(3, 0);
}

void quit_shutdown_scripts() {
	ccUnregisterAllObjects();
}

void quit_check_dynamic_sprites(QuitReason qreason) {
	if ((qreason & kQuitKind_NormalExit) && (_G(check_dynamic_sprites_at_exit)) &&
	        (_GP(game).options[OPT_DEBUGMODE] != 0)) {
		// game exiting normally -- make sure the dynamic sprites
		// have been deleted
		for (size_t i = 1; i < _GP(spriteset).GetSpriteSlotCount(); i++) {
			if (_GP(game).SpriteInfos[i].Flags & SPF_DYNAMICALLOC)
				debug_script_warn("Dynamic sprite %d was never deleted", i);
		}
	}
}

void quit_shutdown_platform(QuitReason qreason) {
	// Be sure to unlock mouse on exit, or users will hate us
	_G(platform)->UnlockMouse();
	_G(platform)->AboutToQuitGame();

	_G(our_eip) = 9016;

	pl_stop_plugins();

	quit_check_dynamic_sprites(qreason);

	_G(platform)->FinishedUsingGraphicsMode();

	if (_G(use_cdplayer))
		_G(platform)->ShutdownCDPlayer();
}

void quit_shutdown_audio() {
	_G(our_eip) = 9917;
	_GP(game).options[OPT_CROSSFADEMUSIC] = 0;
	shutdown_sound();
}

QuitReason quit_check_for_error_state(const char *&qmsg, String &alertis) {
	if (qmsg[0] == '|') {
		return kQuit_GameRequest;
	} else if (qmsg[0] == '!') {
		QuitReason qreason;
		qmsg++;

		if (qmsg[0] == '|') {
			qreason = kQuit_UserAbort;
			alertis = "Abort key pressed.\n\n";
		} else if (qmsg[0] == '?') {
			qmsg++;
			qreason = kQuit_ScriptAbort;
			alertis = "A fatal error has been generated by the script using the AbortGame function. Please contact the game author for support.\n\n";
		} else {
			qreason = kQuit_GameError;
			alertis.Format("An error has occurred. Please contact the game author for support, as this "
			               "is likely to be a scripting error and not a bug in AGS.\n"
			               "(ACI version %s)\n\n", _G(EngineVersion).LongString.GetCStr());
		}

		alertis.Append(get_cur_script(5));

		if (qreason != kQuit_UserAbort)
			alertis.Append("\nError: ");
		else
			qmsg = "";
		return qreason;
	} else if (qmsg[0] == '%') {
		qmsg++;
		alertis.Format("A warning has been generated. This is not normally fatal, but you have selected "
		               "to treat warnings as errors.\n"
		               "(ACI version %s)\n\n%s\n", _G(EngineVersion).LongString.GetCStr(), get_cur_script(5).GetCStr());
		return kQuit_GameWarning;
	} else {
		alertis.Format("An internal error has occurred. Please note down the following information.\n"
		               "(ACI version %s)\n"
		               "\nError: ", _G(EngineVersion).LongString.GetCStr());
		return kQuit_FatalError;
	}
}

void quit_message_on_exit(const String &qmsg, String &alertis, QuitReason qreason) {
	// successful exit or user abort displays no messages
	if ((qreason & (kQuitKind_NormalExit | kQuit_UserAbort)) == 0 && !_G(handledErrorInEditor)) {
		// Display the message (at this point the window still exists)
		sprintf(_G(pexbuf), "%s\n", qmsg.GetCStr());
		alertis.Append(_G(pexbuf));
		_G(platform)->DisplayAlert("%s", alertis.GetCStr());
	}
}

void quit_release_data() {
	resetRoomStatuses();
	_GP(thisroom).Free();
	_GP(play).Free();
	_GP(AssetMgr).reset();
}

void quit_delete_temp_files() {
#ifdef TODO
	al_ffblk    dfb;
	int dun = al_findfirst("~ac*.tmp", &dfb, FA_SEARCH);
	while (!dun) {
		File::DeleteFile(dfb.name);
		dun = al_findnext(&dfb);
	}
	al_findclose(&dfb);
#endif
}

// quit - exits the engine, shutting down everything gracefully
// The parameter is the message to print. If this message begins with
// an '!' character, then it is printed as a "contact game author" error.
// If it begins with a '|' then it is treated as a "thanks for playing" type
// message. If it begins with anything else, it is treated as an internal
// error.
// "!|" is a special code used to mean that the player has aborted (Alt+X)
void quit(const char *quitmsg) {
	if (!_G(abort_engine)) {
		strncpy(_G(quit_message), quitmsg, sizeof(_G(quit_message)) - 1);
		_G(quit_message)[sizeof(_G(quit_message)) - 1] = '\0';
		_G(abort_engine) = true;
	}
}

void quit_free() {
	String alertis;
	if (strlen(_G(quit_message)) == 0)
		strcpy(_G(quit_message), "|bye!");

	const char *quitmsg = _G(quit_message);
	QuitReason qreason = quit_check_for_error_state(quitmsg, alertis);

	if (qreason & kQuitKind_NormalExit)
		save_config_file();

	_G(handledErrorInEditor) = false;

	quit_tell_editor_debugger(_G(quit_message), qreason);

	_G(our_eip) = 9900;

	quit_stop_cd();

	_G(our_eip) = 9020;

	quit_shutdown_scripts();

	quit_shutdown_platform(qreason);

	_G(our_eip) = 9019;

	quit_shutdown_audio();

	_G(our_eip) = 9901;

	shutdown_font_renderer();
	_G(our_eip) = 9902;

	_GP(spriteset).Reset();

	_G(our_eip) = 9907;

	close_translation();

	_G(our_eip) = 9908;

	shutdown_pathfinder();

	engine_shutdown_gfxmode();

	quit_message_on_exit(quitmsg, alertis, qreason);

	quit_release_data();

	// release backed library
	// WARNING: no Allegro objects should remain in memory after this,
	// if their destruction is called later, program will crash!
	allegro_exit();

	_G(platform)->PostAllegroExit();

	_G(our_eip) = 9903;

	quit_delete_temp_files();

	_G(proper_exit) = 1;

	Debug::Printf(kDbgMsg_Alert, "***** ENGINE HAS SHUTDOWN");

	shutdown_debug();

	_G(our_eip) = 9904;
}

} // namespace AGS3
