/******************************************************************************
*                             plasma-toggle_compositing                       *
*******************************************************************************
*                                                                             *
*            Copyright (C) 2008 John Varouhakis                               *
*                                                                             *
*                                                                             *
*   This program is free software; you can redistribute it and/or modify      *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 2 of the License, or         *
*   (at your option) any later version.                                       *
*                                                                             *
*   This program is distributed in the hope that it will be useful,           *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License         *
*   along with this program; if not, write to the Free Software               *
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA  *
*                                                                             *
*                                                                             *
*                                                                             *
*   For further information contact me at johnvarouhakis@gmail.com            *
******************************************************************************/


#include "toggle_compositing.hpp"
#include "kwin_interface.h"

#include <Plasma/IconWidget>
#include <Plasma/ToolTipManager>

#include <ksharedconfig.h>
#include <kconfig.h>

#include <KIcon>
#include <KIconLoader>
#include <KToolInvocation>

#include <QAction>
#include <QSizeF>
#include <QtDBus/QtDBus>
#include <QGraphicsLinearLayout>


ToggleCompositing::ToggleCompositing(QObject *parent, const QVariantList &args)
    :Plasma::Applet(parent, args)
    , m_state( true )
    , m_disabled( false )
    , m_kwinInterface( NULL )
    , m_icon( new Plasma::IconWidget )
    {
    setHasConfigurationInterface(false);
    setBackgroundHints(NoBackground);

    resize(m_icon->sizeFromIconSize(IconSize(KIconLoader::Desktop)));

    m_kwinInterface = new OrgKdeKWinInterface( "org.kde.kwin", "/KWin", QDBusConnection::sessionBus() );

    connect(m_kwinInterface, SIGNAL(compositingToggled(bool)), this, SLOT(compositingToggled(bool)));
    connect(m_icon, SIGNAL(clicked()), this, SLOT(clicked()));
    }

ToggleCompositing::~ToggleCompositing()
    {
    delete m_kwinInterface;
    delete m_icon;
    }

void ToggleCompositing::init()
    {
    createMenu();
    QGraphicsLinearLayout *layout = new QGraphicsLinearLayout(this);
    layout->setContentsMargins(0, 0, 0, 0);
    layout->setSpacing(0);
    layout->addItem(m_icon);
    m_disabled = !readEnabledState();

    if( !m_disabled && m_kwinInterface->compositingActive() )
        m_state = true;
    else
        m_state = false;
    updateIcon();
    }

void ToggleCompositing::createMenu()
    {
    QAction* configureEffects = new QAction( SmallIcon("configure"), i18n("&Configure Desktop Effects..."), this );
    m_actions.append( configureEffects );
    connect( configureEffects, SIGNAL(triggered(bool)), this, SLOT(configureCompositing()));
    QAction* updateState = new QAction( SmallIcon("view-refresh"), i18n("&Update Compositing State"), this );
    m_actions.append( updateState );
    connect( updateState, SIGNAL(triggered(bool)), this, SLOT(updateState()));
    }

QList<QAction*> ToggleCompositing::contextualActions()
    {
    return m_actions;
    }

void ToggleCompositing::configureCompositing()
    {
    QString error;
    KToolInvocation::startServiceByDesktopName( "kwincompositing", QStringList(), &error );
    }

bool ToggleCompositing::readEnabledState()
    {
    // TODO: we don't get the signal when config is changed, so we cannot react
    KSharedConfigPtr mKWinConfig=KSharedConfig::openConfig("kwinrc");
    KConfigGroup config(mKWinConfig, "Compositing");
    // TODO: compositing prefs
    return config.readEntry("Enabled", true);
    }

void ToggleCompositing::updateState()
    {
    bool state = !readEnabledState();
    if( state != m_disabled )
        {
        m_disabled = state;
        if( m_disabled)
            {
            // was active - now disabled
            m_state = false;
            }
        else
            {
            // it was disabled before, so read state from dbus
            m_state = m_kwinInterface->compositingActive();
            }
        updateIcon();
        }
    }

void ToggleCompositing::compositingToggled( bool active )
    {
    m_state = active;
    updateIcon();
    }

void ToggleCompositing::clicked()
    {
    if( !m_disabled )
        m_kwinInterface->toggleCompositing();
    }

void ToggleCompositing::updateIcon()
    {
    Plasma::ToolTipContent data;
    data.setMainText( i18n("Compositing state") );
    if( m_disabled )
        {
        // Compositing is disabled in settings
        data.setSubText( i18n("Compositing is disabled in settings") );
        m_icon->setIcon( KIcon( "dialog-cancel" ) );
        }
    else
        {
        if( m_state )
            {
            data.setSubText( i18n("Compositing is active") );
            m_icon->setSvg( "widgets/onoff_switch", "on" );
            }
        else
            {
            data.setSubText( i18n("Compositing is temporarily suspended") );
            m_icon->setSvg( "widgets/onoff_switch", "off" );
            }
        }
    data.setImage( KIcon("kwin") );
    m_icon->update();
    Plasma::ToolTipManager::self()->setContent( this, data );
    }


void ToggleCompositing::constraintsEvent(Plasma::Constraints constraints)
{
    setBackgroundHints(Plasma::Applet::NoBackground);
    updateIcon();
}

#include "toggle_compositing.moc"
