/*	Conductor

PIRL CVS ID: Conductor.java,v 2.47 2012/04/16 06:04:09 castalia Exp

Copyright (C) 2003-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package	PIRL.Conductor;

//	PIRL packages

import	PIRL.Configuration.Configuration;
import	PIRL.Configuration.Configuration_Exception;
import	PIRL.Database.Database;
import	PIRL.Database.Fields_Map;
import	PIRL.Database.Database_Exception;
import	PIRL.PVL.Parameter;
import	PIRL.PVL.PVL_Exception;
import	PIRL.Utilities.Host;
import	PIRL.Utilities.UNIX_Process;
import	PIRL.Utilities.Styled_Multiwriter;
import	PIRL.Utilities.Styled_Writer;
import	PIRL.Utilities.Multiwriter_IOException;
import	PIRL.Strings.String_Buffer;
import	PIRL.Conductor.Maestro.Local_Theater;
import	PIRL.Conductor.Maestro.Theater_Protocol_Exception;
import	PIRL.Messenger.Message;

//	JCM package for expression parser/evaluator.
import	edu.hws.jcm.data.Parser;
import	edu.hws.jcm.data.ParseError;

import	java.util.Collections;
import	java.util.Vector;
import	java.util.Date;
import	java.util.StringTokenizer;
import	java.io.File;
import	java.io.Writer;
import	java.io.FileWriter;
import	java.io.PrintWriter;
import	java.io.StringWriter;
import	java.io.PrintStream;
import	java.io.FileOutputStream;
import	java.io.IOException;
import	java.net.ConnectException;
import	java.text.ParseException;
import	javax.swing.text.AttributeSet;
import	javax.swing.text.SimpleAttributeSet;
import	javax.swing.text.StyleConstants;
import	java.lang.reflect.Method;


/**	<i>Conductor</i> is a queue management mechanism for the sequential
	processing of data files.
<p>
	A Conductor processes a list of data source files through a list of
	procedures to be invoked on each file. These lists are obtained from
	a <code>Database</code> as a pair of tables. The list of files is
	contained in a <i>Sources</i> table and the procedures are defined in
	a <i>Procedures</i> table. A pair of Sources and Procedures tables
	constitutes a <i>Pipeline</i>: each Sources record is processed in
	the order it occurs in the table (FIFO); each procedure specified by
	a Procedures record is executed in sequence number order. Each
	pipeline has a name that is used to find its database tables, where
	the pipeline tables are named:
<p>
	<blockquote>
	&lt;<i>pipeline</i>&gt;<b>_Sources</b><br>
	&lt;<i>pipeline</i>&gt;<b>_Procedures</b>
	</blockquote>
<p>
	A Conductor must be run with a {@link #Usage command line} argument
	that specifies the pipeline it is to process. Multiple Conductors
	may safely process the same pipeline from the same or separate host
	systems.
<h3>
	Database
</h3>
	Conductor requires a database containing the pipeline tables to be
	accessible. It is accessed using the PIRL {@link Database} package.
	This package abstracts the particulars of database access. The
	Conductor configuration file specifies the database access
	information.
<h4>
	Sources Table
</h4><p>
	A Sources table must contain at least these fields:
<dl>
<dt>Source_Number
<dd>Must be a non-NULL integer value unique for each record. While the
	order does not matter, it is easiest to make this a field with a
	value that is automatically assigned and incremented by the database
	server.
	
<dt>Source_ID
<dd>A string that identifies the file in some user specific manner. It is
	used, along with the Source_Number, to produce what is expected to be
	a unique log filename. If the field value is NULL or empty then the
	filename portion of the Source_Pathname, with any extension removed,
	will be used and this field will be updated.
	
<dt>Source_Pathname
<dd>This is the pathname of the file that is to be processed. The
	pathname is in the syntax of the host system. It is not required that
	the pathname be fully qualified, only that the file can be found
	using the pathname. This field value must not be NULL or empty.
	
<dt>Conductor_ID
<dd>This is a text field that must be NULL (not just empty) for each
	record to be processed. It is filled in with the name of the
	Conductor host system, possibly supplemented by the process ID of the
	Conductor (see the System Dependencies, Conductor_ID section, below),
	when the record is acquired for processing. This field provides an
	exclusive lock against other Conductor processes acquiring the
	record: Once a Conductor sets this field the record is no longer
	available for processing.

<dt>Status
<dd>An indicator of the status of each procedure invoked on the source
	file is recorded in this field. As each procedure is invoked on the
	source file this field is kept current with the status of the
	procedure. The format of the field value is described for the {@link
	#Status_Indicators(String) Status_Indicators} method which, along
	with other <code>Status_xxx</code> methods, provides a convenient
	means for other Java classes to interpret and manipulate these field
	values. <b>Note</b>: This field should initially be NULL or empty. It
	is important that this field only be modified consistent with the
	operation of Conductor (i.e. change at your own risk!).

<dt>Log_Pathname
<dd>The pathname for the log file where the processing of the source file
	is recorded. Normally this field is left empty (or NULL) and the
	Log_Pathname (or Log_Directory) parameter of the Configuration file
	will be used. If either of these is a pathname that refers to an
	existing directory, then Conductor will generate an appropriate log
	filename (see the description of the Log_Directory and Log_Filename
	parameters, below). If this field is empty and both parameters are
	empty or not present, the log file will be written to the current
	working directory (the Log_Directory parameter is empty) However, if
	either this field or the Log_Pathname (or Log_Directory) parameter is
	not empty and does not refer to an existing directory then the 
	pathname is to a regular file then that file will be appended with
	the source file log (a new file will be created if it does not yet
	exist). Both this field value and any parameter used will be
	reference resolved if not empty. <b>N.B.</b>: If a pathname to an
	existing log file pathname is not specified by this field or a
	configuration parameter then any existing file at the generated
	pathname is overwritten. This field is always updated by whatever
	actual log file pathname is used.
</dl><p>
	A Sources table may contain the required fields in any order, and it
	may contain additional fields as desired. For example, it is
	recommended that a timestamp field be provided that will
	automatically be updated with the last update time of each record.
<h4>
	Procedures Table
</h4><p>
	A Procedures table must contain at least these fields:
<dl>
<dt>Sequence
<dd>A real number value that orders the procedure in the sequence of
	processing the source file. The values need not be sequential nor
	must they be in any particular order in the table. All of the
	procedure records will be sorted numerically on this field value so
	that processing of the source file will occur in sequence order. It
	is strongly recommended that the values be unique in the table, but
	this is not required; however there is no certainty of the order of
	processing for procedures with the same sequence number.

<dt>Command_Line
<dd>The command line to be submitted to the system for executing the
	procedure. The command line may contain embedded field and/or
	parameter {@link Reference_Resolver references} to be substituted
	with database field values and/or configuration parameter values.
	Obviously this field must be neither empty nor NULL.

<dt>Success_Status
<dd>An integer value that matches the exits status of the procedure when
	the procedure has completed successfully. The value can be text that
	is subject to embedded reference resolving, but must ultimately be
	convertible to an integer value. If this field is NULL or empty and
	the Success_Message field is not, then the latter field is used
	instead. If both fields are empty then the Empty_Success_Any
	parameter control how this will be interpreted. This field may
	contain text that is reference resolved but must produce an integer
	value.

<dt>Success_Message
<dd>Text to match on the procedure output lines - either stdout or stderr
	- to determine if the procedure completed successfully. This field is
	only used if the Success_Status field is empty or NULL. The text in
	this field is reference resolved. The {@linkplain
	java.lang.String#matches(String) match} against the lines of
	procedure output treats the resolved text as a regular expression
	{@link java.util.regex.Pattern pattern}.

<dt>Time_Limit
<dd>The maximum amount of time, in seconds, to wait for the procedure to
	complete. The value can be text that is subject to embedded reference
	resolving. After resolving any references the result is treated as a
	mathematical expression that must produce a single integer value. A 0
	(zero) or negative value indicates an unlimited wait time. An empty
	or NULL value is equivalent to 0. If the procedure does not complete
	within the specified amount of time it is killed.
	
<dt>On_Failure
<dd>A command line just like the Command_Line field. If the procedure
	defined by the Command_Line field fails to complete successfully,
	then the procedure defined by the On_Failure field is run. No time
	limit is applied to this procedure.
</dl><p>
	A Procedures table may contain the required fields in any order, and
	it may contain additional fields as desired. If a "Description" field
	is present, which is higly recommended, it is used as a text
	description of each procedure included in the processing log. It is
	also recommended that a timestamp field be provided that will
	automatically be updated with the last update time of each record.
<h5>
	Database Connection Resilience
</h5><p>
	A Conductor maintains a connection with the {@link Database} server
	while it is operating. If any access to the Database by Conductor
	fails due to a loss of the connection, Conductor will attempt to
	reconnect and, if successful, repeat the access operation again (a
	connection failure of the repeated access operation does not result
	in a reconnection attempt). If the reconnection fails because the
	connection can not be established with the server, the attempt will
	be retried after a delay period of 5 minutes (this can be overridden
	with the {@link #RECONNECT_DELAY_PARAMETER}). Up to 16 retries (this
	can be overridden with the {@link #RECONNECT_TRIES_PARAMETER}) will
	be attempted before a database access failure is deemed to have
	occurred. <b>N.B.</b>: Database connection resilience does not
	automatically apply to any database access operations by pipeline
	procedures.
<h3>
	Configuration
</h3><p>
	When a Conductor is started it first reads its {@link Configuration
	Configuration} file. This is "Conductor.conf" by default, but another
	filename may be specified on the {@link #Usage command line}. The
	configuration file contains parameter definitions in Parameter Value
	Language ({@linkplain PIRL.PVL.Parser PVL}) format. This file
	contains the information needed to access the database used by
	Conductor as well as any other parameters that may be useful to
	{@linkplain Reference_Resolver resolve} parameter references embedded
	in procedure definition record field values. Since references may
	contain nested references it is quite appropriate for users to
	provide configuration parameters with values that are database field
	references (perhaps with complex conditionals and multiple field
	combinations) so that Command_Line (for example) definitions use the
	user specified parameter references rather than the more complicated
	definitions. This also makes it easy to modify the field reference
	definitions, just by editing the configuration file, without
	necessarily needing to change the contents of a Procedures pipeline
	table.
<p>
	<b>N.B.</b>: By default, when the configuration file is read during
	startup by the application's main method should any parameters have
	the same pathname the last duplicate encountered is given preference.
	This is especially important to keep in mind when the configuration
	file {@link Configuration#Include() includes} another configuration
	file, such as a site-wide configuration. For example, a site-wide
	configuration file included in all pipeline-specific configuration
	files (a typical scenario) might have a Conductor group that includes
	default parameters such as Stop_on_Failure with a small value (e.g. 1
	or 2) to prevent a bug in a pipeline procedure from generating a
	large number of source processing failures, while a configuration
	file for a specific pipeline that is expected to have failures (which
	may actually be branches off to some other pipeline depending on the
	outcome of some condition testing procedure) might have a Conductor
	group that includes a Stop_on_Failure with a large (or possibly zero)
	value. As long as the site-wide configuration file is included before
	the pipeline-specific Conductor/Stop_on_Failure parameter is
	specified the latter will take precedence over the former.
<h4>
	Parameters
</h4><p>
	The Conductor automatically provides a set of parameters in the
	configuration Conductor group:
<dl>
<dt>Class_ID
<dd>The fully qualified Conductor class name with its revision number
	and date.

<dt>Configuration_Source
<dd>The name of the configuration file source. This may be the name of a
	file on the Conductor host system or a URL for a file obtained
	remotely.

<dt>Conductor_ID
<dd>The Conductor identification (see the System Dependencies,
	Conductor_ID section, below).

<dt>Database_Server_Name
<dd>The name of the database Server configuration parameters group. If no
	Server name could be determined, this parameter will not be included.

<dt>Hostname
<dd>The fully qualified hostname of the system where Conductor is running.
	If the hostname can not be obtained the IP (internet protocol) address
	will be used.

<dt>Database_Hostname
<dd>The hostname of the database server system.

<dt>Database_Type
<dd>The type of database server, as known to the {@link Database} access
	package.

<dt>Pipeline
<dd>The simple pipeline name (without any catalog prefix).

<dt>Catalog
<dd>The name of the database catalog where the pipeline tables are
	located.

<dt>Sources_Table
<dd>The full name of the table, including the Catalog prefix, containing
	the Source file records.

<dt>Procedures_Table
<dd>The full name of the table containing the procedure definitions.
</dl><p>
<h5>
	Dynamic source parameters
</h5><p>
	The following parameters are reset for each source record being
	processed:
<dl>
<dt>Log_Directory
<dd>The pathname to the directory where the log file is written. If a
	Log_Pathname field value is present it is used to determine the
	Log_Directory, but only for the current source record. Otherwise the
	Log_Pathname configuration parameter is used. If it is not present or
	empty the Log_Directory parameter is used instead. The default
	Log_Directory is Conductor's current working directory.

<dt>Log_Filename
<dd>The filename (without the directory path) of the source log file. If
	the neither the source record Log_Pathname field nor the Log_Pathname
	or Log_Directory parameters has a non-empty value that does not refer
	to an existing directory - a value that does not refer to an existing
	directory is taken to be the pathname to the log file - then a
	default filename will be generated that has the form:
<blockquote>
	&lt;<i>Pipeline</i>&gt;<b>-</b>&lt;<i>Source_ID</i>&gt;<b>_</b>&lt;<i>Source_Number</i>&gt;<b>.log</b>
</blockquote><p>
	The Pipeline name includes the leading database catalog name
	separated by a period ('.') character.
<p>
	The Source_ID and Source_Number are obtained from the current source
	record. Note, however, that there is a chance that the Source_ID will
	include characters that are unsafe for use as part of a filename.
	Assuming that the only unsafe character is the system property
	"file.separator" character ('/' for Unix), it will be replaced with a
	percent ('%') character.
<p>
	<b>Note</b>: If either the source record field or configuration
	parameter value is not empty and does not refer to a an existing
	directory, then that value will be used unconditionally to determine
	the log filename.

<dt>Source_Number
<dd>The Source_Number field value of the current source record.

<dt>Source_ID
<dd>The Source_ID field value of the current source record.

<dt>Source_Pathname
<dd>The Source_Pathname field value of the current source record in the
	filesystem's fully qualified (absolute) form.

<dt>Source_Directory
<dd>The directory path portion of the Source_Pathname value.

<dt>Source_Filename
<dd>The filename portion (without the directory pathname) of the
	Source_Pathname value.

<dt>Source_Filename_Extension
<dd>The portion of the Source_Filename value following the last period
	('.') character in the name. This will be the empty string if there
	is no extension.

<dt>Source_Filename_Root
<dd>The portion of the Source_Filename value without the extension (the
	portion preceding the last period character). This may be the empty
	string.
</dl><p>
<h5>
	Dynamic procedure parameters
</h5><p>
	The following parameters are reset for each procedure record being
	processed:
<dl>
<dt>Total_Procedure_Records
<dd>The total number of procedure definition records in the procedures
	table. <b>Note</b>: This a a dynamic parameter because the procedures
	table is refreshed each time pipeline processing is started and the
	table may have been changed while the Conductor was waiting.

<dt>Procedure_Count
<dd>The procedure definition record count for the current, or last,
	procedure sequence. The first procedure definition record has a count
	of one. A Procedure_Count of zero means that processing for the
	current source record has not yet commenced.

<dt>Sequence
<dd>The Sequence field value of the current, or last. procedure record.

<dt>Completion_Number
<dd>The completion number for the last procedure that was executed. If
	the procedure ran to completion, whether it was successful or not, it
	will be the exit status value (a non-negative integer value) of the
	procedure. If the procedure did not complete for any reason it will
	be a negative Conductor completion code; the {@link
	#Status_Conductor_Code_Description(int)
	Status_Conductor_Code_Description} static method may be used to
	obtain a brief, one line description of this code.
</dl><p>
<h5>
	Conductor control parameters
</h5><p>
	The following configuration parameters will be used if they are
	present in the Conductor group or parent of this group:
<dl>
<dt>Unresolved_Reference
<dd>The value to use for an unresolved reference. By default an
	unresolved referenced throws a Database_Exception. This can be
	specified with a value beginning with the word "throw" (case
	insensitive). <b>Note</b>: All parameters used by Conductor are
	reference resolved. Those with unresolved references that would throw
	an exception are deemed to be missing parameters. Those values that
	have incorrect reference syntax are left unresolved.

<dt>Empty_Success_Any
<dd>If "true", when Success_Status and Success_Message are both empty or
	NULL the corresponding procedure is always deemed successful when it
	completes. Otherwise this condition implies a zero (0)
	Success_Status. Default: false.

<dt>Min_Source_Records
<dd>The minimum number of source records to be processed in batch
	mode before Conductor stops. Default: 1.

<dt>Max_Source_Records
<dd>The maximum number of source records to obtain at any one time from
	the database. This prevents memory exhaustion if the number of
	unprocessed source records is very large. Must not be less than
	Min_Source_Records. Default: 1000.

<dt>Poll_Interval
<dd>The amount of time (in seconds) to wait before trying to obtain more
	unprocessed source records when querying the source table found no
	unprocessed records. If this value is zero or negative Conductor
	processing will stop instead of waiting to try again. Default: 30.

<dt>Source_Available_Tries
<dd>The number of tries that will be made to confirm that the
	Source_Pathname is accessible - i.e. exists as a regular file that
	can be read - before giving up and declaring the file to be
	inaccessible. After each accessibility check failure, and before the
	next try, a ten second pause will be provided. The intention is to
	give filesystem directory caches time to be synchronized with newly
	created files on remote filesystems. A maximum of 180 retries (30
	minutes wait time) is allowed to prevent Conductor from waiting
	indefinitely. If the value is negative no Source_Pathname confirmation
	will be done. This can be useful if the pipeline does not process a
	source file. Default: 12.

<dt>Reconnect_Tries
<dd>The maximum number of reconnection tries if the database connection
	is lost. Default: 16.

<dt>Reconnect_Delay
<dd>The delay, in seconds, between database reconnection retry attempts.
	Default: 300.

<dt>Stop_on_Failure
<dd>The number of sequential source processing failures that will cause
	Conductor to stop further processing. Zero means source failures will
	never cause processing to stop. "true" or "yes" is equivalent to 1;
	"false" or "no" is equivalent to 0. Default: 0;

<dt>Notify
<dd>A list of zero or more email address that will be sent a notification
	if Conductor processing halts. The reason processing halted will be
	in the email message.
</dl><p>
<h5>
	Stage_Manager parameters
</h5><p>
	The Conductor will try to connect to a Stage_Manager process on the
	local host system. The Stage_Manager provides remote Management
	capabilities. A host system may be running multiple Stage_Maangers,
	each using its own communications port, to provide mutiple Theater
	management contexts for different sets of Conductors. A Conductor can
	operate in only one Theater as determined by its Stage_Manager
	parameters. The following configuration parameters, which must be in
	a Stage_Manager sub-group of the Conductor group, control the
	Conductor connection to the Theater's Stage_Manager:
<dl>
<dt>Require_Stage_Manager
<dd>If "enabled", "true", "yes" or 1 a connection to a Stage_Manager is
	required for this Conductor or it will not run. <b>N.B.</b>: This
	parameter is only read once when the Conductor is first configured;
	when the Conductor is reconfigured on being restarted after a Stop or
	Halt state the initial value is reset in the internal Configuration
	regardless of any change to the external configuration source.

<dt>Timeout
<dd>The amount of time (seconds) to wait for a Stage_Manager connection
	to complete before the connection attempt fails.

<dt>Password
<dd>The password required to authenticate a Stage_Manager connection. The
	value is a text string of any length. If this parameter is present
	its value is masked out in the internal Configuration. If this
	parameter is not present, or has an empty value, and the
	Stage_Manager requires an authenticated connection the connection
	attempt will fail.

<dt>{@link #HELLO_PORT_PARAMETER_NAME} - Get and Set
<dd>The port number to use when listening for a Stage_Manager "Hello"
	broadcast that it is ready for connections.

<dt>{@link #HELLO_ADDRESS_PARAMETER_NAME} - Get and Set
<dd>The multicast address to use when listening for a Stage_Manager "Hello"
	broadcast that it is ready for connections.
</dl><p>
<h3>
	Processing
</h3>
<h4>
	Procedure Records
</h4><p>
	After the Conductor has been initialized using its configuration file
	and connected to the database it begins to process the pipeline.
	<b>Note</b>: If the Conductor is run with a Manager (using the
	-Manager command line option), processing does not begin immediately;
	the Conductor will wait until it is told to start by the Manager. The
	records from the Procedures table are read, confirmed that they
	contain all necessary fields, and sorted into Sequence number order.
	A Conductor may be told to stop processing by a Manager (including a
	remote Manager), which will cause the Conductor to stop further
	processing when  the current source record processing is complete.
	Processing is resumed with the next source record when a Manager
	sends the start signal. Each time pipeline processing is started the
	Procedures is loaded again and the configuration file is read again
	and used to reconfigure the Conductor. Changes to the Procedures
	records and configuration file can safely be made while Conductor is
	running.
</h3>
<h4>
	Source Records
</h4><p>
	When pipeline processing has been started Conductor begins processing
	records from its Sources table. All unprocessed records, up to a
	maximum (set by default to 1000 to prevent memory exhaustion)
	configurable with the Max_Source_Records parameter, are read into an
	internal cache. <b>An unprocessed record has a NULL in its
	Conductor_ID field.</b> These records are processed in first-in
	first-out (FIFO) order.
<p>
	An exclusive lock must be acquired on a record before it can be
	processed. To acquire a lock on a source record an attempt is made
	to update the record's Conductor_ID field to the Conductor's
	identification value (hostname and possibly process ID) with the
	condition that the field value is currently NULL. The update
	operation by the database server is atomic; once the operation is
	started by the database server it will go to completion without the
	possibility of interruption by any other database operation. This
	guarantees that only one process will be able to gain access to any
	source record even in the context of multiple processes contending
	for the same record at the same time. If some other Conductor has
	already acquired the record, as indicated by the failure of the
	update operation (because the Conductor_ID field is no longer NULL
	as required), the record will be removed from the cache and the
	Conductor will try to acquire a lock on the next record in the
	cache. If the update succeeds then the Conductor has acquired
	exclusive control of the record. It will be safe to process the
	source without concern that some other process may interfere.
<p>
	The first step in processing a source record is to open a log file to
	record the processing. If the source record Log_Pathname field is
	empty the user's Log_Pathname configuration parameter will be used.
	If that is absent or empty the Log_Directory parameter will be used.
	If that, too, is absent or empty a default filename will be generated
	- as described in the Log_Filename parameter description, above - and
	the log file will be written the current working directory. If either
	the source record field or configuration parameter is not empty the
	value is resolved for any embedded {@linkplain Reference_Resolver
	references}. If the resulting pathname refers to an existing
	directory the default filename is added. Otherwise the pathname is
	taken to refer to a file to which processing log output is to be
	appended (the file will be created if it does not exist).
	<b.N.B.</b>: In all other cases - whenever a default filename is
	generated - an existing file will be overwritten. The log file
	pathname that is used is always updated to the source record
	Log_Pathname field. <b>Note</b>: A source record will not be
	processed without a log file; it is fatal to Conductor not to have a
	writable log file available for each source record.
<p>
	The log file always begins with the Conductor class identification:
<p>
	<b>PIRL.Conductor.Conductor (2.47 2012/04/16 06:04:09)</b>
<p>
	This line is immediately followed by a {@link
	#SOURCE_FILE_LOG_DELIMITER} line which is expected to be 70 equals
	('=') characters. This is followed by a date and time stamp and the
	source record description including the database server type and
	hostname, the fully qualified name of the Sources Table
	(Sources_Table) and the Source_Number, Source_ID, and
	Source_Pathname values.
<p>
	The Status field of the source record is checked for any status
	indicators from possible previous processing. If present they are
	logged. If the last status indicator is for a failure condition
	this is logged and any further processing of the source is skipped.
	<b>Note</b>: It is possible to (re)start processing of a source
	midway in its procedures pipeline. This is done by first setting
	its Status field to include status indicators for procedures to be
	skipped; e.g. by removing a failure indicator at the end of the
	list after correcting the cause of the failure. Then the
	Conductor_ID field is set to NULL (as long as this is done last it
	will be safe even if the actively being processed by a Conductor).
	When the source record is acquired by a Conductor its processing
	will begin with the next procedure without a status indicator, and
	log output will be appended to its previous log file if present or
	a new file if needed. <b>Caution</b>: Procedure pipelines to be
	used in this way must, of course, ensure that any dependencies on
	previous procedures are taken into account.
<p>
	At this point configuration parameters dependent on the current
	source record are updated.
<p>
	The Source_Pathname file is confirmed to be a normal file (not a
	directory) that is readable. If the file is not accessible the
	Status field of the source record is updated in the database with
	the {@link #INACCESSIBLE_FILE} Conductor completion code, the
	Completion_Number parameter is set to the same value, the condition
	is logged and further processing of the source record is canceled.
<h4>
	Procedures Pipeline
</h4><p>
	The source now enters the procedures pipeline. The procedure
	definition records are all cached and sorted by their Sequence
	number when Conductor first starts. Thus changes to a Procedures
	table will not take effect until after a Conductor (re)starts, and
	it is safe to change a Procedures table while a Conductor is
	running.
<p>
	Each procedure record is applied to the current source record in
	Sequence number order; the Sequence parameter is updated before
	each procedure is processed.
<h5>
	Embedded References
</h5><p>
	All of the required Procedure fields, except the Sequence number,
	may contain embedded {@linkplain Reference_Resolver references}.
	Each embedded reference is effectively a variable that is
	substituted with the value from a database field or configuration
	parameter specified by the reference. References may be arbitrarily
	nested; for example, the condition for selecting a record in a
	database field reference may be a parameter reference supplied in a
	configuration parameter. Reference resolution is also recursive;
	the value obtained from resolving a reference may itself contain
	embedded references. Thus a parameter reference may resolve to a
	parameter value that contains references. This allows the values of
	database fields to contain references to user defined parameters
	that are set as desired in the configuration file without needing
	to change the contents of database tables to effect the change.
<p>
	Reference resolved values in Procedure fields allow dynamic
	definition of procedure attributes. References that are unresolved
	are fatal to Conductor unless the Unresolved_Reference
	configuration parameter has been set to a substitute string (e.g.
	""). References that have incorrect syntax (e.g. unbalanced curly
	brace enclosures) are always fatal to Conductor.
<h5>
	Procedure Execution
</h5><p>
	The Command_Line value is reference resolved and {@linkplain
	#Parse_Command_Line parsed} into an initial command name and
	command arguments. An empty or NULL Command_Line is fatal. Before
	each procedure is run the log file is written with the {@link
	#PROCEDURE_LOG_DELIMITER} line. This is followed by a date and time
	stamp, the Sequence number, the Description field value (if it is
	not empty), and then the command line to be executed.
<p>
	The command name and arguments are passed to the Java Runtime for
	execution as a Process by the host operating system. <b>Note</b>:
	the command is not run in a shell. It is, however, quite
	appropriate to run shell, or any other interpreted language,
	scripts (e.g. PERL). The only restriction on the procedure to be
	run is that it is accessible and executable. If the procedure can
	not be executed for any reason the source record's Status field
	value is appended with Conductor's {@link #NO_PROCEDURE} error
	status. Otherwise the Status field is updated with the host system
	Process ID for the executed procedure; this is always an integer
	value greater than 1 that uniquely identifies the executing
	procedure in the host operating system.
<p>
	All standard output from the procedure is copied into the log file
	with an annotation before each line that indicates whether the
	source is the procedure's stdout or stderr streams. Because these
	are separate streams read by asynchronous threads attached to each
	process stream there can be no guarantee of the relative logging
	order of lines from the two sources; while each stream is always
	logged in the order in which the procedure output to it, the
	uncertainties of system stream buffering and thread scheduling are
	likely to result in lines from stdout appearing in the log before
	or after where they might occur relative to stderr lines appearing
	in a shell terminal listing.
<p>
	Conductor waits for the procedure to complete before proceeding.
	However, it will not wait longer than the number of seconds from
	the Time_Limit field (which may have embedded references and may be
	a mathematical expression). If the value is NULL, empty or zero
	then there is no limit to the amount of time Conductor will wait
	for the procedure to complete. It is generally a good idea to place
	a maximum running time limit on any procedure that could become
	"hung" (for example in a loop or on an inaccessible). If the time
	limit is reached Conductor will destroy the procedure. This is done
	by sending the procedure a terminate signal (SIGTERM). This signal
	can be caught by the procedure so it has an opportunity to clean up
	open files or child processes of its own. For scripts that have
	launched long running computational programs it is correct practice
	to catch the terminate signal and halt these programs; failure to
	do so is likely to leave these child programs running as orphans.
	If a procedure does not catch the terminate signal it will be
	automatically terminated by the operating system. If Conductor must
	terminate a procedure due to a timeout the source record's Status
	field will be updated with Conductor's {@link #PROCEDURE_TIMEOUT}
	error status and the log will be written with notice of the
	timeout. If the procedure completes normally, then the standard
	output streams are drained and copied to the log file and the exit
	status from the procedure is also noted in the log file.
<h5>
	Procedure Status
</h5><p>
	When the procedure execution is done the Completion_Number
	parameter is updated. This will be a negative value if the
	procedure did not run to completion (could not be executed or
	exceed the Time_Limit), otherwise it will be the procedure's exit
	status value.
<p>
	When a procedure completes normally Conductor uses either the
	Success_Status or Success_Message field values to determine if the
	procedure completed successfully. Usually the exit status is set by
	the procedure to a value that indicates if it succeeded. However, it
	may be necessary to examine the output of the procedure if the exit
	status is not reliable. There may also be unfortunate cases where
	there is no reliable indicator and all that can be done is assume
	that because the procedure completed it was successful.
<p>
	If neither the Success_Status or Success_Message field values 
	has been set to a non-empty value and an Empty_Success_Any
	configuration parameter was found with a "true" value then the
	procedure success of the proedure is implied (i.e. in this case
	the procedure is always successful if it completes normally);
	otherwise the Success_Status value is asserted to be "0".
<p>
	If the Success_Status field is not empty it is reference resolved.
	The result is evaluated as a logical expression and if a result is
	obtained it determines if the procedure.was successful. Typically,
	the expression uses a reference to the Completion_Number parameter.
	The logical operators &, |, ~, =, <, >, <>, <=, >= may be used. The
	words "and", "or", and "not" can be used in place of &, |, and ~.
	<b>Caution</b>: Use &, not &&; |, not ||; ~, not !; =, not ==; and
	<>, not !=. A logical expression may contain embedded numeric
	expressions as well.
<p>
	If the Success_Status does not contain a valid logical expression it
	is evaluated as a numeric expression. If the result, cast as an
	integer value, is equal to the procedure's exit status, then the
	procedure succeeded; otherwise it failed. A numeric expression may
	simply be a constant value (the symbols "pi" and "e" are recognized
	as constants) or may use the +, -, *, /, ^ operators; ** may be used
	instead of the ^ exponentiation operator. The tertiary operator ?
	with : may be used following an embedded logical expression such that
	if the logical expression is true then the following value before the
	: is used, else the value after the : is used (e.g. (4<5)?1:2
	evaluates to 1). The functions sin, cos, tan, cot, sec, csc, arcsin,
	arccos, arctan, exp, ln, log2, log10, sqrt, cubert, abs, round,
	floor, ceiling, trunc may also be used with their argument following
	inside parentheses.
<p>
	The Success_Message, if not empty, is used if the Success_Status is
	empty. It is referenced resolved and then matched, as a {@linkplain
	java.util.regex.Pattern regular expression}, against what was
	obtained from the procedure's stdout and stderr. If there is a match
	with either output, then the procedure succeeded; otherwise it
	failed. A resolved Success_Message value that does not produce a
	valid regular expression is fatal to Conductor. <b>Note</b>: Regular
	expressions are very powerful expression matching syntax similar to
	that used by PERL, but also can be daunting to the beginner.
<p>
	Regardless of the outcome of procedure execution the Status field of
	the source record is updated in the database with the procedure's
	{@linkplain #Status_Indicators status indicator}. This indicator
	always includes the Conductor completion code which can be translated
	into a descriptive line of text by the {@link
	#Status_Conductor_Code_Description(int)
	Status_Conductor_Code_Description} static method. If the procedure
	completed with an exit status that value is included in the status
	indicator. The meaning of this value is procedure dependent. Of
	course the log file is also annotated accordingly.
<h5>
	On Failure
</h5><p>
	When Conductor determines that the procedure completed successfully
	it repeats the procedure execution operation with the next procedure
	in the pipeline Sequence. If Conductor determines that the procedure
	did not complete successfully, then it resolves any embedded
	references in the On_Failure field value and uses that as a command
	line for a procedure to be executed. This procedure is executed
	without any time limit. In the log file, where a normal procedure
	would have a {@link #PROCEDURE_LOG_DELIMITER} the On_Failure
	procedure has an {@link #ON_FAILURE_PROCEDURE_LOG_DELIMITER} and no
	Description. Although the completion status of this procedure is not
	included in the final status indicator of the source record's Status
	field it is noted in the log file.
<p>
	When the number of sequential source processing failures reaches
	the Stop_on_Failure amount further processing is halted after
	the On_Failure procedure has been run.
<p>
	When operating under the direction of a Manager (local and/or remote)
	if the Conductor halts for any reason it will send an email message
	to its Notify list and then wait to be told to start processing again.
	Without the possibility of a Manager to take charge the Conductor
	will exit after halting.
<h4>
	Sources Completion
</h4><p>
	The completion of the last of the Procedures in pipeline sequence,
	or the first On_Failure procedure, completes the processing of a
	source record. The log file is now closed. While there is another
	source record in the cache Conductor will continue trying to
	acquire an exclusive database lock. Once the cache is exhausted
	Conductor refreshes it from the Sources table with any new
	unprocessed records. If no unprocessed records are available then
	Conductor will sleep for the number of seconds indicated by the
	Poll_Interval configuration parameter. If no Poll_Interval
	parameter is present the default interval of 30 seconds is used. If
	the interval is less than or equal to 0, then Conductor processing
	will stop when it can find no source records to process.
<h3>
	System Dependencies
</h3><p>
<h4>
	Java
</h4><p>
	Conductor is known to compile and run correctly with Java 1.4 and
	1.5. Java was chosen for the implementation to maximize portability:
	as long as the host system provides a standard Java environment it
	should be able to run Conductor.
<h4>
	Conductor_ID
</h4><p>
	The Conductor_ID value will include the process ID of Conductor if
	it is available. Obtaining the process ID (PID) of Conductor - i.e.
	the Java Virtual Machine (JVM) that runs the Java classes - requires
	using a Java Native Interface (JNI) to the host system function that
	provides this information. Though this is trivial to implement it is
	outside the pure Java implementation of Conductor. Without the JNI
	code the Conductor_ID will only be the hostname of the system on
	which Conductor is running. With the JNI code the Conductor_ID will
	include the JVM PID after the hostname separated by a colon (':')
	character. The availability of the JVM PID will not have any effect
	on Conductor's operation. However, it is quite useful to have the
	JVM PID for procedures to use in disambiguating filenames in a
	parallel processing shared storage environment, and it can assist in
	systems administration work. A Native_Methods.c file that provides
	JNI access to the required system function is included in the source
	code distribution of Conductor. When the Conductor source code is
	compiled Native_Methods.c is also compiled to produce a dynamically
	loadable Native_Methods.so (or .jnilib on Apple OS X/Darwin systems)
	shared object library file in the Conductor/<OS>.<arch>
	subdirectory, where <OS> is the name of the host operating system
	(e.g. Darwin, FreeBSD, Linux or SunOS) and <arch> is the host system
	hardware architecture (e.g. i386, powerpc, x86_64 or sparc).
	<b>N.B.</b>: The library file must be copied to a location on the
	host system where it can be found by the dynamic linker (e.g.
	/usr/local/lib) when Conductor runs. The JNI library file can be
	built separately from the Java class files - if, for example,
	multiple operating systems and/or architecures are being used - by
	running "make jni" (GNU make may be named gmake) in the Conductor
	source code directory. The Native_Methods.c file requires the
	$(JNI_ROOT)/include/jni.h file included with the Java Software
	Development Kit (SDK) distribution. $(JNI_ROOT) is /usr/java by
	default, but a JNI_ROOT environment variable can be set with an
	alternative location before the JNI library is compiled.
<p>
	@author		Bradford Castalia, Christian Schaller - UA/PIRL
	@version	2.47
	@see		PIRL.Database
	@see		PIRL.Conductor.Maestro
*/
public class Conductor
	implements
		Management
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Conductor.Conductor (2.47 2012/04/16 06:04:09)";

//	Conductor process identification.
private static String
	Conductor_ID;


// Configuration:

/**	The default configuration filename.
*/
public static final String
	DEFAULT_CONFIGURATION_FILENAME		= "Conductor.conf";
static
	{
	Configuration.Default_Source (DEFAULT_CONFIGURATION_FILENAME);
	}

/**	The Configuration object containing the configuration parameters.
*/
protected Configuration
	The_Configuration					= null;

/**	The default {@link Configuration#Duplicate_Parameter_Action(int) action}
	should a duplicate parameter pathname occur in the Conductor
	Configuration file.
*/
public static final int
	DEFAULT_DUPLICATE_PARAMETER_ACTION	= Configuration.PREFER_LAST_PARAMETER;

/**	Conductor Configuration parameters.
*/
public static final String
	CONDUCTOR_GROUP						= "Conductor",

	//	The name of parameters in the CONDUCTOR_GROUP -

	//	Conductor:
	CONFIGURATION_SOURCE_PARAMETER		= "Configuration_Source",
	DATABASE_SERVER_NAME_PARAMETER		= "Database_Server_Name",
		DATABASE_SERVER_PARAMETER			= Database.SERVER,
	HOSTNAME_PARAMETER					= "Hostname",
	DATABASE_HOSTNAME_PARAMETER			= "Database_Hostname",
	DATABASE_TYPE_PARAMETER				= "Database_Type",
	PIPELINE_PARAMETER					= "Pipeline",
	CATALOG_PARAMETER					= "Catalog",
	PROCEDURES_TABLE_PARAMETER			= "Procedures_Table",
	SOURCES_TABLE_PARAMETER				= "Sources_Table",
	UNRESOLVED_REFERENCE_PARAMETER		= "Unresolved_Reference",
		UNRESOLVED_REFERENCE_THROWS			= "THROW",
	EMPTY_SUCCESS_ANY_PARAMETER			= "Empty_Success_Any",
	MIN_SOURCE_RECORDS_PARAMETER		= "Min_Source_Records",
	MAX_SOURCE_RECORDS_PARAMETER		= "Max_Source_Records",
	POLL_INTERVAL_PARAMETER				= "Poll_Interval",
	SOURCE_AVAILABLE_TRIES_PARAMETER	= "Source_Available_Tries",
	STOP_ON_FAILURE_PARAMETER			= "Stop_on_Failure",
	RECONNECT_TRIES_PARAMETER			= "Reconnect_Tries",
	RECONNECT_DELAY_PARAMETER			= "Reconnect_Delay",
	NOTIFY_PARAMETER					= "Notify",

	//	Source:
	SOURCE_NUMBER_PARAMETER				= "Source_Number",
	SOURCE_PATHNAME_PARAMETER			= "Source_Pathname",
	SOURCE_ID_PARAMETER					= "Source_ID",
	CONDUCTOR_ID_PARAMETER				= "Conductor_ID",

	SOURCE_DIRECTORY_PARAMETER			= "Source_Directory",
	SOURCE_FILENAME_PARAMETER			= "Source_Filename",
	SOURCE_FILENAME_ROOT_PARAMETER		= "Source_Filename_Root",
	SOURCE_FILENAME_EXTENSION_PARAMETER	= "Source_Filename_Extension",
	LOG_PATHNAME_PARAMETER				= "Log_Pathname",
	LOG_DIRECTORY_PARAMETER				= "Log_Directory",
	LOG_FILENAME_PARAMETER				= "Log_Filename",

	SOURCE_SUCCESS_COUNT				= "Source_Success_Count",
	SOURCE_FAILURE_COUNT				= "Source_Failure_Count",
	TOTAL_FAILURE_COUNT					= "Total_Failure_Count",

	//	Procedure:

	TOTAL_PROCEDURE_RECORDS_PARAMETER	= "Total_Procedure_Records",
	PROCEDURE_COUNT_PARAMETER			= "Procedure_Count",
	PROCEDURE_SEQUENCE_PARAMETER		= "Sequence",
	PROCEDURE_COMPLETION_NUMBER_PARAMETER = "Completion_Number",

	//	Stage_Manager:

	REQUIRE_STAGE_MANAGER_PARAMETER
		= "Stage_Manager/Require_Stage_Manager",
	STAGE_MANAGER_PASSWORD_PARAMETER
		= "Stage_Manager/"
			+ PIRL.Conductor.Maestro.Stage_Manager.PASSWORD_PARAMETER_NAME,
	STAGE_MANAGER_PORT_PARAMETER
		= "Stage_Manager/"
			+ PIRL.Conductor.Maestro.Stage_Manager.PORT_PARAMETER_NAME,
	STAGE_MANAGER_TIMEOUT_PARAMETER
		= "Stage_Manager/Timeout",
	HELLO_PORT_PARAMETER_NAME
		= "Stage_Manager/"
			+ PIRL.Conductor.Maestro.Stage_Manager.HELLO_PORT_PARAMETER_NAME,
	HELLO_ADDRESS_PARAMETER_NAME
		= "Stage_Manager/"
			+ PIRL.Conductor.Maestro.Stage_Manager.HELLO_ADDRESS_PARAMETER_NAME;


//	Database:

private String
	Database_Server_Name		= null;

/**	The Database object used to access the database server.
*/
protected Database
	The_Database				= null;

//	Number of database connection retries before giving up.
private int
	Reconnect_Tries				= 16,
	Reconnect_Delay				= 300;	// seconds.

/**	The name of the database catalog containing the pipeline tables.
*/
protected String
	Catalog						= null;

//	This is static so it can be used by Exit if construction fails.
/**	The name of the pipeline (<catalog>.<pipeline name>) being managed.
*/
protected static String
	Pipeline					= null;

/**	The Reference_Resolver object being used.
*/
protected Reference_Resolver
	Resolver					= null;

/**	The default value to be used by the {@link Reference_Resolver} if a
	reference can not be resolved.
<p>
	A null value means that the Reference_Resolver will throw an exception
	if a reference can not be resolved.
<p>
	This value may be overriden by the {@link
	#UNRESOLVED_REFERENCE_PARAMETER} of the configuration file. A
	parameter value of {@link #UNRESOLVED_REFERENCE_THROWS} (case
	insensitive) is equivalent to a null default value.
*/
public static final String
	RESOLVER_DEFAULT_VALUE		= null;	//	Throws an exception.

//	Mathematical expression parser (JCM).
private Parser
	Expression_Parser			= new Parser ();


//	Sources:

/**	Sources table field names.
*/
public static final String[]
	SOURCES_FIELD_NAMES =
		{
		SOURCE_NUMBER_PARAMETER,
		SOURCE_PATHNAME_PARAMETER,
		SOURCE_ID_PARAMETER,
		CONDUCTOR_ID_PARAMETER,
		"Status",
		LOG_PATHNAME_PARAMETER
		};

/**	Sources table fields indexes.
*/
public static int
	SOURCE_NUMBER_FIELD			= 0,
	SOURCE_PATHNAME_FIELD		= 1,
	SOURCE_ID_FIELD				= 2,
	CONDUCTOR_ID_FIELD			= 3,
	STATUS_FIELD				= 4,
	LOG_PATHNAME_FIELD			= 5;
private Fields_Map
	Sources_Map					= null;

/**	Sources table name suffix.
*/
public static final String
	SOURCES_TABLE_NAME_SUFFIX	= "_Sources";
private String
	Sources_Table				= null;
private Vector<Vector<String>>
	//	Cache of potential source records for processing.
	Source_Records				= new Vector<Vector<String>> (0);
private Vector<String>
	//	Current source record being processed.
	Source_Record				= null,
	//	Current source record Status field values.
	Source_Status				= null;

/**	The minimum value for the Max_Source_Records value.

	The value may be set in the configuration by the {@link
	#MIN_SOURCE_RECORDS_PARAMETER}. The default is {@link
	#MIN_SOURCE_RECORDS_DEFAULT}.
<p>
	When operating in batch mode, this value control the minimum
	number of source records to be processed before stopping.
<p>
	@see	#MAX_SOURCE_RECORDS_DEFAULT
*/
public static final int
	MIN_SOURCE_RECORDS_DEFAULT	= 1;
private int
	Min_Source_Records			= MIN_SOURCE_RECORDS_DEFAULT;

/**	The maximum number of unprocessed source records that will be
	obtained when the Conductor cache is refreshed.
<p>
	This value may be set in the configuration by the {@link
	#MAX_SOURCE_RECORDS_PARAMETER}. The default is {@link
	#MAX_SOURCE_RECORDS_DEFAULT}.
<p>
	@see	#MIN_SOURCE_RECORDS_DEFAULT
*/
public static final int
	MAX_SOURCE_RECORDS_DEFAULT	= 1000;
private int
	Max_Source_Records			= MAX_SOURCE_RECORDS_DEFAULT;

/**	Default number of source file availability tests.
<p>
	Due to NFS filesystems latency it is possible that a source file
	that has just been created and registered in a Sources table by a
	Conductor on one system will not appear if accessed too soon by a
	Conductor on another system. The workaround is to repeatedly try to
	see the file, with a ten second delay between tries. The difficulty
	is knowing how many tries to make before giving up. As of this
	writing it seems that sufficient tries should be allowed for up to
	two minutes of trying.
<p>
	The Configuration {@link #SOURCE_AVAILABLE_TRIES_PARAMETER} can be
	used to override the default.
*/
public static final int
	SOURCE_AVAILABLE_TRIES_DEFAULT	= 12;

/**	Maximum number of source file availability tests.

	A maximum of 30 minutes of trying is allowed.
<p>
	@see	#SOURCE_AVAILABLE_TRIES_DEFAULT
*/
public static final int
	SOURCE_AVAILABLE_TRIES_MAX		= 180;

/** When the {@link #SOURCE_AVAILABLE_TRIES_PARAMETER} is this value
	source file availability confirmation is disabled.
*/
public static final int
	SOURCE_AVAILABLE_NO_CHECK		= -1;

private int
	Source_Available_Tries			= SOURCE_AVAILABLE_TRIES_DEFAULT;

//	Source record processing statistics.
private long
	Batch_Sources_Count				= 0,
	Source_Success_Count			= 0,
	Source_Failure_Count			= 0,
	Total_Failure_Count				= 0;

//	Time, in seconds. to wait between trying for new Sources.
/**	The polling interval, in seconds, for unprocessed source
	records when no unprocessed source records are obtained from
	the Sources table.
<p>
	This value may be set by the configuration {@link
	#POLL_INTERVAL_PARAMETER}.
<p>
	A value of zero sets batch mode: Conductor is to stop when no
	unprocessed source records are available and at least
	Min_Source_Records have been processed. If polling for more sources
	is required in batch mode a polling interval of {@link
	#BATCH_POLL_INTERVAL} seconds will be used.
<p>
	@see	#MIN_SOURCE_RECORDS_DEFAULT
*/
public static final int
	DEFAULT_POLL_INTERVAL			= 30,
	BATCH_POLL_INTERVAL				= 5;
private static final int
	UNSET_INTEGER_VALUE				= -999999999;
private volatile int
	Poll_Interval					= DEFAULT_POLL_INTERVAL,
	Poll_Interval_Override			= UNSET_INTEGER_VALUE;


//	Procedures:

/**	Procedures table field names.
*/
public static final String[]
	PROCEDURES_FIELD_NAMES =
		{
		PROCEDURE_SEQUENCE_PARAMETER,
		"Description",
		"Command_Line",
		"Success_Status",
		"Success_Message",
		"Time_Limit",
		"On_Failure"
		};

/**	Procedures table fields indexes.
*/
public static final int
	SEQUENCE_FIELD				= 0,
	DESCRIPTION_FIELD			= 1,
	COMMAND_LINE_FIELD			= 2,
	SUCCESS_STATUS_FIELD		= 3,
	SUCCESS_MESSAGE_FIELD		= 4,
	TIME_LIMIT_FIELD			= 5,
	ON_FAILURE_FIELD			= 6;
protected Fields_Map
	Procedures_Map				= null;

/**	Procedures table name suffix.
*/
public static final String
	PROCEDURES_TABLE_NAME_SUFFIX	= "_Procedures";

/**	The name of the pipeline procedures table in the database.
*/
protected String
	Procedures_Table			= null;

/**	The content of the pipeline procedures table, without the field
	names, sorted by sequence number.
*/
protected Vector<Vector<String>>
	Procedure_Records			= new Vector<Vector<String>> (0);
//	A copy of the procedure record currently being used.
private Vector<String>
	Procedure_Record			= null;
private boolean
	Procedure_has_Status;

private static String
	COMMAND_LINE_ARGUMENTS_DELIMITERS = " \t\n\r";

//	Booleans for Run_Procedure.
private static final boolean
	NORMAL						= true,
	ON_FAILURE					= false;

/**	Conductor procedure completion code.
<p>
	@see	#Status_Indicators(String)
*/
public static final int
	PROCEDURE_SUCCESS			= 0,
	PROCEDURE_FAILURE			= 1,
	INACCESSIBLE_FILE			= -1,
	UNRESOLVABLE_REFERENCE		= -2,
	NO_PROCEDURE				= -3,
	PROCEDURE_TIMEOUT			= -4,
	BAD_REGEX					= -5,
	INVALID_DATABASE_ENTRY		= -6;

/**	Conductor status failure code descriptions.
<p>
	@see	#Status_Conductor_Code_Description
*/
public static final String[]
	FAILURE_DESCRIPTION =
		{
		"Inaccessable file",
		"Unresolvable reference in Procedures table",
		"Procedure could not be executed",
		"Procedure timeout",
		"Invalid regular expression",
		"Invalid database table entry"
		};

/**	The default for whether or not empty Success_Status and
	Success_Message fields in a procedure definition may imply any
	completion of the procedure is a success.
<p>
	This value may be overridden by the {@link #EMPTY_SUCCESS_ANY_PARAMETER}
	configuruation parameter.
*/
public static final boolean
	DEFAULT_EMPTY_SUCCESS_ANY	= false;
private boolean
	Empty_Success_Any			= DEFAULT_EMPTY_SUCCESS_ANY;


//	Pipeline processing:

private Thread
	Processor					= null,
	Main_Thread_Waiting			= null;

/**	The default maximum number of sequential source processing failures.
*/
public static final int
	DEFAULT_STOP_ON_FAILURE		= 0;
private volatile int
	Stop_on_Failure				= DEFAULT_STOP_ON_FAILURE,
	Stop_on_Failure_Override	= UNSET_INTEGER_VALUE,
	Sequential_Failures			= 0;

/**	Processing state: Source records are being processed.
*/
public static final int
	RUNNING						= 3;

/**	Processing state: No unprocessed source records are available
	and the {@link #Poll_Interval() poll interval} for new records is
	positive.
<p>
	Between attempts to {@link #Load_Source_Records() load new source
	records} that have not been unprocessed the processing thread will
	sleep for the poll interval. Once additional unprocessed source
	records have been obtained the {@link #RUNNING} state will be
	entered, unless the {@link #RUN_TO_WAIT} state has been set.
*/
public static final int
	POLLING						= 2;

/**	Processing state: When the current source record completes processing
	the {@link #WAITING} state will be entered unless a failure condition
	caused the {@link #HALTED} state to occur.
<p>
	<b>N.B.</b>: If the Conductor is repeatedly trying to confirm the
	existence of a source file or to reconnect to the database when the
	stop request is received retrying will be discontinued which can be
	expected to result in a source record processing failure or
	database connectivity failure halt.
<p>
	This state is only entered when the user requests that processing
	{@link #Stop() stop} while the {@link #RUNNING} or {@link #POLLING}
	state is in effect.
*/
public static final int
	RUN_TO_WAIT					= 1;

/**	Processing state: Idle; waiting for a {@link #Start() start} request.
*/
public static final int
	WAITING						= -1;

/**	Processing state: A failure condition caused processing to halt.
<p>
	The problem may be the result of the maximum number of sequential
	source records processing procedure failures having occured, a
	database access failure, or some other system error. This state
	remains in effect until a {@link #Start() start} request is received.
*/
public static final int
	HALTED						= -2;

private volatile int
	Processing_State			= WAITING;

//	Indicates that the pipeline processing thread is sleeping.
private volatile boolean
	Sleeping					= false;

//	Any exception that occurs in the processing thread run.
private Exception
	Processing_Exception		= null;
	
private Vector<Processing_Listener>
	Processing_Listeners		= new Vector<Processing_Listener> ();


//	Miscellaneous.

private static final char
	FILE_SEPARATOR				= File.separatorChar,
	FILE_SEPARATOR_SUBSTITUTE	= '%';

protected static final String
	NL							= System.getProperty ("line.separator");


//	Logging:

private Styled_Multiwriter
	Logger						= new Styled_Multiwriter ();

//	Log file.
private String
	Log_File_Directory			= null;
private Writer
	Log_File_Writer				= null;

//	Procedure output.
private Stream_Logger
	stdout_Logger				= null,
	stderr_Logger				= null;

/**	Prefix applied to procedure stdout lines.
*/
public static final String
	STDOUT_NAME					= "stdout";

/**	Prefix applied to procedure stderr lines.
*/
public static final String
	STDERR_NAME					= "stderr";

/**	Marks the beginning of source file processing in a log file.
*/
public static final String
	SOURCE_FILE_LOG_DELIMITER =
	"======================================================================" + NL;

/**	Marks the beginning of procedure processing in a log file.
*/
public static final String
	PROCEDURE_LOG_DELIMITER =
	"----------------------------------------------------------------------" + NL;

/**	Marks the beginning of On_Failure procedure processing in a log file.
*/
public static final String
	ON_FAILURE_PROCEDURE_LOG_DELIMITER =
	"||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||" + NL;

private static final SimpleAttributeSet
	NOTICE_STYLE				= new SimpleAttributeSet (),
	HIGHLIGHT_STYLE				= new SimpleAttributeSet (),
	MARKER_STYLE				= new SimpleAttributeSet ();
static
	{
	StyleConstants.setBold			(NOTICE_STYLE, true);
	StyleConstants.setForeground	(NOTICE_STYLE, Colors.NOTICE_STYLE);

	StyleConstants.setForeground	(HIGHLIGHT_STYLE, Colors.HIGHLIGHT_STYLE);

	StyleConstants.setFontFamily	(MARKER_STYLE, "Monospaced");
	StyleConstants.setBold			(MARKER_STYLE, true);
	}


//	Exit status:

/**	Conductor success exit status (0).
*/
public static final int
	EXIT_SUCCESS				= 0;

/**	Command line syntax problem exit status (1).
*/
public static final int
	EXIT_COMMAND_LINE_SYNTAX	= 1;

/**	Configuration problem exit status (2).
*/
public static final int
	EXIT_CONFIGURATION_PROBLEM	= 2;

/**	Configuration problem exit status (3).
*/
public static final int
	EXIT_DATABASE_PROBLEM		= 3;

/**	I/O failure exit status (4).
*/
public static final int
	EXIT_IO_FAILURE				= 4;

/**	The number of sequential source processing failures reached the
	Stop-on-Failure amount.
*/
public static final int
	EXIT_TOO_MANY_FAILURES		= 6;

/**	The {@link #Require_Stage_Manager required} Stage_Manager
	connection could not be established.
*/
public static final int
	EXIT_STAGE_MANAGER			= 7;

/**	An unexpected exception occured (9).
*/
public static final int
	EXIT_UNEXPECTED_EXCEPTION	= 9;


//	This is static so it can be used by Exit if construction fails.
/**	A list of recipients to receive notification of Conductor failure.
<p>
	The list is obtained from the Configuration {@link #NOTIFY_PARAMETER}.
*/
private static Vector<Object>
	Notify_List					= null;


// Debug control.
private static final int
	DEBUG_OFF					= 0,
	DEBUG_SETUP					= 1 << 0,
	DEBUG_CONSTRUCTOR			= 1 << 1,
	DEBUG_CONFIG				= 1 << 2,
	DEBUG_DATABASE				= 1 << 3,
	DEBUG_PIPELINE				= 1 << 4,
	DEBUG_GET_SOURCE			= 1 << 5,
	DEBUG_PROCESS_SOURCE		= 1 << 6,
	DEBUG_SET_PARAMETER			= 1 << 7,
	DEBUG_PROCEDURE_STATUS		= 1 << 8,
	DEBUG_RUN_PROCEDURE			= 1 << 9,
	DEBUG_PARSE_COMMAND_LINE	= 1 << 10,
	DEBUG_TABLE_UPDATE			= 1 << 11,
	DEBUG_DATABASE_CONNECTION	= 1 << 12,
	DEBUG_MESSAGES				= 1 << 13,
	DEBUG_MANAGEMENT			= 1 << 14,
	DEBUG_PROCESSING_EVENT		= 1 << 15,
	DEBUG_NOTIFICATION			= 1 << 16,
	DEBUG_EXIT					= 1 << 17,
	DEBUG_TO_FILE				= 1 << 18,
	DEBUG_ALL					= -1,

	DEBUG						= DEBUG_OFF;

private static final String
	DEBUG_FILENAME_BASE			= "Conductor";
private static String
	DEBUG_Filename				= null;
private static final PrintStream
	stdout						= System.out;

/*==============================================================================
	Constructors
*/
/**	Construct a Conductor for a pipeline from a Configuration.
<p>
	The Configuration object is expected to contain all the necessary
	information Conductor needs to connect to the database as well as
	any other Conductor parameters it might use.
<p>
	@param	pipeline			The name of the pipeline to be managed.
	@param	configuration		A Configuration object.
	@param	database_server_name	The name of a Configuration Group that
		will provide the database server access parameters.
	@throws	Database_Exception	if there is a problem connecting to the
		database.
	@throws	Configuration_Exception	if there is a problem with the
		configuration file.
	@throws	IOException	if a connection could not be made to the
		Stage_Manager and {@link #Require_Stage_Manager} is true;
*/
public Conductor
	(
	String			pipeline,
	Configuration	configuration,
	String			database_server_name
	)
throws
	Database_Exception,
	Configuration_Exception,
	IOException
{
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		(">>> Conductor");
Pipeline = pipeline;

//	Setup the initial Configuration.
The_Configuration = Preconfigure (configuration);

//	Establish the Database connection.
try
	{
	The_Database = new Database (The_Configuration);
	if ((Database_Server_Name = database_server_name) == null)
		Database_Server_Name = The_Database.Default_Server_Name ();
	if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
		System.out.println
			("    Connecting to Database Server " + Database_Server_Name);
	The_Database.Connect (Database_Server_Name);
	}
catch (Database_Exception exception)
	{
	throw Database_Error
		("Unable to connect to the database." + NL
		+ exception.getMessage ());
	}

//	Construct the Reference_Resolver.
Resolver = new Reference_Resolver (The_Database);

//	Update basic configuration parameters.
try {Postconfigure (The_Configuration);}
catch (Configuration_Exception exception)
	{
	try {The_Database.Disconnect ();}
	catch (Database_Exception except) {}
	throw exception;
	}

//	Initialize the Procedures and Sources records.
Load_Procedure_Records ();
Load_Source_Records ();

//	Establish a connection to the Stage_Manager.
String
	report = Connect_to_Stage_Manager ();
if (report != null)
	stdout.println (report);
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("<<< Conductor");
}

/**	Construct a Conductor for a pipeline from a Configuration.
<p>
	The Configuration object is expected to contain all the necessary
	information Conductor needs to connect to the database as well as
	any other Conductor parameters it might use.
<p>
	The database server access parameters will be obtained from the
	Configuration Group named by the first entry in the Server parameter
	list.
<p>
	@param	pipeline			The name of the pipeline to be managed.
	@param	configuration		A Configuration object.
	@throws	Database_Exception	if there is a problem connecting to the
		database.
	@throws	Configuration_Exception	if there is a problem with the
		configuration file.
	@throws	IOException	if a connection could not be made to the
		Stage_Manager and {@link #Require_Stage_Manager} is true;
*/
public Conductor
	(
	String			pipeline,
	Configuration	configuration
	)
throws
	Database_Exception,
	Configuration_Exception,
	IOException
{this (pipeline, configuration, null);}


/**	Constructs an uninititalized Conductor.
*/
protected Conductor ()
{}
	
/*==============================================================================
	Configuration
*/
/**	Set the effective configuration.
<p>
	If no Configuration is provided an effort is made to load the {@link
	#DEFAULT_CONFIGURATION_FILENAME}.
<p>
	Essential configuration information is obtained from, and set in,
	the {@link #CONDUCTOR_GROUP} parameters of the Configuration:
<dl>
<dt>{@link Local_Theater#CLASS_ID_PARAMETER_NAME} - Set
<dd>The {@link #ID} of this class.

<dt>{@link #CONFIGURATION_SOURCE_PARAMETER} - Set
<dd>The {@link Configuration#Source() source} of the Configuration that
	is being used.

<dt>{@link #HOSTNAME_PARAMETER} - Set
<dd>The {@link Host#FULL_HOSTNAME} of the host system.

<dt>{@link #CONDUCTOR_ID_PARAMETER} - Set
<dd>The identifying name of this Conductor. This will be the {@link
	Host#SHORT_HOSTNAME} of the system. If the host system {@link #PID()
	process ID} can be obtained it is appended to the hostname after a
	colon (':') delimiter.

<dt>{@link #NOTIFY_PARAMETER} - Get
<dd>A list of email address to be sent a {@link Notify} message if
	Conductor stops because the {@link #STOP_ON_FAILURE_PARAMETER} limit
	has been reached or an exception has been thrown. If this parameter
	is not present or has an empty value no notification message will
	be sent.

<dt>{@link #REQUIRE_STAGE_MANAGER_PARAMETER} - Get and Set
<dd>A {@link Configuration#Enabled(String, boolean) flag} that determines
	if a connection to a Stage_Manager is required for this Conductor.
	<b>N.B.</b>: This parameter is only read once when the Conductor
	is first configured; when the Conductor is reconfigured on being
	restarted after a Stop or Halt state the initial value is reset in
	the internal Configuration regardless of any change to the external
	configuration source. Default: {@link #Require_Stage_Manager}.

<dt>{@link #STAGE_MANAGER_PORT_PARAMETER} - Get and Set
<dd>The port number to use for connecting to a Stage_Manager. Default:
	{@link PIRL.Conductor.Maestro.Local_Theater#Default_Port() the
	default port for Theater Management}.

<dt>{@link #STAGE_MANAGER_TIMEOUT_PARAMETER} - Get and Set
<dd>The amount of time (seconds) to wait for a Stage_Manager connection
	to complete before the connection attempt fails.

<dt>{@link #STAGE_MANAGER_PASSWORD_PARAMETER} - Get
<dd>The password required to authenticate a Stage_Manager connection. If
	this parameter is present its value is masked out in the internal
	Configuration. If this parameter is not present, or its value is
	empty, and the Stage_Manager requires an authenticated connection the
	connection attempt will fail.

<dt>{@link #HELLO_PORT_PARAMETER_NAME} - Get and Set
<dd>The port number to use when listening for a Stage_Manager "Hello"
	broadcast that it is ready for connections.

<dt>{@link #HELLO_ADDRESS_PARAMETER_NAME} - Get and Set
<dd>The multicast address to use when listening for a Stage_Manager "Hello"
	broadcast that it is ready for connections.

<dt>{@link #SOURCE_SUCCESS_COUNT} - Set
<dd>The number of source records that have been successfully processed
	since this Conductor was started.

<dt>{@link #SOURCE_FAILURE_COUNT} - Set
<dd>The number of source records that have resulted in a failed procedure
	condition during processing.

<dt>{@link #TOTAL_FAILURE_COUNT} - Set
<dd>The total number of source records that failed to be processed for
	any reason.
</dl>
<p>
	@param	configuration	The Configuration to be used. If null the
		{@link #DEFAULT_CONFIGURATION_FILENAME} will be tried. If that
		fails the filename will be qualified relative to the location of
		this class in the CLASSPATH.
	@return	The Configuration that was used. This will be the same as
		the specified configuation if it was not null; otherwise it will
		be the default configuration that was loaded.
	@throws	Configuration_Exception	If the specified configuration is
		null and a default configuration source could not be found and
		loaded; or there was a problem setting a configuration value.
*/
protected Configuration Preconfigure
	(
	Configuration	configuration
	)
throws Configuration_Exception
{
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		(">>> Conductor.Preconfigure");
if (configuration == null)
	{
	if ((DEBUG & DEBUG_CONFIG) != 0)
		System.out.println
			("    Trying " + Configuration.Default_Source ());
	try {configuration = new Configuration ((String)null);}
	catch (IllegalArgumentException exception)
		{
		throw new Configuration_Exception (ID + NL
			+ "Unable to find the configuration file \""
				+ Configuration.Default_Source () + "\".");
		}
	}
//	Reset the default configuration source to the current source.
Configuration.Default_Source (configuration.Source ());
configuration.Case_Sensitive (false);

//	Done early so the value will be available for the DEBUG_Filename.
Theater_Management.Default_Port
	((int)configuration.Get_Number (Config_Pathname
		(STAGE_MANAGER_PORT_PARAMETER),
			Theater_Management.Default_Port ()));

if (The_Configuration == null)
	{
	//	First time initialization only.

	//	ID of this Conductor process.
	Conductor_ID = Host.SHORT_HOSTNAME;
	int
		PID = PID ();
	if (PID != 0)
		Conductor_ID += ":" + PID;

	//	Stage_Manager required?
	Require_Stage_Manager = configuration.Enabled (Config_Pathname
		(REQUIRE_STAGE_MANAGER_PARAMETER),
			Require_Stage_Manager);

	if ((DEBUG & DEBUG_TO_FILE) != 0)
		{
		if (DEBUG_Filename == null)
			{
			DEBUG_Filename =
				System.getProperty ("user.home")
				+ File.separatorChar + DEBUG_FILENAME_BASE
				+ '-' + Conductor_ID
				+ '_' + Theater_Management.Default_Port ()
				+ ".DEBUG";
			System.out.println
				("*** Redirecting System.out to " + DEBUG_Filename);
			try
				{
				System.setOut
					(new PrintStream
						(new FileOutputStream (DEBUG_Filename), true));
				}
			catch (IOException exception)
				{
				System.out.println
					("!!! Unable to open " + DEBUG_Filename);
				}
			System.out.println
				(DEBUG_Filename + NL
				+ ID + NL);
			}
		}
	}
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		("    Conductor.Preconfigure: Input Configuration -" + NL
		+ configuration.Description ());

//	Class ID.
Config_Value (configuration, Local_Theater.CLASS_ID_PARAMETER_NAME,
	ID);

//	ID of this Conductor process.
Config_Value (configuration, CONDUCTOR_ID_PARAMETER,
	Conductor_ID);

//	Hostname.
Config_Value (configuration, HOSTNAME_PARAMETER,
	Host.FULL_HOSTNAME);

//	Configuration source.
Config_Value (configuration, CONFIGURATION_SOURCE_PARAMETER,
	configuration.Source ());

//	Stage_Manager:

Config_Value (configuration, REQUIRE_STAGE_MANAGER_PARAMETER,
	String.valueOf (Require_Stage_Manager));

Config_Value (configuration, STAGE_MANAGER_PORT_PARAMETER,
	new Integer (Theater_Management.Default_Port ()));

Theater_Management.Receive_Timeout
	((int)configuration.Get_Number (Config_Pathname
		(STAGE_MANAGER_TIMEOUT_PARAMETER),
			Theater_Management.Receive_Timeout ()));
Config_Value (configuration, STAGE_MANAGER_TIMEOUT_PARAMETER,
	new Integer (Theater_Management.Receive_Timeout ()));

Theater_Management.Key (Stage_Manager_Key = Config_Value (configuration,
	STAGE_MANAGER_PASSWORD_PARAMETER));
if (Stage_Manager_Key != null)
	//	Mask out the Stage_Manager password, and only this password.
	Config_Value (configuration, STAGE_MANAGER_PASSWORD_PARAMETER,
		Processing_Changes.MASKED_PASSWORD);

Theater_Management.Auto_Open
	(! Require_Stage_Manager,
	(int)configuration.Get_Number (Config_Pathname
		(HELLO_PORT_PARAMETER_NAME),
			Theater_Management.Hello_Listener_Port ()),
	Config_Value (configuration,
		HELLO_ADDRESS_PARAMETER_NAME));
Config_Value (configuration, HELLO_PORT_PARAMETER_NAME,
	new Integer (Theater_Management.Hello_Listener_Port ()));
Config_Value (configuration, HELLO_ADDRESS_PARAMETER_NAME,
	Theater_Management.Hello_Listener_Address ());

//	Who to notify on Conductor failure.
Notify_List = configuration.Get (Config_Pathname (NOTIFY_PARAMETER));
if (Notify_List.isEmpty ())
	Notify_List = null;
else
	for (int
			index = 0;
			index < Notify_List.size ();
			index++)
		if (! (Notify_List.get (index) instanceof String))
			throw new Configuration_Exception (Error_Message
				("Unexpected subarray in the configuration " + NOTIFY_PARAMETER
					+ " parameter at entry " + index + '.'));

//	Processing status values.
Config_Value (configuration, SOURCE_SUCCESS_COUNT,
	Source_Success_Count);
Config_Value (configuration, SOURCE_FAILURE_COUNT,
	Source_Failure_Count);
Config_Value (configuration, TOTAL_FAILURE_COUNT,
	Total_Failure_Count);

if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		("    Pre-Configuration -" + NL
		+ configuration.Description () + NL
		+"<<< Conductor.Preconfigure");
return configuration;
}

/**	Update the configuration and application control values after
	the Database and Reference_Resolver have been constructred.
<p>
	The following {@link #CONDUCTOR_GROUP} parameters of the
	Configuration are affected:
<dl>
<dt>{@link #DATABASE_SERVER_NAME_PARAMETER} - Set
<dd>The name of the Group of parameters that provided the database server
	access information used to establishs a {@link Database#Connect(String)
	connection}.

<dt>{@link #DATABASE_TYPE_PARAMETER} - Set
<dd>The Type of {@link PIRL.Database.Data_Port} class used to access the
	database server.

<dt>{@link #DATABASE_HOSTNAME_PARAMETER} - Set
<dd>The hostname where the database server is located.

<dt>{@link #CATALOG_PARAMETER} - Get and Set
<dd>The name of the database catalog containing the pipeline tables.
	This value is only obtained once from the initial Configuration, and
	then it is obtained from the {@link #CONDUCTOR_GROUP} only if it
	could not be obtained as part of the user specified pipeline name
	nor from the database access parameters of the {@link
	#DATABASE_SERVER_NAME_PARAMETER} Group. This is a required value. If
	it can not be obtained a Configuration_Exception will be thrown. It
	is always set in the specified configuration.

<dt>{@link #PIPELINE_PARAMETER} - Set
<dd>The name of the pipeline to be processed. This required parameter is
	user specified.

<dt>{@link #PROCEDURES_TABLE_PARAMETER} - Set
<dd>The name of the procedures definitions table in the database.

<dt>{@link #SOURCES_TABLE_PARAMETER} - Set
<dt>The name of the sources records table in the database.

<dt>{@link #RECONNECT_TRIES_PARAMETER} - Get and Set
<dd>The maximum number of database server {@link #Connect_to_Database()
	reconnect} tries to do if the database connection is lost.

<dt>{@link #UNRESOLVED_REFERENCE_PARAMETER} - Get and Set
<dd>The default value to be used by the {@link Reference_Resolver} if a
	reference can not be resolved. A parameter value of {@link
	#UNRESOLVED_REFERENCE_THROWS} (case insensitive) means that the
	Reference_Resolver will throw an exception if a reference can not be
	resolved. If the Configuration does not contain this parameter the
	{@link #RESOLVER_DEFAULT_VALUE} will be used; a null default value
	means that the Reference_Resolver will throw an exception if a
	reference can not be resolved.

<dt>{@link #LOG_PATHNAME_PARAMETER} or {@link #LOG_DIRECTORY_PARAMETER} - Get
<dd>The directory where log files will be written if a source record does
	not specify a log directory. The default log directory is the current
	working directory.

<dt>{@link #EMPTY_SUCCESS_ANY_PARAMETER} - Get and Set
<dd>A {@link Configuration#Enabled(String, boolean) flag} that
	determines whether or not empty Success_Status and Success_Message
	fields in a procedure definition may imply any completion of the
	procedure is a success. If this parameter is not present the
	{@link #DEFAULT_EMPTY_SUCCESS_ANY} value will be used.

<dt>{@link #MIN_SOURCE_RECORDS_PARAMETER} - Get and Set
<dd>The minimum number of unprocessed source records to be obtained from
	the database when the Conductor cache is refreshed. When Contuctor
	is operating in batch mode (the {@link #POLL_INTERVAL_PARAMETER
	polling interval} is zero) this value specifies the minimum number
	of source records to be processed before processing will stop.
	The minimum for this value is 1.

<dt>{@link #MAX_SOURCE_RECORDS_PARAMETER} - Get and Set
<dd>The maximum number of unprocessed source records to be obtained from
	the database when the Conductor cache is refreshed. This value is
	used to prevent excessive memory consumption when a large number of
	unprocessed source records are available. The minimum for this value
	is limited to Min_Source_Records.

<dt>{@link #SOURCE_AVAILABLE_TRIES_PARAMETER} - Get and Set
<dd>The maximum number of times to try and confirm that the
	Source_Pathname is available before the source record is processed.
	If this parameter is not present the {@link
	#SOURCE_AVAILABLE_TRIES_DEFAULT} will be used. The value is limited
	to be no more than {@link #SOURCE_AVAILABLE_TRIES_MAX}. If the value
	is negative it will be set to (@link #SOURCE_AVAILABLE_NO_CHECK} and
	source availability confirmation will be disabled. <b>N.B.</b> Source
	availability confirmation ensures that procedures will not fail due
	to a missing source file; disabling this confirmation would be
	appropriate for a pipeline that does not process a Source_Pathname,
	or for which the Source_Pathname is not a filename but some other
	information used by a procedure.

<dt>{@link #POLL_INTERVAL_PARAMETER} - Get and Set
<dd>The polling interval, in seconds, for unprocessed source records
	when the no unprocessed source records are obtained. A value of zero
	means that Conductor is to stop if when no unprocessed source
	records are available. If this parameter is not present the {@link
	#DEFAULT_POLL_INTERVAL} will be used.

<dt>{@link #STOP_ON_FAILURE_PARAMETER} - Get and Set
<dd>The number of sequential source record processing failures at which
	Conductor is to stop processing. A value less than or equal to zero
	means that processing is never to stop due to sequential failures.
	This parameter may have a boolean value which, if true, is
	equivalent to the number one; if false sequential failures will
	never cause processing to stop. If this parameter is not present the
	{@link #DEFAULT_STOP_ON_FAILURE} value will be used.
</dl>
<p>
	@param	configuration	The Configuration to be used. If null nothing
		is done.
	@throws	Configuration_Exception	If there was a problem setting a
		configuration value.
*/
protected void Postconfigure
	(
	Configuration	configuration
	)
	throws Configuration_Exception
{
if (configuration == null)
	return;
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		(">>> Conductor.Postconfigure");

//	Database information.
if (Database_Server_Name != null)
	Config_Value (configuration, DATABASE_SERVER_NAME_PARAMETER,
		Database_Server_Name);
Config_Value (configuration, DATABASE_TYPE_PARAMETER,
	The_Database
		.Data_Port ()
		.Configuration ()
		.Get_One (Database.TYPE));
Config_Value (configuration, DATABASE_HOSTNAME_PARAMETER,
	The_Database
		.Data_Port ()
		.Configuration ()
		.Get_One (Configuration.HOST));


if (Procedures_Table == null)
	{
	//	First time initialization only.

	//	Database catalog containing the tables.
	Catalog = null;
	try {Catalog = The_Database.Catalog_Name (Pipeline);}
	catch (Database_Exception exception) {}
	if (Catalog == null ||
		Catalog.length () == 0)
		{
			//	Try for a Catalog name from the Configuration Conductor group.
		if ((Catalog = Config_Value (configuration, CATALOG_PARAMETER))
				== null &&
			//	Try for a Catalog name from the Database server group.
			(Catalog = The_Database.Configuration ().Get_One (CATALOG_PARAMETER))
				== null)
			throw new Configuration_Exception (Error_Message
				("The database catalog "
					+ "containing the procedures and sources tables" + NL
				+"could not be determined from the pipeline name \""
					+ Pipeline + '"' + NL
				+"nor from a \"" + CATALOG_PARAMETER + "\" parameter in the "
					+ configuration.Source () + " configuration file."));
		try {Pipeline = The_Database.Table_Reference (Catalog, Pipeline);}
		catch (Database_Exception exception)
			{
			throw new Configuration_Exception (Error_Message
				("Unable to assemble the fully qualified pipleline name" + NL
				+"  from the catalog \"" + Catalog + "\" and pipeline \""
					+ Pipeline + "\" names." + NL
				+ exception.getMessage ()));
			}
		}

	//	Table references for the procedures and sources.
	try {Sources_Table = The_Database.Table_Reference
			(Catalog, Pipeline + SOURCES_TABLE_NAME_SUFFIX);}
	catch (Database_Exception exception)
		{
		throw new Configuration_Exception (Error_Message
			("Unable to assemble the fully qualified sources table name" + NL
			+"  from the catalog \"" + Catalog + "\" and sources \""
				+ Pipeline + SOURCES_TABLE_NAME_SUFFIX + "\" names." + NL
			+ exception.getMessage ()));
		}
	try {Procedures_Table = The_Database.Table_Reference
		(Catalog, Pipeline + PROCEDURES_TABLE_NAME_SUFFIX);}
	catch (Database_Exception exception)
		{
		throw new Configuration_Exception (Error_Message
			("Unable to assemble the fully qualified procedures table name" + NL
			+"  from the catalog \"" + Catalog + "\" and procedures \""
				+ Pipeline + PROCEDURES_TABLE_NAME_SUFFIX + "\" names." + NL
			+ exception.getMessage ()));
		}
	}
Config_Value (configuration, CATALOG_PARAMETER,
	Catalog);
try {Config_Value (configuration, PIPELINE_PARAMETER,
		The_Database.Table_Name (Pipeline));}
catch (Database_Exception exception)
	{
	throw new Configuration_Exception (Error_Message
		("Unable to extract the pipeline name from \"" + Pipeline + "\"." + NL
		+ exception.getMessage ()));
	}
Config_Value (configuration, SOURCES_TABLE_PARAMETER,
	Sources_Table);
Config_Value (configuration, PROCEDURES_TABLE_PARAMETER,
	Procedures_Table);


//	Database reconnect parameters.
if ((Reconnect_Tries = (int)configuration.Get_Number (Config_Pathname
		(RECONNECT_TRIES_PARAMETER), Reconnect_Tries)) < 0)
	Reconnect_Tries = 0;
Config_Value (configuration, RECONNECT_TRIES_PARAMETER,
	new Integer (Reconnect_Tries));

if ((Reconnect_Delay = (int)configuration.Get_Number (Config_Pathname
		(RECONNECT_DELAY_PARAMETER), Reconnect_Delay)) < 0)
	Reconnect_Delay = 0;
Config_Value (configuration, RECONNECT_DELAY_PARAMETER,
	new Integer (Reconnect_Delay));


//	Unresolved reference default value.
String
	default_value =
		Config_Value (configuration, UNRESOLVED_REFERENCE_PARAMETER);
if (default_value == null)
	Config_Value (configuration, UNRESOLVED_REFERENCE_PARAMETER,
		((default_value = RESOLVER_DEFAULT_VALUE) == null) ?
			UNRESOLVED_REFERENCE_THROWS : default_value);
else if (default_value.toUpperCase ().startsWith (UNRESOLVED_REFERENCE_THROWS))
	default_value = null;
Resolver.Default_Value (default_value);


//	Log file pathname.
Log_File_Directory = Config_Value (configuration, LOG_PATHNAME_PARAMETER);
if (Log_File_Directory == null ||
	Log_File_Directory.length () == 0)
	//	Try for the alternate log directory parameter.
	Log_File_Directory = Config_Value (configuration, LOG_DIRECTORY_PARAMETER);

//	Empty Success fields may imply any completion is success.
Empty_Success_Any = configuration.Enabled (Config_Pathname
	(EMPTY_SUCCESS_ANY_PARAMETER),
		DEFAULT_EMPTY_SUCCESS_ANY);
Config_Value (configuration, EMPTY_SUCCESS_ANY_PARAMETER,
		String.valueOf (Empty_Success_Any));


//	Minimum number of source records to process in batch mode.
Min_Source_Records = (int)configuration.Get_Number (Config_Pathname
	(MIN_SOURCE_RECORDS_PARAMETER),
		MIN_SOURCE_RECORDS_DEFAULT);
if (Min_Source_Records < 1)
	Min_Source_Records = 1;
Config_Value (configuration, MIN_SOURCE_RECORDS_PARAMETER,
	new Integer (Min_Source_Records));


//	Maximum number of source records to obtain in a select.
Max_Source_Records = (int)configuration.Get_Number (Config_Pathname
	(MAX_SOURCE_RECORDS_PARAMETER),
		MAX_SOURCE_RECORDS_DEFAULT);
if (Max_Source_Records < Min_Source_Records)
	Max_Source_Records = Min_Source_Records;
Config_Value (configuration, MAX_SOURCE_RECORDS_PARAMETER,
	new Integer (Max_Source_Records));


//	Source file availability retries.
Source_Available_Tries = (int)configuration.Get_Number (Config_Pathname
	(SOURCE_AVAILABLE_TRIES_PARAMETER),
		SOURCE_AVAILABLE_TRIES_DEFAULT);
if (Source_Available_Tries > SOURCE_AVAILABLE_TRIES_MAX)
	Source_Available_Tries = SOURCE_AVAILABLE_TRIES_MAX;
if (Source_Available_Tries < 0)
	Source_Available_Tries = SOURCE_AVAILABLE_NO_CHECK;
Config_Value (configuration, SOURCE_AVAILABLE_TRIES_PARAMETER,
	new Integer (Source_Available_Tries));


//	Source records poll interval.
Poll_Interval = (int)configuration.Get_Number (Config_Pathname
	(POLL_INTERVAL_PARAMETER), DEFAULT_POLL_INTERVAL);
if (Poll_Interval < 0)
	Poll_Interval = 0;
Config_Value (configuration, POLL_INTERVAL_PARAMETER,
	new Integer (Poll_Interval));


//	When to stop processing after sequential failures.
Stop_on_Failure = DEFAULT_STOP_ON_FAILURE;
String
	text = Config_Value (configuration, STOP_ON_FAILURE_PARAMETER);
if (text != null)
	{
	if (text.equalsIgnoreCase ("ENABLED") ||
		text.equalsIgnoreCase ("TRUE") ||
		text.equalsIgnoreCase ("YES") ||
		text.equalsIgnoreCase ("ON") ||
		text.equalsIgnoreCase ("Y"))
		Stop_on_Failure = 1;
	else
		{
		try
			{
			Stop_on_Failure = Integer.parseInt (text);
			if (Stop_on_Failure < 0)
				Stop_on_Failure = 0;
			}
		catch (NumberFormatException exception)
			{Stop_on_Failure = 0;}
		}
	}
Config_Value (configuration, STOP_ON_FAILURE_PARAMETER,
	new Integer (Stop_on_Failure));

if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		("    Post-Configuration -" + NL
		+ configuration.Description () + NL
		+"<<< Conductor.Postconfigure");
}


private Configuration Reconfigure ()
	throws Configuration_Exception
{
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		(">>> Conductor.Reconfigure");
Configuration
	configuration = Preconfigure (null);
Postconfigure (configuration);

//	Apply override values that are not unset.
if (Poll_Interval_Override != UNSET_INTEGER_VALUE)
	Config_Value (configuration, POLL_INTERVAL_PARAMETER,
		new Integer (Poll_Interval = Poll_Interval_Override));
if (Stop_on_Failure_Override != UNSET_INTEGER_VALUE)
	Config_Value (configuration, STOP_ON_FAILURE_PARAMETER,
		new Integer (Stop_on_Failure = Stop_on_Failure_Override));

The_Configuration = configuration;
Report_Processing_Event (new Processing_Changes ()
		.Configuration (The_Configuration));

Resolver.Parameters (configuration);

if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		("<<< Conductor.Reconfigure");
return configuration;
}

/**	Get a String parameter value from the configuration.
<p>
	The parameter is sought in the CONDUCTOR_GROUP, but defaults are
	acceptable.
<p>
	If a value is found it is resolved for embedded references. An
	unresolved reference that would throw a Database_Exception results
	in a null value. A syntax error (ParseException) leaves the value
	unchanged.
<p>
	@param	name	The name of the Assignment parameter for which to get
		a value. If not an absolute pathname an absolute pathname for the
		name in the {@link #CONDUCTOR_GROUP} is used.
	@return	The first value String for the named parameter. This will
		be null if no parameter by that name exists, or it contained
		an unresolvable reference.
*/
protected String Config_Value
	(
	String			name
	)
{return Config_Value (The_Configuration, name);}


private String Config_Value
	(
	Configuration	configuration,
	String			name
	)
{
name = configuration.Get_One (Config_Pathname (name));
if (name != null &&
	Resolver != null)
	{
	try {name = Resolver.Resolve  (name);}
	catch (Database_Exception exception) {name = null;}
	catch (ParseException exception) {}
	catch (Unresolved_Reference exception) {}
	}
return name;
}

/**	Set a parameter in the configuration.
<p>
	The parameter is set in the {@link #CONDUCTOR_GROUP}.
<p>
	@param	name	The name of the Assignment parameter to have its
		value set. If not an absolute pathname an absolute pathname for the
		name in the {@link #CONDUCTOR_GROUP} is used.
	@param	value	An Object to use for the parameter's value.
		<b.N.B.</b>: If null, the parameter will have no value; it will
		be a Token.
	@return	true if an existing parameter by the same name was
		replaced; false if the parameter is being set for the
		first time.
	@throws	Configuration_Exception	If there was a problem setting
		the parameter.
	@see	Configuration#Set(String, Object)
*/
protected boolean Config_Value
	(
	String			name,
	Object			value
	)
throws Configuration_Exception
{return Config_Value (The_Configuration, name, value);}


private static boolean Config_Value
	(
	Configuration	configuration,
	String			name,
	Object			value
	)
throws Configuration_Exception
{
if ((DEBUG & DEBUG_SET_PARAMETER) != 0)
	System.out.println
		(">-< Conductor.Config_Value: " + name + " = " + value);
try {return configuration.Set (Config_Pathname (name), value);}
catch (Configuration_Exception exception)
	{
	throw new Configuration_Exception (Error_Message
		("Unable to set configuration parameter: " + name + " = " + value + NL
		+ exception.getMessage ()));
	}
}

/**	Get an absolute Conductor Configuration pathname.
<p>
	If the specified name is not an absolute pathname the {@link
	#CONDUCTOR_GROUP} is used as the root for to make an {@link
	Configuration#Absolute_Pathname(String, String) absolute pathname}
	from the name.
<p>
	@param	name	The parameter name to be made absolute.
	@return	An absolute Configuration pathname String.
*/
public static String Config_Pathname
	(
	String			name
	)
{return Configuration.Absolute_Pathname (CONDUCTOR_GROUP, name);}


//	Management implementation.
/**	Get the Conductor Configuration.
<p>
	<b>N.B.</b>: All "password" (case insensitive) parameters will have
	their values masked out.
<p>
	@return	A Configuration containing a copy of the current Conductor
		Configuration.
*/
public Configuration Configuration ()
{
Configuration
	configuration = null;
try
	{
	configuration = new Configuration (The_Configuration);

	//	Mask all password values.
	configuration.Case_Sensitive (false);
	configuration.Set_All ("PASSWORD", Processing_Changes.MASKED_PASSWORD);
	}
catch (Configuration_Exception exception)
	{
	//	This shouldn't be possible: the configuration has been vetted.
	configuration = null;
	}
return configuration;
}

/**	Gets the process ID for this application.
<p>
	A native method is to be used to obtain the PID that uniquely
	identifies the current process on the host system.
<p>
	@return	A PID integer. This will be zero if a PID can not
		be obtained because the native method is not present.
*/
private int PID ()
{return Native_Methods.PID ();}
	
/*==============================================================================
	Database
*/
/**	Establish the Database connection.
<p>
	The connection will be retried up to the number of times specified
	by the RECONNECT_TRIES_PARAMETER with the number of seconds delay
	between retries specified by the RECONNECT_DELAY_PARAMETER.
	Connection retries stop when the connection to the database is
	successful, the number of retries is exhausted, or an exception
	occurs that is not associated with a connection failure.
<p>
	@return	The last Database_Exception that occured. This will be null
		if the connection was successful.
*/
protected Database_Exception Connect_to_Database ()
{
if ((DEBUG & DEBUG_DATABASE_CONNECTION) != 0)
	System.out.println
		(">>> Conductor.Connect_to_Databse");
int
	retry = 0;
while (true)
	{
	try {The_Database.Disconnect ();}
	catch (Database_Exception exception) {}
	try
		{
		if ((DEBUG & DEBUG_DATABASE_CONNECTION) != 0)
			System.out.println
				("    Connecting to Database Server " + Database_Server_Name);
		The_Database.Connect (Database_Server_Name);
		if ((DEBUG & DEBUG_DATABASE_CONNECTION) != 0)
			System.out.println ("<<< Connect_to_Databse: Connected");
		return null;
		}
	catch (Database_Exception exception)
		{
		if (Processing_State == RUN_TO_WAIT ||
			! exception.Disconnected () ||
			retry >= Reconnect_Tries)
			return Database_Error
				("Unable to connect to the database." + NL
				+ retry + " retr" + ((retry == 1) ? "y" : "ies")
					+ " attempted with "
					+ Reconnect_Delay + " second delay." + NL
				+ exception.getMessage ());
		}
	catch (Configuration_Exception exception)
		{
		return Database_Error
			("Unable to connect to the database." + NL
			+ exception.getMessage ());
		}
	if (Reconnect_Delay > 0)
		{
		if ((DEBUG & DEBUG_DATABASE_CONNECTION) != 0)
			System.out.println
				("    Connect_to_Databse: Sleeping "
					+ Reconnect_Delay + " seconds");
		Sleep (Reconnect_Delay);
		}
	++retry;
	if ((DEBUG & DEBUG_DATABASE_CONNECTION) != 0)
		System.out.println ("    Connect_to_Databse: Retry " + retry);
	}
}

/**	Resolve a reference
<p>
	If the reference can not be resolved because the Database has
	become disconnected the Database is {@link #Connect_to_Database()
	connected} and the reference resolution is tried again.
<p>
	@param	reference	The reference String to be {@link
		Reference_Resolver#Resolve(String) resolved}.
	@return	The resolved reference String.
	@throws	Database_Exception	If there was a problem accessing the
		Database.
	@throws	ParseException	If the reference contained a mathematical
		expression that could not be correctly parsed.
	@throws	Unresolved_Reference	If the reference could not be resolved
		and a non-null {@link #Resolver_Default_Value(String) default
		value} had not been assigned.
	@see	Reference_Resolver
*/
protected String Resolve
	(
	String	reference
	)
	throws Database_Exception, ParseException, Unresolved_Reference
{
try {return Resolver.Resolve (reference);}
catch (Database_Exception exception)
	{
	if (exception.Disconnected ())
		//	Try to reconnect.
		Connect_to_Database ();
	else
		throw exception;
	return Resolver.Resolve (reference);
	}
}

//	Management implementation.
/**	Set the Conductor default Reference_Resolver value.
<p>
	Normally when the Conductor Reference_Resolver is unable to resolve a
	reference it will throw an {@link #Processing_Exception() exception}
	and enter the halted {@link #Processing_State() processing state}, or
	exit if it is not {@link #Connected_to_Stage_Manager() connected to a
	Stage_Manager} at the time. If, however, the {@link
	Reference_Resolver#Default_Value(String) Reference_Resolver default
	value} is set to a non-null String that value will be used for the
	unresolved_reference instead of throwing an exception.
<p>
	If the specified value is different from the current Reference_Resolver
	default value the Configuration {@link #UNRESOLVED_REFERENCE_PARAMETER}
	is set to this value (or {@link #UNRESOLVED_REFERENCE_THROWS} if the
	value is null) and processing event notification with the updated
	Configuration is sent to all listeners.
<p>
	@param	value	The default Reference_Resolver String value. If this
		starts with {@link #UNRESOLVED_REFERENCE_THROWS} (case insensitive)
		null will be used.
	@return	This Conductor Management object.
	@see	Reference_Resolver
*/
public Management Resolver_Default_Value
	(
	String	value
	)
{
if (value.toUpperCase ().startsWith (UNRESOLVED_REFERENCE_THROWS))
	value = null;

String
	resolver_value = Resolver.Default_Value ();
if ((resolver_value == null &&
	          value != null) ||
	(resolver_value != null &&
	! resolver_value.equals (value)))
	{
	try
		{
		Config_Value (UNRESOLVED_REFERENCE_PARAMETER,
			(value == null) ? UNRESOLVED_REFERENCE_THROWS : value);
		Report_Processing_Event (new Processing_Changes ()
				.Configuration (The_Configuration));

		synchronized (Resolver) {Resolver.Default_Value (value);}
		}
	catch (Configuration_Exception exception)
		{/*	Don't set an unreportable value. */}
	}
return this;
}

//	Management implementation.
/**	Get the Conductor default Reference_Resolver value.
<p>
	@return	The default Reference_Resolver String value.
	@see	#Resolver_Default_Value(String)
*/
public String Resolver_Default_Value ()
{synchronized (Resolver) {return Resolver.Default_Value ();}}

/**	Copy a database table.
<p>
	@param	field_names	A Vector of field name Strings in the same
		order as the table record entries.
	@param	table	A Vector of records; each record is a Vector of
		field value Strings any of which may be null.
	@return	A Vector containing a copy of the field_names as its first
		entry followed by a copy of each record in the table.
*/
private static Vector<Vector<String>> Table
	(
	Vector<String>			field_names,
	Vector<Vector<String>>	table
	)
{
if (field_names == null ||
	table == null)
	return null;
int
	row = 0,
	rows = table.size ();
Vector<Vector<String>>
	copy = new Vector<Vector<String>> (rows + 1);
copy.add (Record (field_names));
while (row < rows)
	copy.add (Record (table.get (row++)));
return copy;
}

/**	Copy a database record.
<p>
	@param	record	A Vector of field value Strings any of which may be null.
	@return	A Vector containing a copy of the record.
*/
private static Vector<String> Record
	(
	//	Guaranteed to be non-null.
	Vector<String>	record
	)
{
int
	column = 0,
	columns = record.size ();
Vector<String>
	copy = new Vector<String> (columns);
while (column < columns)
	copy.add (record.get (column++));
return copy;
}

/*------------------------------------------------------------------------------
	Procedures
*/
//	Management implementation.
/**	Get the procedures table.
<p>
	The table is expected to be delivered with the record fields in
	{@link Conductor#PROCEDURES_FIELD_NAMES} order, and with the records in
	processing order.
<p>
	<b>N.B.</b>: The entire contents of the database procedures table is
	delivered.
<p>
	@return	A table Vector of record Vectors of field Strings. The first
		record contains the field names.
*/
public Vector<Vector<String>> Procedures ()
{
if (Procedure_Records == null)
	return null;
synchronized (Procedure_Records)
	{return Table
		(Procedures_Map.Actual_to_User (Procedures_Map.All_Actual_Fields ()),
		Procedure_Records);}
}

/**	Get the {@link #Procedures_Table} from the Database.
<p>
	The entire Procedures_Table is obtained from the Database.
<p>
	If a {@link Database_Exception#Disconnected () disconnected
	Database_Exception} is thrown an attempt is made to {@link
	#Connect_to_Database() reconnect} to the Database.
<p>
	@return	A Vector containing the procedures table. Each entry is a
		Vector of fields with the first entry being the field names for
		the records that follow.
	@throws	Database_Exception	If the the procedures table could not be
		obtained.
*/
protected Vector<Vector<String>> Get_Procedures_Table ()
	throws Database_Exception
{
if ((DEBUG & DEBUG_DATABASE) != 0)
	System.out.println
		(">>> Conductor.Get_Procedures_Table" + NL
		+"    Procedures_Table: " + Procedures_Table);
Vector<Vector<String>>
	table = null;
Database_Exception
	thrown = null;
try {table = The_Database.Select (Procedures_Table);}
catch (Database_Exception exception)
	{
	thrown = exception;
	if (thrown.Disconnected () &&
		(thrown = Connect_to_Database ()) == null)
		{
		try {table = The_Database.Select (Procedures_Table);}
		catch (Database_Exception except) {thrown = except;}
		}
	}
if (thrown != null)
	throw Database_Error
		("Unable to load the procedures table: " + Procedures_Table + NL
		+ thrown.getMessage ());
if ((DEBUG & DEBUG_DATABASE) != 0)
	{
	for (int
			index = 0;
			index < table.size ();
			index++)
		System.out.println (table.get (index));
	System.out.println ("<<< Get_Procedures_Table");
	}
return table;
}

/**	Load the Procedure_Records table.
<p>
	The entire Procedures table is cached. The first, field names,
	record is removed and used to construct a Field_Map
	of required field names/indexes to the available field names.
<p>
	The database table is first loaded into a temporary records list. A
	temporary Fields_Map is constructed from the first, field names,
	record which is removed from the table. The map is used to confirm
	that all the required fields are present. Then the fields of each
	record are mapped to their expected order. All the records are then
	sorted on the {@link #SEQUENCE_FIELD} using a {@link
	String_Vector_Comparator} to do the comparisons.
<p>
	If the tentative new table content is different than the current
	Procedure_Records table the latter is set to the former, and the
	Procedures_Map set to the new map, while synchronized to prevent the
	Management from obtaining it while it is in an inconsistent state.
	Then the Configuration is updated: the {@link
	#TOTAL_PROCEDURE_RECORDS_PARAMETER} is set to the number of table
	records (not including the field names record, which was removed to
	contrstuct the Fields_Map), the {@link #PROCEDURE_COUNT_PARAMETER} is
	reset to zero, the {@link #PROCEDURE_SEQUENCE_PARAMETER} is reset to
	the empty string, and the {@link
	#PROCEDURE_COMPLETION_NUMBER_PARAMETER} is reset to {@link
	#PROCEDURE_SUCCESS}. Finally a {@link
	#Report_Processing_Event(Processing_Changes) processing event report}
	is sent with {@link Processing_Changes#Procedures_Changed(boolean)
	procedures changed} set.
<p>
	<b>N.B.</b>: The Procedure_Records and the Procedures_Map are not
	changed, nor a processing event reported, unless a valid table with
	different contents from the current table is loaded.
<p>
	@throws	Database_Exception	If the {@link #Get_Procedures_Table()
		procedures table} could not be obtained from the Database,
		it was empty, or any required {@link #PROCEDURES_FIELD_NAMES}
		are missing.
	@throws Configuration_Exception	If the Configuration could not
		be updated with the total number of procedures records.
	@see	Fields_Map
*/
protected void Load_Procedure_Records ()
	throws Database_Exception, Configuration_Exception
{
if ((DEBUG & DEBUG_DATABASE) != 0)
	System.out.println
		(">>> Conductor.Load_Procedure_Records" + NL
		+"    Procedures_Table: " + Procedures_Table);

//	Get the procedures table.
Vector<Vector<String>>
	procedure_records = Get_Procedures_Table ();
if (procedure_records.size () < 2)
	//	No procedures.
	throw Database_Error
		("An empty " + Procedures_Table + " procedures table was loaded.");

//	Construct the map of field names/indexes to field numbers.
Fields_Map
	procedures_map = new Fields_Map
		(PROCEDURES_FIELD_NAMES, procedure_records.remove (0));

//	Confirm that all the fields (except DESCRIPTION_FIELD) are present.
String[]
	fields = (String[])PROCEDURES_FIELD_NAMES.clone ();
fields[DESCRIPTION_FIELD] = null;
try {procedures_map.Confirm_Fields (fields);}
catch (Database_Exception exception)
	{
	synchronized (Procedure_Records)
		{
		Procedure_Records = procedure_records;
		Procedures_Map = procedures_map;
		}
	throw Database_Error (exception.getMessage () + NL
		+ "in the " + Procedures_Table + " table.");
	}

//	Map all the records so their fields are in the expected order.
for (int
		index = 0;
		index < procedure_records.size ();
		index++)
	procedure_records.set (index,
		procedures_map.Actual_to_User
			(procedure_records.get (index)));

//	Sort the procedure records by sequence number.
Collections.sort (procedure_records,
	new String_Vector_Comparator (SEQUENCE_FIELD));

if (! Procedure_Records.equals (procedure_records))
	{
	//	Update to the new procedures.
	synchronized (Procedure_Records)
		{
		Procedure_Records = procedure_records;
		Procedures_Map = procedures_map;
		}

	//	Update the Configuration.
	Config_Value (TOTAL_PROCEDURE_RECORDS_PARAMETER,
		new Integer (Procedure_Records.size ()));
	Config_Value (PROCEDURE_COUNT_PARAMETER,
		new Integer (0));
	Config_Value (PROCEDURE_SEQUENCE_PARAMETER,
		"");
	Config_Value (PROCEDURE_COMPLETION_NUMBER_PARAMETER,
		new Integer (PROCEDURE_SUCCESS));

	//	Report the change.
	Report_Processing_Event (new Processing_Changes ()
		.Procedures_Changed (true));
	if ((DEBUG & DEBUG_DATABASE) != 0)
		{
		System.out.println
			("    " + (Procedure_Records.size ())
				+ " Procedure_Records -");
		for (int
				index = 0;
				index < Procedure_Records.size ();
				index++)
			System.out.println (Procedure_Records.get (index));
		}
	}
else
if ((DEBUG & DEBUG_DATABASE) != 0)
	System.out.println
		("    Procedure_Records unchanged");
if ((DEBUG & DEBUG_DATABASE) != 0)
	System.out.println
		("<<< Load_Procedure_Records");
}


private String Procedure_Field
	(
	int		field
	)
{return Procedure_Record.get (field);}


private void Procedure_Field
	(
	int		field,
	String	value
	)
{Procedure_Record.set (field, value);}

/*------------------------------------------------------------------------------
	Sources
*/
//	Management implementation.
/**	Get the current cache of source records.
<p>
	The table will be delivered with the record fields in
	{@link Conductor#SOURCES_FIELD_NAMES} order, and with the records
	sorted by increasing {@link Conductor#SEQUENCE_FIELD} order.
<p>
	<b>N.B.</b>: Only the contents of the Conductor source records cache
	is delivered. The contents of the database sources table may be
	much, much larger.
<p>
	@return	A table Vector of record Vectors of field Strings. The first
		record contains the field names.
*/
public Vector<Vector<String>> Sources ()
{
if (Source_Records == null)
	return null;
synchronized (Source_Records)
	{return Table
		(Sources_Map.Actual_to_User (Sources_Map.All_Actual_Fields ()),
		Source_Records);}
}

/**	Get the {@link #Sources_Table} from the Database.
<p>
	Only those Sources_Table records with a NULL {@link
	#CONDUCTOR_ID_PARAMETER} field value are selected. Up to
	Max_Source_Records are obtained.
<p>
	If a {@link Database_Exception#Disconnected () disconnected
	Database_Exception} is thrown an attempt is made to {@link
	#Connect_to_Database() reconnect} to the Database.
<p>
	@return	A Vector containing the selected sources table records. Each
		entry is a Vector of fields with the first entry being the field
		names for the records that follow.
	@throws	Database_Exception	If the the sources table records could
		not be obtained.
*/
private Vector<Vector<String>> Get_Sources_Table ()
	throws Database_Exception
{
if ((DEBUG & DEBUG_DATABASE) != 0)
	System.out.println
		(">>> Conductor.Get_Sources_Table");
Vector<Vector<String>>
	table = null;
Vector<String>
	table_name = new Vector<String> (1);
table_name.add (Sources_Table);
Database_Exception
	thrown = null;
try {table = The_Database.Select (table_name, null,
		SOURCES_FIELD_NAMES[CONDUCTOR_ID_FIELD] + " IS NULL",
		Max_Source_Records);}
catch (Database_Exception exception)
	{
	thrown = exception;
	if (thrown.Disconnected () &&
		(thrown = Connect_to_Database ()) == null)
		{
		try {table = The_Database.Select (table_name, null,
				SOURCES_FIELD_NAMES[CONDUCTOR_ID_FIELD] + " IS NULL",
				Max_Source_Records);}
		catch (Database_Exception except) {thrown = except;}
		}
	}
if (thrown != null)
	throw Database_Error
		("Unable to select from the sources table: " + Sources_Table + NL
		+ thrown.getMessage ());
if ((DEBUG & DEBUG_DATABASE) != 0)
	{
	for (int
			index = 0;
			index < table.size ();
			index++)
		System.out.println (table.get (index));
	System.out.println ("<<< Get_Sources_Table");
	}
return table;
}

/**	Load the Sources_Records table.
<p>
	If the source records cache is not empty nothing is done.
<p>
	Records in the Sources table for which the Processing_Host field is
	NULL are cached. The maximum size of the cache is determined by the
	{@link #Max_Source_Records} value, which can not be less than the
	{@link #Min_Source_Records} value.
<p>
	The first, field names, record from the Database is removed and
	used to construct the Sources_Map Field_Map of required field
	names/indexes to the available field names.
<p>
	@return	true if additional source records were loaded; false if
		no unprocessed source records are available.
	@throws	Database_Exception	If the {@link #Get_Sources_Table()
		sources table records} could not be obtained from the Database
		or any required {@link #SOURCES_FIELD_NAMES} are missing.
	@see	Fields_Map
*/
protected boolean Load_Source_Records ()
	throws Database_Exception
{
if ((DEBUG & DEBUG_DATABASE) != 0)
	System.out.println
		(">>> Conductor.Load_Source_Records");
if (Source_Records.isEmpty ())
	{
	synchronized (Source_Records)
		{
		Source_Records = Get_Sources_Table ();

		//	Construct the map of field names/indexes to field numbers.
		Sources_Map = new Fields_Map
			(SOURCES_FIELD_NAMES, Source_Records.remove (0));

		//	Confirm that all the fields are present.
		try {Sources_Map.Confirm_Fields (SOURCES_FIELD_NAMES);}
		catch (Database_Exception exception)
			{
			throw Database_Error (exception.getMessage () + NL
				+ "in the " + Sources_Table + " table.");
			}

		if (! Source_Records.isEmpty ())
			{
			//	Map all the records so their fields are in the expected order.
			for (int
					index = 0;
					index < Source_Records.size ();
					index++)
				Source_Records.set (index,
					Sources_Map.Actual_to_User
						(Source_Records.get (index)));
			}
		}

	//	Report the change.
	Report_Processing_Event (new Processing_Changes ()
		.Sources_Refreshed (true));
	}
if ((DEBUG & DEBUG_DATABASE) != 0)
	System.out.println ("<<< Load_Source_Records: "
		+ (! Source_Records.isEmpty ()));
return ! Source_Records.isEmpty ();
}

/**	Acquire an exclusive lock on a source record from the Sources table.
<p>
	While the Source_Records cache is not empty and no record has been
	acquired:
<p>
	The record at the front of the cache is removed and becomes the
	tentative source record. <b>N.B.</b>: This has the intended
	effect of consuming the cache.
<p>
	An attempt is made to mark the tentative source record in the
	Sources table as unavailable for processing. This must be done in
	the context of multiple processes contending for the same record at
	the same time.
<p>
	The key, literally, of this operation is the assumption that the
	SQL UPDATE operation will be atomic on the database server. Once
	the operation is started by the database server it will go to
	completion without the possibility of interruption by any other
	database operation. This guarantees that only one process will be
	able to gain access to any source record. It will be safe to
	process the source without concern that some other process may
	interfere.
<p>
	An UPDATE operation on the unique Sources table record having the
	target SOURCE_NUMBER_FIELD is done to set the CONDUCTOR_ID_FIELD
	field to the Conductor_ID, with the condition that its
	CONDUCTOR_ID_FIELD field is currently NULL. If the UPDATE returns
	0, then the field value was not set (it was not NULL) and so we did
	not acquire an exclusive lock on the record (some other process has
	claimed it). If it returns 1, then the field value was changed from
	NULL to our Conductor_ID and so we have acquired an exclusive lock
	on the record.
<p>
	When an exclusive lock has been acquired on a tentative source record
	it is set as the current Source_Record. Then the Configuration {@link
	#SOURCE_NUMBER_PARAMETER}, {@link #SOURCE_PATHNAME_PARAMETER}, {@link
	#SOURCE_ID_PARAMETER} and {@link #LOG_PATHNAME_PARAMETER} are set
	from the corresponding fields of the record; the other source record
	parameters - {@link #SOURCE_DIRECTORY_PARAMETER}, {@link
	#SOURCE_FILENAME_PARAMETER}, {@link #SOURCE_FILENAME_ROOT_PARAMETER},
	{@link #SOURCE_FILENAME_EXTENSION_PARAMETER}, {@link
	#LOG_DIRECTORY_PARAMETER}, and {@link #LOG_FILENAME_PARAMETER} - are
	reset to the empty string (they will be filled in during {@link
	#Process_Source() source processing}).
<p>
	@return	true if an exclusive lock on the record was acquired;
		false otherwise.
	@throws	Database_Exception	if there was a problem accessing
		the Database.
	@throws	Configuration_Exception	if there was a problem setting the
		Configuration parameters.
*/
private boolean Acquire_Source_Record ()
	throws
		Database_Exception,
		Configuration_Exception
{
if ((DEBUG & DEBUG_GET_SOURCE) != 0)
	System.out.println
		(">>> Conductor.Acquire_Source_Record: "
			+ Source_Records.size () + " unprocessed records.");
Vector<String>
	record = null;
boolean
	acquired = false;
while (! acquired)
	{
	if (Source_Records.isEmpty ())
		{
		if ((DEBUG & DEBUG_GET_SOURCE) != 0)
			System.out.println
				("    Source_Records cache is empty");
		break;
		}

	//	Pull the next record from the front of the list.
	record = Source_Records.remove (0);
	if ((DEBUG & DEBUG_GET_SOURCE) != 0)
		System.out.println
			("      Source_Number: "
			+ Sources_Map.entry (record, SOURCE_NUMBER_FIELD) + NL
			+"          Source_ID: "
			+ Sources_Map.entry (record, SOURCE_ID_FIELD) + NL
			+"    Source_Pathname: "
			+ Sources_Map.entry (record, SOURCE_PATHNAME_FIELD));

	/*	Assemble the database record acquisition operation.

		This MUST be an atomic database operation that acquires the
		record with an exclusive lock, vis a vis other Conductor process
		contending for the same record. 
	*/
	String
		SQL_update =
			"UPDATE " + Sources_Table
			+" SET "
				+ SOURCES_FIELD_NAMES[CONDUCTOR_ID_FIELD]
				+ "='" + Conductor_ID + "'"
			+" WHERE "
				+ SOURCES_FIELD_NAMES[SOURCE_NUMBER_FIELD]
				+ "=" + Sources_Map.entry (record, SOURCE_NUMBER_FIELD) +
			" AND "
				+ SOURCES_FIELD_NAMES[CONDUCTOR_ID_FIELD]
				+ " is NULL";
	if ((DEBUG & DEBUG_GET_SOURCE) != 0)
		System.out.println
			("    SQL Update -" + NL
			+ SQL_update);

	//	Send the SQL to the database server.
	Database_Exception
		thrown = null;
	try {acquired = (The_Database.Update (SQL_update) == 1);}
	catch (Database_Exception exception)
		{
		thrown = exception;
		if (thrown.Disconnected () &&
			(thrown = Connect_to_Database ()) == null)
			{
			try {acquired = (The_Database.Update (SQL_update) == 1);}
			catch (Database_Exception except) {thrown = except;}
			}
		}
	if (thrown != null)
		throw Database_Error
			("Unable to acquire a lock on source record "
				+ Sources_Map.entry (record, SOURCE_NUMBER_FIELD) + '.' + NL
			+ thrown.getMessage ());
	}
if (acquired)
	{
	Source_Record = record;
	Source_Field (CONDUCTOR_ID_FIELD, Conductor_ID);
	String
		value;
	if ((value = Source_Record.get (SOURCE_NUMBER_FIELD)) == null)
		value = "";
	Config_Value (SOURCE_NUMBER_PARAMETER, value);
	if ((value = Source_Record.get (SOURCE_ID_FIELD)) == null)
		value = "";
	Config_Value (SOURCE_ID_PARAMETER, value);
	if ((value = Source_Record.get (SOURCE_PATHNAME_FIELD)) == null)
		value = "";
	Config_Value (SOURCE_PATHNAME_PARAMETER, value);
	if ((value = Source_Record.get (LOG_PATHNAME_FIELD)) == null)
		value = "";
	Config_Value (LOG_PATHNAME_PARAMETER, value);
	Config_Value (SOURCE_DIRECTORY_PARAMETER, "");
	Config_Value (SOURCE_FILENAME_PARAMETER, "");
	Config_Value (SOURCE_FILENAME_ROOT_PARAMETER, "");
	Config_Value (SOURCE_FILENAME_EXTENSION_PARAMETER, "");
	Config_Value (LOG_DIRECTORY_PARAMETER, "");
	Config_Value (LOG_FILENAME_PARAMETER, "");
	}
if ((DEBUG & DEBUG_GET_SOURCE) != 0)
	System.out.println
		("<<< Acquire_Source_Record: " + acquired);
return acquired;
}


private String Source_Field
	(
	int		field
	)
{return Source_Record.get (field);}


private void Source_Field
	(
	int		field,
	String	value
	)
{Source_Record.set (field, value);}


private void Update_Source_Record
	(
	int		field,
	String	value
	)
	throws Database_Exception
{
if ((DEBUG & DEBUG_TABLE_UPDATE) != 0)
	System.out.println
		(">>> Conductor.Update_Source_Record: field " + field);
if (field < 0 || field >= SOURCES_FIELD_NAMES.length)
	throw Database_Error
		("Can't update Sources field number " + field + "." + NL
		+"No such field: Programming bug in the use of Update_Source_Record!",
		Source_Field (SOURCE_NUMBER_FIELD));
if ((DEBUG & DEBUG_TABLE_UPDATE) != 0)
	System.out.println
		("    " + SOURCES_FIELD_NAMES[field] + " = " + value);
//	Assemble the SQL statement.
String
	SQL_update =
		"UPDATE " + Sources_Table
		+ " SET "
			+ SOURCES_FIELD_NAMES[field] + "='" + value + "'"
		+ " WHERE "
			+ SOURCES_FIELD_NAMES[SOURCE_NUMBER_FIELD]
			+ "=" + Source_Field (SOURCE_NUMBER_FIELD);
if ((DEBUG & DEBUG_TABLE_UPDATE) != 0)
	System.out.println
		("    SQL Update -" + NL
		+ SQL_update);

//	Send the SQL to the database server.
Database_Exception
	thrown = null;
int
	count = 0;
try {count = The_Database.Update (SQL_update);}
catch (Database_Exception exception)
	{
	thrown = exception;
	if (thrown.Disconnected () &&
		(thrown = Connect_to_Database ()) == null)
		{
		try {count = The_Database.Update (SQL_update);}
		catch (Database_Exception except) {thrown = except;}
		}
	}
if (thrown != null)
	throw Database_Error
		("Update failed -" + NL
		+ thrown.getMessage ());
if (count == 0)
	throw Database_Error
		("Update failed -" + NL
		+ SQL_update);

//	Update the Source_Record.
Source_Field (field, value);

//	Report the change.
Report_Processing_Event (new Processing_Changes ()
	.Source_Record (Source_Record));

if ((DEBUG & DEBUG_TABLE_UPDATE) != 0)
	System.out.println
		("<<< Update_Source_Record");
}

/*==============================================================================
	Pipeline Processing
*/
//	Management implementation.
/**	Get the current Conductor processing state.
<p>
	The possible processing state values are:
<p>
<dl>
<dt>{@link Conductor#RUNNING}
<dd>Source records are being processing.

<dt>{@link Conductor#POLLING}
<dd>No unprocessed source records are currently available for processing;
	the Conductor is {@link #Poll_Interval(int) polling} for source
	records to process.

<dt>{@link Conductor#RUN_TO_WAIT}
<dd>When processing of the current source record completes Conductor will
	go into the waiting state.

<dt>{@link Conductor#WAITING}
<dd>The Conductor is waiting to be told to being processing.

<dt>{@link Conductor#HALTED}
<dd>A problem condition caused the Conductor to halt processing. The
	problem may be the result of the maximum number of {@link
	#Stop_on_Failure(int) sequential failures} of source record
	processing having occured, a database access failure, or some other
	{@link #Processing_Exception system error}.
</dl>
<p>
	The WAITING and HALTED state codes are negative; all others are positive.
<p>
	@return	A Conductor processing state code.
*/
public int Processing_State ()
{return Processing_State;}

//	Management implementation.
/**	Get the current Conductor processing conditions state.
<p>
	All Processing_Changes state variables will be set to the current
	values from this Conductor, except the flag variables -
	{@link Processing_Changes#Sources_Refreshed(boolean)}, 
	{@link Processing_Changes#Procedures_Changed(boolean)} and
	{@link Processing_Changes#Exiting(boolean)} - will always be false.
<p>
	@return A Processing_Changes object containing the values of the
		current Conductor processing state variables.
*/
public Processing_Changes State ()
{
if ((DEBUG & DEBUG_PROCESSING_EVENT) != 0)
	System.out.println
		(">>> Conductor.State");
Processing_Changes
	state = null;
try
	{
	state = new Processing_Changes ()
		.Configuration (The_Configuration)
		.Processing_State (Processing_State)
		.Source_Record (Source_Record)
		.Procedure_Record (Procedure_Record)
		.Sequential_Failures (Sequential_Failures)
		.Error_Condition ((Processing_Exception == null) ?
			null : Processing_Exception.toString ());
	}
catch (Configuration_Exception exception)
	{/* Shouldn't happen with a good configuration. */}
if ((DEBUG & DEBUG_PROCESSING_EVENT) != 0)
	System.out.println
		(state + NL
		+"<<< Conductor.State");
return state;
}

//	Management implementation.
/**	Start pipeline processing.
<p>
	If a pipeline processing thread is not running one is started.
	Otherwise, if the {@link #RUN_TO_WAIT} state is in effect it
	is reset to the {@link #RUNNING} state.
<p>
<h4>Pipeline Processing:
</h4><p>
	The Conductor reconfigures itself by re-reading its Configuration
	source in case it was changed while processing was waiting.
	<b>N.B.</b>: The {@link #Poll_Interval(int) poll interval} and
	{@link #Stop_on_Failure(int) stop-on-failure limit} are not
	reset if they were set to a non-negative value.
<p>
	The {@link #Load_Procedure_Records() table of procedures} is
	refreshed in case it was changed while processing was waiting.
<p>
	Source procesing will continue indefinately unless: the {@link
	#Poll_Interval() polling interval} for newly available sources is
	zero (i.e. batch mode) and at least Min_Source_Records have been
	processed in the current batch; processing of the current source has
	completed and further processing has been flagged to {@link #Stop()
	stop}; the number of sequential procedure failures has reached its
	(@link #Stop_on_Failure() limit}, or an unrecoverable exception
	occurred during processing, in which case the {@link
	#Processing_Exception() processing exception} will be non-null. The
	possible processing exception types are a Configuration_Exception,
	Database_Exception or IOException; any other exception is due to a
	programming error (any exception is caught and saved for possible
	subsequent retrieval);
<p>
	The current contents of the {@link #Load_Source_Records() source
	records cache} is processed first. An unprocessed source record is
	{@link #Acquire_Source_Record() acquired} from the cache and {@link
	#Process_Source() processed}. Source record acquisition is always
	done in the order in which source records occur in the list of
	records obtained from the database; there is no priority ordering.
	Only after the cache is emtpy and the polling interval {@link
	#Sleep(long) sleep time} has passed will it be refreshed. If there is
	no polling interval the cache will not be refreshed and processing
	will stop. The polling interval may be interrupted by stopping and
	{@link #Start() restarting} processing.
*/
public void Start ()
{
if ((DEBUG & DEBUG_MANAGEMENT) != 0)
	System.out.println
		(">>> Conductor.Start" + NL
		+"    Processor == null? " + (Processor == null));
if (Processor == null)
	{
	if ((DEBUG & DEBUG_MANAGEMENT) != 0)
		System.out.println ("    The Processor will be started");
	Processor = new Thread (new Pipeline_Processor ());
	Processor.start ();
	}
else
if (Processing_State == RUN_TO_WAIT)
	{
	if ((DEBUG & DEBUG_MANAGEMENT) != 0)
		System.out.println
			("    Processing_State changing from RUN_TO_WAIT to RUNNING");
	Report_Processing_Event (new Processing_Changes ()
		.Processing_State (Processing_State = RUNNING));
	}
if ((DEBUG & DEBUG_MANAGEMENT) != 0)
	System.out.println ("<<< Start");
}


private class Pipeline_Processor
	implements Runnable
{
public void run ()
{
Process_Pipeline ();
Processor = null;
}
}	//	End of Pipeline_Processor class.


/**	Pipeline processing.
<p>
	@see	#Start()
*/
private void Process_Pipeline ()
{
if ((DEBUG & DEBUG_PIPELINE) != 0)
	System.out.println
		(">>> Conductor.Process_Pipeline: " + Pipeline);
Processing_Exception = null;
Processing_Changes
	processing_changes = null;
Report_Processing_Event (new Processing_Changes ()
	.Processing_State (Processing_State = RUNNING));

try
	{
	Reconfigure ();
	Load_Procedure_Records ();
	Load_Source_Records ();
	Batch_Sources_Count = 0;

	while (Processing_State > RUN_TO_WAIT)
		{
		if (Acquire_Source_Record ())
			{
			if ((DEBUG & DEBUG_PIPELINE) != 0)
				System.out.println
					("    Acquire_Source_Record true");
			if (Processing_State == POLLING)
				Report_Processing_Event (new Processing_Changes ()
					.Processing_State (Processing_State = RUNNING));
			Process_Source ();
			}
		else
			{
			if ((DEBUG & DEBUG_PIPELINE) != 0)
				System.out.println
					("    Acquire_Source_Record false" + NL
					+"    Poll_Interval = " + Poll_Interval + NL
					+"    Source_Success_Count = " + Source_Success_Count + NL
					+"     Total_Failure_Count = " + Total_Failure_Count);
			int
				poll_interval = Poll_Interval;
			if (poll_interval <= 0)
				{
				//	Batch mode.
				if ((DEBUG & DEBUG_PIPELINE) != 0)
					System.out.println
						("    Batch mode -" + NL
						+"    Poll_Interval = " + Poll_Interval + NL
						+"    BATCH_POLL_INTERVAL = " + BATCH_POLL_INTERVAL + NL
						+"    Batch_Sources_Count = " + Batch_Sources_Count + NL
						+"    Min_Source_Records  = " + Min_Source_Records);
				poll_interval = BATCH_POLL_INTERVAL;
				if (Batch_Sources_Count >= Min_Source_Records)
					{
					//	Batch mode completed.
					if ((DEBUG & DEBUG_PIPELINE) != 0)
						System.out.println
							("    Batch mode completed");
					break;
					}
				}

			if (! Load_Source_Records ())
				{
				if ((DEBUG & DEBUG_PIPELINE) != 0)
					System.out.println
						("    Load_Source_Records false" + NL
						+"    Poll_Interval = " + Poll_Interval + NL
						+"    Processing_State = " + Processing_State);
				if (Poll_Interval > 0 &&
					Processing_State > RUN_TO_WAIT)
					{
					if (Processing_State != POLLING)
						Report_Processing_Event (new Processing_Changes ()
							.Processing_State (Processing_State = POLLING));
					if ((DEBUG & DEBUG_PIPELINE) != 0)
						System.out.println
							("    Sleep for " + Poll_Interval + "seconds");
					Sleep (poll_interval);
					}
				else
					break;
				}
			}
		}
	}
catch (Exception exception)
	{
	if ((DEBUG & DEBUG_PIPELINE) != 0)
		System.out.println
			("==> Process_Pipeline: Exception - " + NL
			+ exception);
	Processing_State = HALTED;
	Processing_Exception = exception;
	try {processing_changes = new Processing_Changes ()
			.Configuration (The_Configuration);}
	catch (Configuration_Exception except) {}
	processing_changes
		.Sequential_Failures (Sequential_Failures)
		.Error_Condition (Processing_Exception.toString ());
	}

if (Processing_State != HALTED)
	Processing_State = WAITING;
if ((DEBUG & DEBUG_PIPELINE) != 0)
	System.out.println
		("==> Process_Pipeline: State " + Processing_State);

if (processing_changes == null)
	processing_changes = new Processing_Changes ();
Report_Processing_Event (processing_changes
	.Processing_State (Processing_State));

if (Processing_State == HALTED)
	{
	if ((DEBUG & DEBUG_PIPELINE) != 0)
		System.out.println
			("    Processing HALTED");
	Send_Failure_Notification (new Status_Report
			(this, null, Processing_Exception));
	}

if ((DEBUG & DEBUG_PIPELINE) != 0)
	System.out.println
		("     Main_Thread_Waiting != null? " 
			+ (Main_Thread_Waiting != null) + NL
		+"      Connected_to_Stage_Manager? "
			+ Connected_to_Stage_Manager () + NL
		+"                  Poll_Interval = "
			+ Poll_Interval + NL
		+"    Theater_Management.Auto_Open? "
			+ Theater_Management.Auto_Open ());
if (Main_Thread_Waiting != null &&		//	No local Management.
	! Connected_to_Stage_Manager () &&	//	No remote Management.
	(Poll_Interval <= 0 ||				//	Batch mode.
	! Theater_Management.Auto_Open ()))	//	Not expecting remote Management.
	{
	if ((DEBUG & DEBUG_PIPELINE) != 0)
		System.out.println
			("    Stopping main thread");
	Main_Thread_Waiting.interrupt ();	//	Stop the main thread.
	}

Processor = null;
}

/**	Puts the Conductor processing to sleep for some time.
<p>
	A lock is acquired on the Conductor object and it is put in a wait
	state for the specified amount of time. This blocks all further
	processing (and should not consume CPU resources) until the sleep
	period has expired, at which time the method will return.
<p>
	<b>Note</b>: The process will awake earlier than the sleep time if
	the wait is externally interrupted.
<p>
	@param	seconds	The number of seconds to sleep.
*/
private void Sleep
	(
	long	milliseconds
	)
{
if ((DEBUG & DEBUG_PIPELINE) != 0)
	System.out.println
	(">-< Conductor.Sleeping for "
		+ ((double)milliseconds / 1000.0) + " seconds ...");
if (milliseconds > 0)
	{
	try
		{
		Sleeping = true;
		Processor.sleep (milliseconds);
		}
	catch (InterruptedException exception)
		{
		if ((DEBUG & DEBUG_PIPELINE) != 0)
			System.out.println
				("Woke early!");
		}
	}
Sleeping = false;
}

private void Sleep
	(
	int		seconds
	)
{Sleep ((long)(seconds * 1000));}

//	Management implementation.
/**	Set the time interval to poll for unprocessed source records.
<p>
	The new value will override the value set when the Conductor is
	{@link #Postconfigure(Configuration) reconfigured} after being
	stopped. However, if the value is negative then the value from the
	reconfiguration will be used. In this case a zero value will still be
	set for the current poll interval; this will cause the Conductor to
	stop if it is currently polling or when no unprocessed source records
	can be obtained from the database.
<p>
	@param	seconds	The number of seconds between querying the database
		for unprocessed source records. If negative zero will be used.
	@return	This Management interface.
*/
public Management Poll_Interval
	(
	int		seconds
	)
{
if (seconds < 0)
	{
	seconds = 0;
	Poll_Interval_Override = UNSET_INTEGER_VALUE;
	}
else
	Poll_Interval_Override = seconds;

if (seconds != Poll_Interval)
	{
	try
		{
		Config_Value (POLL_INTERVAL_PARAMETER,
			new Integer (Poll_Interval = seconds));
		Report_Processing_Event (new Processing_Changes ()
			.Configuration (The_Configuration));
		}
	catch (Configuration_Exception exception) {/* Previously set */}

	if (Poll_Interval <= 0 &&
		Processing_State == POLLING)
		Stop ();
	}
return this;
}

/**	Get the interval at which the Conductor will poll for unprocessed
	source records.
<p>
	@return	The interval, in seconds, at which the Conductor will poll
		for unprocessed source records. If zero, polling has been
		disabled.
	@see	#Poll_Interval(int)
*/
public int Poll_Interval ()
{return Poll_Interval;}

//	Management implementation.
/**	Set the sequential failure limit at which to halt processing source
	records.
<p>
	The new value will override the value set when the Conductor is
	{@link #Postconfigure(Configuration) reconfigured} after being
	stopped. However, if the value is negative then the value from the
	reconfiguration will be used. In this case the current value will not
	be changed.
<p>
	@param	failure_count	The number of sequential failures at which to
		halt processing. Zero means never halt. If negative the current
		value is not changed.
	@return	This Management interface.
*/
public Management Stop_on_Failure
	(
	int		failure_count
	)
{
if (failure_count < 0)
	Stop_on_Failure_Override = UNSET_INTEGER_VALUE;
else
	{
	Stop_on_Failure_Override = failure_count;
	if (failure_count != Stop_on_Failure)
		{
		try
			{
			Config_Value (STOP_ON_FAILURE_PARAMETER,
				new Integer (Stop_on_Failure = failure_count));
			Report_Processing_Event (new Processing_Changes ()
				.Configuration (The_Configuration));
			}
		catch (Configuration_Exception exception) {/* Previously set */}
		}
	}
return this;
}

//	Management implementation.
/**	Get the number of Conductor sequential source processing failures
	at which to stop processing.
<p>
	@return	The number of Conductor sequential source processing failures
		at which to stop processing. If zero sequential processing
		failures will never cause processing to stop.
	@see	#Stop_on_Failure(int)
*/
public int Stop_on_Failure ()
{return Stop_on_Failure;}

//	Management implementation.
/**	Get the count of sequential source processing failures that the
	Conductor has accumulated.
<p>
	<b>N.B.</b>: A processing event notification is sent to all listeners
	
<p>
	@return	The count of sequential source processing failures that the
		Conductor has accumulated.
	@see	#Stop_on_Failure(int)
	@see	#Reset_Sequential_Failures()
*/
public int Sequential_Failures ()
{return Sequential_Failures;}

//	Management implementation.
/**	Reset the count of sequential source processing failures that the
	Conductor has accumulated.
<p>
	If the {@link #Processing_State() processing state} is {@link #HALTED}
	it is reset to {@link #WAITING}.
<p>
	If the count of sequential source processing failures and/or the
	processing state was changed a processing event notification with
	these changes is sent to all listeners.
<p>
	@return	This Conductor Management interface..
*/
public Management Reset_Sequential_Failures ()
{
Processing_Changes
	changes = null;

if (Sequential_Failures != 0)
	changes = new Processing_Changes ()
		.Sequential_Failures (Sequential_Failures = 0);

if (Processing_State == HALTED)
	{
	if (changes == null)
		changes = new Processing_Changes ();
	changes
		.Processing_State (Processing_State = WAITING);
	}

Report_Processing_Event (changes);
return this;
}

//	Management implementation.
/**	Get the exception that caused Conductor processing to halt.
<p>
	<b>N.B.</b>: When Conductor processing is {@link #Start() started}
	the previous processing exception is cleared.
<p>
	@return	The Exception that caused processing to halt. This will be
		null if processing did not halt as the result of an exception,
		or the current processing state is not halted.
*/
public Exception Processing_Exception ()
{return Processing_Exception;}

//	Management implementation.
/**	Stop pipeline processing after the current source processing has
	completed.
<p>
	If the Conductor is in a positive {@link #Processing_State()
	processing state} it will enter the {@link Conductor#RUN_TO_WAIT
	run-to-wait} state in which will enter the {@link Conductor#WAITING
	waiting} state when the current source record completes processing.
	If the Conductor is in the {@link Conductor#POLLING polling} state
	it will immediately stop polling for new source records. There will
	be no effect for any negative state.
*/
public void Stop ()
{
if ((DEBUG & DEBUG_MANAGEMENT) != 0)
	System.out.println
		(">>> Conductor.Stop");
Thread
	processor = Processor;
if (processor != null)
	{
	if ((DEBUG & DEBUG_MANAGEMENT) != 0)
		System.out.println
			("    The Processor is running; RUN_TO_WAIT state set");
	if (Processing_State > RUN_TO_WAIT)
		{
		if ((DEBUG & DEBUG_MANAGEMENT) != 0)
			System.out.println
				("    Setting RUN_TO_WAIT state");
		Report_Processing_Event (new Processing_Changes ()
			.Processing_State (Processing_State = RUN_TO_WAIT));
		}

	if (Sleeping)
		{
		if ((DEBUG & DEBUG_MANAGEMENT) != 0)
			System.out.println ("    Interrupting the sleeping Processor");
		synchronized (Logger)
			{
			try {processor.interrupt ();}
			catch (SecurityException exception) {}
			}
		}
	}
if ((DEBUG & DEBUG_MANAGEMENT) != 0)
	System.out.println ("<<< Stop");
}

//	Management implementation.
/**	Immediately stop processing and exit.
<p>
	Any open log file is closed. The database server is disconnected. An
	{@link Processing_Changes#Exiting(boolean) exiting} processing event
	is sent to all {@link #Add_Processing_Listener(Processing_Listener)
	processing listeners}. The application exits with a {@link
	#EXIT_SUCCESS success status}.
<p>
	<b>N.B.</b>: If source processing is {@link #Processing_State() running}
	it is aborted.
*/
public void Quit ()
{
if ((DEBUG & DEBUG_EXIT) != 0)
	System.out.println
		(">-< Conductor.Quit");
Exit (new Status_Report (this));
}

/*------------------------------------------------------------------------------
	Processing Events
*/
//	Management implementation.
/**	Register a processing state change listener.
<p>
	The Conductor sends its {@link Processing_Event processing event}
	notifications to all registered listeners.
<p>
	@param	listener	A Processing_Listener.
	@return	This Conductor Management object.
	@see	Processing_Listener
*/
public Management Add_Processing_Listener
	(
	Processing_Listener	listener
	)
{
if ((DEBUG & DEBUG_PROCESSING_EVENT) != 0)
	System.out.println
		(">>> Conductor.Add_Processing_Listener -" + NL
		+ listener);
boolean
	report = false;
synchronized (Processing_Listeners)
	{
	if (listener != null &&
		! Processing_Listeners.contains (listener))
		report = Processing_Listeners.add (listener);
	}
if (report)
	{
	//	Report all processing states to the new listener.
	listener.Processing_Event_Occurred (new Processing_Event
		(this, State ()));
/*
	if ((DEBUG & DEBUG_PROCESSING_EVENT) != 0)
		{
		synchronized (this)
		{
		try {wait (5000);}
		catch (InterruptedException exception) {}
		}
		}
*/
	}
if ((DEBUG & DEBUG_PROCESSING_EVENT) != 0)
	System.out.println
		("<<< Add_Processing_Listener");
return this;
}

//	Management implementation.
/**	Unregister a processing state change listener.
<p>
	@param	listener	The Processing_Listener to be removed from the
		Management list of registered listeners.
	@return	true	If the listener was registered and is now removed;
		false if it was not registered.
	@see	#Add_Processing_Listener(Processing_Listener)
*/
public boolean Remove_Processing_Listener
	(
	Processing_Listener	listener
	)
{
synchronized (Processing_Listeners)
	{return Processing_Listeners.remove (listener);}
}


private void Report_Processing_Event
	(
	Processing_Changes	changes
	)
{
if (changes == null)
	return;
if ((DEBUG & DEBUG_PROCESSING_EVENT) != 0)
	System.out.println
		(">>> Conductor.Report_Processing_Event");

synchronized (Processing_Listeners)
	{
	int
		index = Processing_Listeners.size ();
	if (index != 0)
		{
		Processing_Event
			event = new Processing_Event (this, changes);
		while (--index >= 0)
			{
			try {Processing_Listeners.get (index)
					.Processing_Event_Occurred (event);}
			catch (Exception exception)
				{/* Ignore all exceptions. */}
			}
		}
	}
if ((DEBUG & DEBUG_PROCESSING_EVENT) != 0)
	{
	/*	synchronized (this)
		{
		try {wait (5000);}
		catch (InterruptedException exception) {}
		}
	*/
	System.out.println
		("<<< Report_Processing_Event");
	}
}

/*------------------------------------------------------------------------------
	Source Record Processing
*/
/**	Process the current Source_Record.
*/
private void Process_Source ()
	throws
		Configuration_Exception,
		Database_Exception,
		IOException
{
if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
	System.out.println
		(">>> Conductor.Process_Source");
++Batch_Sources_Count;
String
	error_report = null;

Source_Status = Status_Indicators (Source_Field (STATUS_FIELD));
Config_Value (PROCEDURE_COUNT_PARAMETER,
	new Integer (Source_Status.size ()));
Procedure_has_Status = false;

/*
	Ensure that the File is constructed with an absolute pathname.
	This is necessary to be able to obtain the File's directory
	path (getParent) which will be null if the pathname String
	used to construct the File is relative.
*/
File
	source_file = new File (new File
		(Source_Field (SOURCE_PATHNAME_FIELD)).getAbsolutePath ());
String
	source_number = Source_Field (SOURCE_NUMBER_FIELD),
	source_id = Source_Field (SOURCE_ID_FIELD),
	source_log_pathname = Source_Field (LOG_PATHNAME_FIELD),
	log_pathname = source_log_pathname;

//	Source ID.
if (source_id == null ||
	source_id.length () == 0)
	{
	//	Use the filename portion of the source pathname.
	source_id = source_file.getName ();
	int
		index = source_id.lastIndexOf ('.');
	if (index >= 0)
		//	Remove the extension.
		source_id = source_id.substring (0, index);
	//	Add the source ID to the source record.
	Update_Source_Record (SOURCE_ID_FIELD, source_id);
	}
else
	Report_Processing_Event (new Processing_Changes ()
		.Source_Record (Source_Record));

if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
	System.out.println
		("      Source_Number: " + source_number + NL
		+"          Source_ID: " + source_id + NL
		+"    Source_Pathname: " + source_file.getAbsolutePath ());

//	Ensure that the previous log file is closed.
Close_Log_File ();

/*..............................................................................
	Open a log file.

	If the LOG_PATHNAME_FIELD is empty (or null) the
	LOG_DIRECTORY_PARAMETER (or LOG_PATHNAME_PARAMETER) from
	Log_File_Directory will be used. If both are empty the default
	filename will be used with no directory (i.e. relative to the CWD).
	If either are not empty the resulting pathname string is reference
	resolved. If the resolved pathname string refers to an existing
	directory the FILE_SEPARATOR and default filename will be appended to
	the pathname.

	The default log filename is:

	[<Log_File_Directory>/]<Pipeline>-<Source_ID>_<Source_Number>.log
	
	The Pipeline name includes the leading database catalog name
	separated by a period ('.') character.

	The Source_ID and the Source_Number are obtained from the current
	source record. Note, however, that there is a chance that the
	Source_ID will include characters that are unsafe for use as part of
	a file name. For now, we'll assume that the only unsafe character is
	the system's FILE_SEPARATOR character ('/' for Unix) which will be
	replaced with the FILE_SEPARATOR_SUBSTITUTE (probably '%') character.
	Another option would be to URL-encode the Source_ID, but this results
	in an ugly filename.

	Note: If either the LOG_PATHNAME_FIELD or the Log_File_Directory
	value is not empty and does not refer to a directory logging output
	will be appended to that file (it will be created if it does not
	exist).
*/
boolean
	append = false;

if (log_pathname == null ||
	log_pathname.length () == 0)
	//	Use the Log_File_Directory by default (may actually be a file pathname).
	log_pathname = Log_File_Directory;

if (log_pathname == null ||
	log_pathname.length () == 0)
	//	Use the derived filename in the CWD when no default Log_File_Directory.
	log_pathname =
		Pipeline + '-' +
		source_id.replace (FILE_SEPARATOR, FILE_SEPARATOR_SUBSTITUTE) + '_' +
		source_number + ".log";
else
	{
	//	Resolve the pathname, whether from source record or Log_File_Directory.
	try {log_pathname = Resolve (log_pathname);}
	catch (Exception exception)
		{
		error_report =
			"Unable to resolve reference in parameter \""
			+ LOG_DIRECTORY_PARAMETER + "\"" + NL
			+ exception.getMessage ();
		Config_Value (TOTAL_FAILURE_COUNT,
			new Long (++Total_Failure_Count));
		++Sequential_Failures;
		throw Database_Error (error_report, source_number);
		}
	if (new File (log_pathname).isDirectory ())
		log_pathname +=
			FILE_SEPARATOR +
			Pipeline + '-' +
			source_id.replace (FILE_SEPARATOR, FILE_SEPARATOR_SUBSTITUTE) + '_' +
			source_number + ".log";
	else
		//	Pathname is to file; append to it (if it exists).
		append = true;
	}
if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
	System.out.println
		("  Log file pathname: " + log_pathname
		+ (append ? "(append)" : ""));
File
	log_file = new File (new File (log_pathname).getAbsolutePath ());

//	Open the new log file.
try {Log_File_Writer = new FileWriter (log_file, append);}
catch (IOException exception)
	{
	error_report = Error_Message
		("Unable to open log file: " + log_file.getAbsolutePath ()) + NL
		+ exception.getMessage ();
	Config_Value (TOTAL_FAILURE_COUNT,
		new Long (++Total_Failure_Count));
	++Sequential_Failures;
	throw new IOException (error_report);
	}

//	Add the log file writer to the Logger.
Add_Log_Writer (Log_File_Writer);

if (! log_pathname.equals (source_log_pathname))
	//	Update the log file pathname in the source record.
	Update_Source_Record (LOG_PATHNAME_FIELD, log_file.getAbsolutePath ());

//	Write the source identification into the log file.
Log_Message (ID + NL);
Log_Message (SOURCE_FILE_LOG_DELIMITER, MARKER_STYLE);
Log_Message (new Date () + NL + NL);
Log_Message
	("Configuration: " + Config_Value (CONFIGURATION_SOURCE_PARAMETER) + NL
	+"Database: " + Config_Value (DATABASE_TYPE_PARAMETER)
		+" server on host "
		+ Config_Value (DATABASE_HOSTNAME_PARAMETER) + NL
	+"Source: " + Sources_Table + NL
	+ SOURCES_FIELD_NAMES[SOURCE_NUMBER_FIELD] + ": "
		+ source_number + NL
	+ SOURCES_FIELD_NAMES[SOURCE_ID_FIELD] + ": "
		+ source_id + NL
	+ SOURCES_FIELD_NAMES[SOURCE_PATHNAME_FIELD] + ": "
		+ source_file.getAbsolutePath () + NL + NL,
	HIGHLIGHT_STYLE);

/*..............................................................................
	Status indicators check.

	The Source_Status Vector is paired with the Procedure_Records Vector.
	Each procedure that is processed gets a status indicator in the
	Source_Status Vector. The Procedure_Records are processed in the
	order they have been sorted.
*/
if (! Source_Status.isEmpty ())
	{
	// Check the last procedure for successful completion.
	if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
		System.out.println
			("    Initial status: " + Source_Status);
	Log_Message
		("The source appears to have been previously processed" + NL
		+"producing status "
			+ Status_Field_Value (Source_Status) + NL + NL,
		NOTICE_STYLE);
	int
		conductor_code = UNSET_INTEGER_VALUE;
	try {conductor_code = Status_Conductor_Code
			(Source_Status.lastElement ());}
	catch (NumberFormatException exception) {}
	if (conductor_code != PROCEDURE_SUCCESS)
		{
		error_report =
			"Previous failure condition: "
				+ Status_Conductor_Code_Description (conductor_code) + '.' + NL
			+"Processing skipped for this source.";
		Log_Message (error_report + NL + NL, NOTICE_STYLE);
		Config_Value (TOTAL_FAILURE_COUNT,
			new Long (++Total_Failure_Count));
		Report_Processing_Event (new Processing_Changes ()
			.Configuration (The_Configuration)
			.Sequential_Failures (++Sequential_Failures));
		if (Stop_on_Failure != 0 &&
			Sequential_Failures >= Stop_on_Failure)
			Processing_State = HALTED;
		Close_Log_File ();
		if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
			System.out.println
				("<<< Process_Source (Prior failure condition)");
		return;
		}
	}

if (Source_Available_Tries == SOURCE_AVAILABLE_NO_CHECK)
	Log_Message
		("Source file availability check disabled." + NL + NL, HIGHLIGHT_STYLE);
else
	{
	/*..........................................................................
		Confirm access to a readable source file.
	*/
	/*
		Due to NFS filesystems latency it is possible that a source file
		that has just been registered will not appear if accessed too soon.
		So this hack provides a ten second delay on failure between three
		tries to access the file.
	*/
	for (int
			tries = Source_Available_Tries;
			tries > 0 &&
				! source_file.canRead ();
			tries--)
		{
		if (Source_Available_Tries > 1 &&
			Source_Available_Tries == tries)
			Log_Message
				("Can't access file " + source_file.getAbsolutePath () + NL
				+"Will retry up to " + Source_Available_Tries + " times ..." + NL
				+ NL,
				NOTICE_STYLE);
		Sleep (10);
		if (Processing_State == RUN_TO_WAIT)
			{
			Log_Message
				("Processing has been interrupted.",
				NOTICE_STYLE);
			break;
			}
		}
	if (! source_file.canRead () ||
		! source_file.isFile ())
		{
		error_report =
			"Can't access file " + source_file.getAbsolutePath () + NL
			+"Processing of this file canceled.";
		Log_Message
			(error_report + NL + NL, NOTICE_STYLE);
		Procedure_Status (INACCESSIBLE_FILE);
		Config_Value
			(PROCEDURE_COMPLETION_NUMBER_PARAMETER,
			new Integer (INACCESSIBLE_FILE));
		Config_Value (TOTAL_FAILURE_COUNT,
			new Long (++Total_Failure_Count));
		Report_Processing_Event (new Processing_Changes ()
			.Configuration (The_Configuration)
			.Sequential_Failures (++Sequential_Failures));
		if (Stop_on_Failure != 0 &&
			Sequential_Failures >= Stop_on_Failure)
				Processing_State = HALTED;
		Close_Log_File ();
		if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
			System.out.println
				("<<< Process_Source (No Source File!)");
		return;
		}
	}

/*..............................................................................
	Configuration parameters update.
*/
Config_Value (SOURCE_NUMBER_PARAMETER,	  source_number);
Config_Value (SOURCE_ID_PARAMETER,        source_id);
Config_Value (SOURCE_PATHNAME_PARAMETER,  source_file.getAbsolutePath ());
Config_Value (SOURCE_DIRECTORY_PARAMETER, source_file.getParent ());
String
	name = source_file.getName ();
Config_Value (SOURCE_FILENAME_PARAMETER,  name);
int
	index = name.lastIndexOf ('.');
if (index < 0)
	index = name.length ();
Config_Value (SOURCE_FILENAME_ROOT_PARAMETER,      name.substring (0, index));
if (index < name.length ())
	index++;
Config_Value (SOURCE_FILENAME_EXTENSION_PARAMETER, name.substring (index));
Config_Value (LOG_FILENAME_PARAMETER,              log_file.getName ());
Config_Value (LOG_DIRECTORY_PARAMETER,             log_file.getParent ());

/*..............................................................................
	Process the procedures.
*/
Log_Message
	("Processing host: " + Conductor_ID + NL
	+"Procedures: " + Procedures_Table + NL + NL,
	HIGHLIGHT_STYLE);
int
	conductor_status = PROCEDURE_SUCCESS,
	procedure_status;
String
	procedure_sequence = null;

Procedures_Sequence:
while (Source_Status.size () < Procedure_Records.size ())
	{
	//	Get a copy of the next procedure record for processing.
	Procedure_Record =
		new Vector<String> (Procedure_Records.get (Source_Status.size ()));
	procedure_sequence = Procedure_Field (SEQUENCE_FIELD);
	Procedure_has_Status = false;

	if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
		System.out.println
			("=== Procedure sequence: " + procedure_sequence);
	Config_Value (PROCEDURE_COUNT_PARAMETER,
		new Integer (Source_Status.size () + 1));
	Config_Value (PROCEDURE_SEQUENCE_PARAMETER,
		procedure_sequence);
	Report_Processing_Event (new Processing_Changes ()
		.Configuration (The_Configuration)
		.Procedure_Record (Procedure_Record));

	//	Get the Command_Line and resolve it.
	String
		command_line = Procedure_Field (COMMAND_LINE_FIELD);
	if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
		System.out.println
			("    " + PROCEDURES_FIELD_NAMES[COMMAND_LINE_FIELD] + ": "
			+ command_line);
	if (command_line == null ||
		(command_line = command_line.trim ()).length () == 0)
		{
		conductor_status = INVALID_DATABASE_ENTRY;
		error_report =
			"Invalid field \""
			+ PROCEDURES_FIELD_NAMES[COMMAND_LINE_FIELD] + "\"" + NL
			+ "  with value: " + ((command_line == null) ? "NULL" : "empty");
		}
	if (error_report == null)
		{
		try
			{
			if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
				System.out.println
					("    Resolving the command line");
			command_line = Resolve (command_line);
			Procedure_Field (COMMAND_LINE_FIELD, command_line);
			}
		catch (Exception exception)
			{
			conductor_status = UNRESOLVABLE_REFERENCE;
			error_report =
				"Unable to resolve reference in field \""
				+ PROCEDURES_FIELD_NAMES[COMMAND_LINE_FIELD] + "\"" + NL
				+ exception.getMessage ();
			}
		}

	//	Get the Time_Limit and resolve it.
	String
		time_limit_string = null;
	if (error_report == null)
		{
		time_limit_string = Procedure_Field (TIME_LIMIT_FIELD);
		if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
			System.out.println
				("    " + PROCEDURES_FIELD_NAMES[TIME_LIMIT_FIELD] + ": "
				+ time_limit_string);
		if (time_limit_string == null ||
			time_limit_string.trim ().length () == 0)
			time_limit_string = "0";
		if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
			System.out.println
				("    Resolving the time limit");
		try {time_limit_string = Resolve (time_limit_string);}
		catch (Exception exception)
			{
			conductor_status = UNRESOLVABLE_REFERENCE;
			error_report =
				"Unable to resolve reference in field \""
				+ PROCEDURES_FIELD_NAMES[TIME_LIMIT_FIELD] + "\"" + NL
				+ exception.getMessage ();
			}
		}
	
	//	Evaluate the Time_Limit value.
	int
		time_limit = 0;
	if (error_report == null)
		{
		try {
			if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
				System.out.println
					("    Expression parsing the time limit");
			time_limit =
				(int)(Expression_Parser.parse (time_limit_string).getVal ());
			Procedure_Field (TIME_LIMIT_FIELD, String.valueOf (time_limit));
			if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
				System.out.println
					("    Time limit: " + time_limit);
			}
		catch (ParseError exception)
			{
			conductor_status = INVALID_DATABASE_ENTRY;
			error_report =
				"Unable to evaluate expression in field \""
				+ PROCEDURES_FIELD_NAMES[TIME_LIMIT_FIELD] + "\"" + NL
				+"   with value: " + Resolver.Pattern () + NL
				+"  resolved to: " + time_limit_string + NL
				+"  At index " + exception.context.pos
					+ ((exception.context.tokenString == null) ?
						"" :
						(" for token \""
							+ exception.context.tokenString + '"'))
					+ ":" + NL
				+"  " + exception.getMessage ();
			}
		}

	if (error_report != null)
		{
		if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
			System.out.println
				("    Error report -" + NL
				+ error_report);
		Log_Message (error_report + NL + NL, NOTICE_STYLE);
		Config_Value (TOTAL_FAILURE_COUNT,
			new Long (++Total_Failure_Count));
		++Sequential_Failures;
		Procedure_Status (conductor_status);
		Report_Processing_Event (new Processing_Changes ()
			.Procedure_Record (Procedure_Record));
		Close_Log_File ();
		throw Database_Error
			(error_report, source_number, procedure_sequence);
		}

	/*..........................................................................
		Run the procedure.
	*/
	if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
		System.out.println
			("    Running the procedure");
	Report_Processing_Event (new Processing_Changes ()
		.Procedure_Record (Procedure_Record));

	procedure_status = Run_Procedure (command_line, time_limit, NORMAL);

	if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
		System.out.println
			("    Procedure status: " + procedure_status);
	Config_Value (PROCEDURE_COMPLETION_NUMBER_PARAMETER,
		new Integer (procedure_status));
	Report_Processing_Event (new Processing_Changes ()
		.Configuration (The_Configuration));
	if (procedure_status < 0)
		{
		//	The procedure failed to execute correctly.
		if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
			System.out.println
				("    Procedure failed to execute");
		Config_Value (TOTAL_FAILURE_COUNT,
			new Long (++Total_Failure_Count));
		Procedure_Status (conductor_status = procedure_status);
		break Procedures_Sequence;
		}

	/*..........................................................................
		Check the competion status.

		The procedure completed and returned an exit status.
		Determine if the procedure completed successfully or not.
	*/
	//	Ensure non-null trimmed strings.
	String
		success_status_string = Procedure_Field (SUCCESS_STATUS_FIELD);
	if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
		System.out.println
			("    " + PROCEDURES_FIELD_NAMES[SUCCESS_STATUS_FIELD] + ": "
			+ success_status_string);
	if (success_status_string == null)
		success_status_string = "";
	else
		success_status_string = success_status_string.trim ();

	String
		success_message = Procedure_Field (SUCCESS_MESSAGE_FIELD);
	if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
		System.out.println
			("    " + PROCEDURES_FIELD_NAMES[SUCCESS_MESSAGE_FIELD] + ": "
			+ success_message);
	if (success_message == null)
		success_message = "";
	else
		success_message = success_message.trim ();

	if (success_status_string.length () == 0 &&
		success_message.length () == 0)
		{
		if (Empty_Success_Any)
			{
			if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
				System.out.println
					("    Empty_Success_Any");
			conductor_status = PROCEDURE_SUCCESS;
			Log_Message
				("Procedure completed with implied success." + NL + NL);
			}
		else
			{
			if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
				System.out.println
					("    Implicit success value 0");
			Log_Message
				(NL + "The "
				+ PROCEDURES_FIELD_NAMES[SUCCESS_STATUS_FIELD] + " and "
				+ PROCEDURES_FIELD_NAMES[SUCCESS_MESSAGE_FIELD]
				+" fields are both empty." + NL
				+"Asserting " + PROCEDURES_FIELD_NAMES[SUCCESS_STATUS_FIELD]
					+ " = 0." + NL);
			if (procedure_status == 0)
				{
				if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
					System.out.println
						("    Procedure success");
				conductor_status = PROCEDURE_SUCCESS;
				Log_Message
					("Procedure completed with success status." + NL + NL);
				}
			else
				{
				if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
					System.out.println
						("    Procedure failure");
				conductor_status = PROCEDURE_FAILURE;
				Log_Message
					("Expected exit status 0." + NL + NL,
					NOTICE_STYLE);
				}
			}
		}
	else if (success_status_string.length () != 0)
		{
		//	Use Success_Status.
		int
			success_status_value;
		String
			status_string = null;
		if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
			System.out.println
				("    Resolving success status");
		try {status_string = Resolve (success_status_string);}
		catch (Exception exception)
			{
			error_report =
				"Unable to resolve reference in field \""
				+ PROCEDURES_FIELD_NAMES[SUCCESS_STATUS_FIELD] + "\"" + NL
				+ exception.getMessage ();
			if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
				System.out.println
					("    Error report -" + NL
					+ error_report);
			Log_Message (error_report + NL + NL, NOTICE_STYLE);
			Config_Value (TOTAL_FAILURE_COUNT,
				new Long (++Total_Failure_Count));
			++Sequential_Failures;
			Procedure_Status (UNRESOLVABLE_REFERENCE, procedure_status);
			Report_Processing_Event (new Processing_Changes ()
				.Procedure_Record (Procedure_Record));
			Close_Log_File ();
			throw Database_Error
				(error_report, source_number, procedure_sequence);
			}

		//	Attempt expression evaluation.
		try
			{
			//	Assume it's a logical expression.
			if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
				System.out.println
					("    Logical expression parsing the success status");
			success_status_value =
				(int)(Expression_Parser.parseLogical (status_string).getVal ());
			if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
				System.out.println
					("    Success status condition: " + success_status_value);
			Procedure_Field (SUCCESS_STATUS_FIELD,
				String.valueOf (success_status_value));
			if (success_status_value == 0)
				{
				if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
					System.out.println
						("    Procedure failure");
				conductor_status = PROCEDURE_FAILURE;
				Log_Message
					("Procedure completed with success status condition false."
					+ NL
					+"Conditional: " + success_status_string + NL
					+"Resolves to: " + status_string + NL + NL,
						NOTICE_STYLE);
				}
			else
				{
				if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
					System.out.println
						("    Procedure success");
				conductor_status = PROCEDURE_SUCCESS;
				Log_Message
					("Procedure completed with success status condition true."
						+ NL + NL);
				}
			}
		catch (ParseError parse_error)
			{
			//	Try again as a numeric expression.
			try
				{
				if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
					System.out.println
						("    Expression parsing the success status");
				success_status_value =
					(int)(Expression_Parser.parse (status_string).getVal ());
				if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
					System.out.println
						("    Success status value: " + success_status_value);
				Procedure_Field (SUCCESS_STATUS_FIELD,
					String.valueOf (success_status_value));
				if (procedure_status == success_status_value)
					{
					if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
						System.out.println
							("    Procedure success");
					conductor_status = PROCEDURE_SUCCESS;
					Log_Message
						("Procedure completed with success status." + NL + NL);
					}
				else
					{
					if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
						System.out.println
							("    Procedure failure");
					conductor_status = PROCEDURE_FAILURE;
					Log_Message
						("Expected exit status "
							+ success_status_value + '.' + NL + NL,
						NOTICE_STYLE);
					}
				}
			catch (ParseError exception)
				{
				error_report =
					"Unable to evaluate expression in field \""
					+ PROCEDURES_FIELD_NAMES[SUCCESS_STATUS_FIELD]
						+ "\"" + NL
					+"   with value: " + Resolver.Pattern () + NL
					+"  resolved to: " + status_string + NL
					+"  At index " + exception.context.pos
						+ ((exception.context.tokenString == null) ?
							"" :
							(" for token \""
								+ exception.context.tokenString + '"'))
						+ ":" + NL
					+"  " + exception.getMessage ();
				if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
					System.out.println
						("    Error report -" + NL
						+ error_report);
				Log_Message (error_report + NL + NL, NOTICE_STYLE);
				Config_Value (TOTAL_FAILURE_COUNT,
					new Long (++Total_Failure_Count));
				++Sequential_Failures;
				Procedure_Status (INVALID_DATABASE_ENTRY, procedure_status);
				Report_Processing_Event (new Processing_Changes ()
					.Procedure_Record (Procedure_Record));
				Close_Log_File ();
				throw Database_Error
					(error_report, source_number, procedure_sequence);
				}
			}
		}
	else
		{
		//	Use Success_Message.
		if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
			System.out.println
				("    Resolving the success message");
		try {success_message = Resolve (success_message);}
		catch (Exception exception)
			{
			error_report =
				"Unable to resolve reference in field \""
				+ PROCEDURES_FIELD_NAMES[SUCCESS_MESSAGE_FIELD] + "\"" + NL
				+ exception.getMessage ();
			if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
				System.out.println
					("    Error report -" + NL
					+ error_report);
			Log_Message (error_report + NL + NL, NOTICE_STYLE);
			Config_Value (TOTAL_FAILURE_COUNT,
				new Long (++Total_Failure_Count));
			++Sequential_Failures;
			Procedure_Status (UNRESOLVABLE_REFERENCE, procedure_status);
			Report_Processing_Event (new Processing_Changes ()
				.Procedure_Record (Procedure_Record));
			Close_Log_File ();
			throw Database_Error
				(error_report, source_number, procedure_sequence);
			}
		// Attempt to match success_message to stdout, stderr.
		try
			{
			Procedure_Field (SUCCESS_MESSAGE_FIELD, success_message);
			if (stdout_Logger.Buffer ().toString ()
					.matches (success_message) ||
				stderr_Logger.Buffer ().toString()
					.matches (success_message))
				{
				if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
					System.out.println
						("    Procedure success");
				conductor_status = PROCEDURE_SUCCESS;
				Log_Message
					("Procedure completed with success message." + NL + NL);
				}
			else
				{
				if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
					System.out.println
						("    Procedure failure");
				conductor_status = PROCEDURE_FAILURE;
				Log_Message
					("Couldn't match success message regular expression -" + NL
					+ success_message + NL
					+"  resolved from success message pattern -" + NL
					+ Resolver.Pattern () + NL + NL,
					NOTICE_STYLE);
				}
			}
		catch (Exception exception)
			{
			// Success_Message did not resolve to a proper regex.
			error_report =
				"Invalid regular expression in field \""
				+ PROCEDURES_FIELD_NAMES[SUCCESS_MESSAGE_FIELD] + "\"" + NL
				+ "   with value: " + Resolver.Pattern () + NL
				+ "  resolved to: " + success_message + NL
				+ exception.getMessage ();
			if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
				System.out.println
					("    Error report -" + NL
					+ error_report);
			Log_Message (error_report + NL + NL, NOTICE_STYLE);
			Config_Value (TOTAL_FAILURE_COUNT,
				new Long (++Total_Failure_Count));
			++Sequential_Failures;
			Procedure_Status (BAD_REGEX, procedure_status);
			Report_Processing_Event (new Processing_Changes ()
				.Procedure_Record (Procedure_Record));
			Close_Log_File ();
			throw Database_Error
				(error_report, source_number, procedure_sequence);
			}
		}

	if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
		System.out.println
			("    Reporting procedure record");
	Procedure_Status (conductor_status, procedure_status);
	Report_Processing_Event (new Processing_Changes ()
		.Procedure_Record (Procedure_Record));
	if (conductor_status != PROCEDURE_SUCCESS)
		break Procedures_Sequence;
	}	//	Procedures_Sequence

if (conductor_status != PROCEDURE_SUCCESS)
	{
	//	Run the On_Failure procedure.
	if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
		System.out.println
			("    Reporting failure");
	Log_Message ("Procedure failed." + NL + NL, NOTICE_STYLE);
	Config_Value (SOURCE_FAILURE_COUNT,
		new Long (++Source_Failure_Count));
	Config_Value (TOTAL_FAILURE_COUNT,
		new Long (++Total_Failure_Count));
	++Sequential_Failures;
	Report_Processing_Event (new Processing_Changes ()
		.Configuration (The_Configuration)
		.Sequential_Failures (Sequential_Failures));

	String
		on_failure = Procedure_Field (ON_FAILURE_FIELD);
	if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
		System.out.println
			("    " + PROCEDURES_FIELD_NAMES[ON_FAILURE_FIELD] + ": "
			+ on_failure);
	if (on_failure == null ||
		(on_failure = on_failure.trim ()).length () == 0)
		{
		Log_Message
			("No " + PROCEDURES_FIELD_NAMES[ON_FAILURE_FIELD]
			+" procedure." + NL + NL,
			NOTICE_STYLE);
		}
	else
		{
		try
			{
			if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
				System.out.println
					("    Resolving the on failure command");
			on_failure = Resolve (on_failure);
			Procedure_Field (ON_FAILURE_FIELD, on_failure);
			}
		catch (Exception exception)
			{
			error_report =
				"Unable to resolve reference in field \""
				+ PROCEDURES_FIELD_NAMES[ON_FAILURE_FIELD] + "\"" + NL
				+ exception.getMessage ();
			if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
				System.out.println
					("    Error report -" + NL
					+ error_report);
			Log_Message (error_report + NL + NL, NOTICE_STYLE);
			Close_Log_File ();
			throw Database_Error (error_report);
			}

		//	Run the on-failure procedure.
		if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
			System.out.println
				("    Running the on failure command");
		Report_Processing_Event (new Processing_Changes ()
			.Procedure_Record (Procedure_Record));

		procedure_status = Run_Procedure (on_failure, 0, ON_FAILURE);

		if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
			System.out.println
				("    On failure procedure status: " + procedure_status);
		if (procedure_status < 0)
			Log_Message
				(error_report = PROCEDURES_FIELD_NAMES[ON_FAILURE_FIELD]
					+" procedure failed: "
					+ Status_Conductor_Code_Description (procedure_status)
					+ ' ' + NL + NL,
				NOTICE_STYLE);
		else
			Log_Message
				(error_report = PROCEDURES_FIELD_NAMES[ON_FAILURE_FIELD]
					+" procedure completed with status "
					+ procedure_status + NL + NL);
		}
	if (Stop_on_Failure != 0 &&
		Sequential_Failures >= Stop_on_Failure)
		Processing_State = HALTED;
	}
else
	{
	//	Source completed processing successfully.
	if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
		System.out.println
			("    Reporting success");
	Config_Value (SOURCE_SUCCESS_COUNT,
		new Long (++Source_Success_Count));
	Report_Processing_Event (new Processing_Changes ()
		.Configuration (The_Configuration)
		.Sequential_Failures (Sequential_Failures = 0));
	}

// Close the log file.
Close_Log_File ();
if ((DEBUG & DEBUG_PROCESS_SOURCE) != 0)
	System.out.println
		("<<< Process_Source");
}

/*==============================================================================
	Procedure Processing
*/
/**	Run a system procedure.
<p>
	The command line is {@link #Parse_Command_Line(String) parsed} into
	an argument vector and submitted to <code>Runtime.exec</code>. If
	the <code>exec</code> failed the NO_PROCEDURE status (a negative
	value) is returned.
<p>
	The system process ID (PID) is obtained from the resulting Process
	and used to set the initial status of the procedure.
<p>
	Logger threads are started to send the stdout and stderr streams
	from the process to appropriate Writers.
<p>
	A <code>waitFor</code> on the Process is started that will block
	until either the procedure completes or the specified timeout
	period has expired.
<p>
	When the procedure completes its exit status is obtained. The exit
	status is expected to be a positive value (including zero). If the
	procedure times out the exit status is set to PROCEDURE_TIMEOUT,
	which is a negative value.
<p>
	The logger threads are always signaled to end and then allowed to
	finish on their own before returning from this method.
<p>
	@param	command_line	The command line String that will run
		the procedure on the host system.
	@param	time_limit		The maximum amount of time, in seconds,
		to wait for the running process to complete.
	@param	normal_procedure	A flag to indicate that the PID of
		the running process is to be used to set the initial
		{@link #Procedure_Status(int) Procedure_Status}.
	@return	An status code int. This will be positive if the procedure
		ran to completion; otherwise a negative Conductor procedure
		completion code will be returned.
	@throws	Database_Exception	if the {@link #Procedure_Status(int)
		Procedure_Status} could not be set to the PID.
	@throws	IOException	if a {@link Log_Message(String, AttributeSet)
		Log_Message} could not be written.
	@see Runtime.exec(String[])
*/
private int Run_Procedure
	(
	String	command_line,
	int		time_limit,
	boolean	normal_procedure
	)
	throws Database_Exception, IOException
{
if ((DEBUG & DEBUG_RUN_PROCEDURE) != 0)
	System.out.println
		(">>> Conductor.Run_Procedure:" + NL
		+"    command line: " + command_line + NL
		+"      time limit: " + time_limit + NL
		+"          normal: " + normal_procedure);
//	Log the start of a new procedure.
String
	description = null;
if (normal_procedure)
	{
	Log_Message (PROCEDURE_LOG_DELIMITER, MARKER_STYLE);
	description = Procedure_Field (DESCRIPTION_FIELD);
	}
else
	Log_Message (ON_FAILURE_PROCEDURE_LOG_DELIMITER, MARKER_STYLE);
Log_Message ("Procedure start time: " + new Date () + NL + NL);
Log_Message
	(PROCEDURES_FIELD_NAMES[SEQUENCE_FIELD] + ": "
		+ Procedure_Field (SEQUENCE_FIELD) + NL
	+ ((description == null) ? "" :
		PROCEDURES_FIELD_NAMES[DESCRIPTION_FIELD] + ": "
		+ description + NL)
	+ (normal_procedure ?
		PROCEDURES_FIELD_NAMES[COMMAND_LINE_FIELD] :
		PROCEDURES_FIELD_NAMES[ON_FAILURE_FIELD]) + ": "
		+ command_line + NL + NL,
	HIGHLIGHT_STYLE);

//	Attempt to start the procedure.
UNIX_Process
	procedure;
try {procedure = new UNIX_Process (Runtime.getRuntime ()
		.exec (Parse_Command_Line (command_line)));}
catch (IOException exception)
	{
	//	Couldn't execute the procedure!
	String
		message = exception.getMessage ();
	if (message == null)
		message = "";
	else
		message += NL;
	Log_Message
		("Procedure could not be executed." + NL + message + NL,
		NOTICE_STYLE);
	if ((DEBUG & DEBUG_RUN_PROCEDURE) != 0)
		System.out.println
			("<<< Run_Procedure: " + NO_PROCEDURE);
	return NO_PROCEDURE;
	}
catch (NoSuchFieldException exception)
	{
	String
		message =
			"Unable to obtain the process ID." + NL
			+"The Process does not appear to be a UNIX Process." + NL
			+ exception.getMessage () + NL;
	Log_Message (message, NOTICE_STYLE);
	throw new IOException (ID + NL
		+ message);
	}
if (normal_procedure)
	Procedure_Status (procedure.ID ());
if ((DEBUG & DEBUG_RUN_PROCEDURE) != 0)
	System.out.println
		("    PID = " + procedure.ID ());

//	Start Stream_Logger threads for the procedure's stdout and stderr.
stdout_Logger = new Stream_Logger
	(STDOUT_NAME, procedure.getInputStream (), Logger);
stderr_Logger = new Stream_Logger
	(STDERR_NAME, procedure.getErrorStream (), Logger);
stdout_Logger.start ();
stderr_Logger.start ();

int
	exit_status = 0;
String
	exit_message = null;
SimpleAttributeSet
	style = null;
//	Wait for the procedure to complete or timeout.
try
	{
	//	N.B.: Only the least significant 8 bits are used.
	exit_status = procedure.waitFor (time_limit * 1000) & 0xFF;
	exit_message = "Procedure completed with status " + exit_status + ".";
	}
catch (IOException exception)
	{
	procedure.destroy ();
	exit_status = PROCEDURE_TIMEOUT;
	exit_message = "Procedure timeout after " + time_limit + " seconds." + NL
		+ exception.getMessage ();
	style = NOTICE_STYLE;
	}
catch (InterruptedException exception)
	{
	procedure.destroy ();
	exit_status = PROCEDURE_TIMEOUT;
	exit_message = "Procedure interrupted!" + NL
		+ exception.getMessage ();
	style = NOTICE_STYLE;
	}
if ((DEBUG & DEBUG_RUN_PROCEDURE) != 0)
	System.out.println
		("    " + exit_message);

//	Shutdown the loggers.
if (stdout_Logger.isAlive ())
	{
	Sleep (stdout_Logger.Polling_Interval () * 2);
	if ((DEBUG & DEBUG_RUN_PROCEDURE) != 0)
		System.out.println
			("    stdout_Logger.Close");
	stdout_Logger.Close ();
	while (stdout_Logger.isAlive ())
		{
		if ((DEBUG & DEBUG_RUN_PROCEDURE) != 0)
			System.out.println
				("    Waiting for stdout_Logger to end....");
		try {stdout_Logger.join ();}
		catch (InterruptedException e) {}
		}
	}
if (stderr_Logger.isAlive ())
	{
	if ((DEBUG & DEBUG_RUN_PROCEDURE) != 0)
		System.out.println
			("    stderr_Logger.Close");
	Sleep (stderr_Logger.Polling_Interval () * 2);
	stderr_Logger.Close ();
	while (stderr_Logger.isAlive ())
		{
		if ((DEBUG & DEBUG_RUN_PROCEDURE) != 0)
			System.out.println
				("    Waiting for stderr_Logger to end....");
		try {stderr_Logger.join ();}
		catch (InterruptedException e) {}
		}
	}
try
	{
	procedure.getInputStream ().close ();
	procedure.getErrorStream ().close ();
	procedure.getOutputStream ().close ();
	}
catch (IOException exception) {}

Log_Message (NL + "Procedure end time: " + new Date () + NL);
Log_Message (exit_message + NL, style);
if ((DEBUG & DEBUG_RUN_PROCEDURE) != 0)
	System.out.println
		("<<< Run_Procedure: " + exit_status + NL
		+"    " + exit_message);
return exit_status;
}

/**	Parse a String into command line arguments.
<p>
	The command line arguments are delimited by any combination of
	space (' '), tab ('\t'), new-line ('\n') or carriage return ('\r')
	characters. However, character sequences in quotes - either single
	('\'') or double ('"') quote characters - remain unbroken.
<p>
	After parsing each argument String is also scanned to convert
	escaped characters - preceded by a backslash ('\') - into their
	unescaped character equivalents.
<p>
	@param	command_line	A String to be parsed.
	@return	An array of argument Strings.
*/
public static String[] Parse_Command_Line
	(
	String	command_line
	)
{
if ((DEBUG & DEBUG_PARSE_COMMAND_LINE) != 0)
	System.out.println 
		(">>> Conductor.Parse_Command_Line: " + command_line);
if (command_line == null)
	command_line = "";
Vector<String>
	arguments = new Vector<String> ();
int
	start = 0,
	end,
	last = command_line.length ();

//	Collect argument sequences.
while (start < last)
	{
	//	Skip leading whitespace characters.
	if (COMMAND_LINE_ARGUMENTS_DELIMITERS
			.indexOf (command_line.charAt (start++)) >= 0)
		continue;
	end = --start;
	String
		argument = "";

	//	Seek the next unquoted whitespace character.
	while (end < last &&
			COMMAND_LINE_ARGUMENTS_DELIMITERS
				.indexOf (command_line.charAt (end)) < 0)
		{
		//	Non-whitespace character.
		char
			character = command_line.charAt (end);
		if (character == '"' ||
			character == '\'')
			{
			//	Quoted sequence.
			if (start < end)
				//	Append the leading unquoted sequence to the argument.
				argument += command_line.substring (start, end);
			start = ++end;
			
			//	Find the end of the quoted section.
			while ((end = command_line.indexOf (character, end)) > 0 &&
					command_line.charAt (end - 1) == '\\')
				//	Escaped quote.
				end++;
			if (end < 0)
				{
				//	Unclosed quote; goes to the end of the string.
				end = last;
				break;
				}
			if (start < end)
				//	Append this quoted sequence to the argument.
				argument += command_line.substring (start, end);
			start = ++end;
			}
		else
			end++;
		}
	if (start < end)
		//	Append the sequence up to this point to the argument.
		argument += command_line.substring (start, end);
	//	Remove any escapes.
	argument = new String_Buffer (argument).escape_to_special ().toString ();
	arguments.add (argument);
	start = end;
	}
if ((DEBUG & DEBUG_PARSE_COMMAND_LINE) != 0)
	System.out.println 
		("<<< Parse_Command_Line: " + arguments);
String[]
	args_array = new String[arguments.size ()];
arguments.toArray (args_array);
return args_array;
}

/*------------------------------------------------------------------------------
	Procedure status
*/
private void Procedure_Status
	(
	int		conductor_status
	)
	throws Database_Exception
{
String
	status_value = Status_Indicator (conductor_status);
if (Procedure_has_Status)
	Source_Status.setElementAt (status_value, Source_Status.size () - 1);
else
	{
	Procedure_has_Status = true;
	Source_Status.add (status_value);
	}
Update_Source_Record (STATUS_FIELD, Status_Field_Value (Source_Status));
}

private void Procedure_Status
	(
	int		conductor_status,
	int		procedure_status
	)
	throws Database_Exception
{

String
	status_value = Status_Indicator (conductor_status, procedure_status);
if (Procedure_has_Status)
	Source_Status.setElementAt (status_value, Source_Status.size () - 1);
else
	{
	Procedure_has_Status = true;
	Source_Status.add (status_value);
	}
Update_Source_Record (STATUS_FIELD, Status_Field_Value (Source_Status));
}
	
/**	Parse a Source table Status field value into a Vector of procedure
	status indicator Strings.
<p>
	A Source table Status field value contains a comma delimited list
	of procedure status indicators. Each has the form:
<blockquote>
	&lt;<i>PID</i>&gt; | &lt;<i>Conductor code</i>&gt;[(&lt;<i>procedure exit status</i>&gt;)]
</blockquote><p>
	The PID is the positive, non-zero value of the system's process ID
	for a running procedure. This value should only be present if the
	procedure is currently executing.
<p>
	When the procedure processing has completed the PID is replaced
	with a Conductor procedure completion code. This will be zero if
	Conductor determined that the procedure completed successfully. It
	will be one (1) if the procedure completed unsuccessfully. It will
	be a negative value if the procedure could not be run to completion
	for any reason; the code value in this case indicates the reason.
<p>
	When the procedure has been run to completion, whether successful
	or not, its exit status value is appended inside parentheses to the
	Conductor procedure completion code.
<p>
	@param	status_field	The String from a Source table Status field
		value (may be null).
	@return	A Vector of procedure status indicator Strings.
*/
public static Vector<String> Status_Indicators
	(
	String	status_field
	)
{
Vector<String>
	status = new Vector<String> ();
if (status_field != null)
	{
	StringTokenizer
		tokenizer = new StringTokenizer (status_field, ",");
	while (tokenizer.hasMoreTokens ())
		status.add (tokenizer.nextToken ());
	}
return status;
}

/**	Get the Conductor procedure completion status code value from a
	procedure status indicator String.
<p>
	@param	status	A procedure status indicator String.
	@return	The Conductor procedure completion status code value.
	@throws	NumberFormatException	if a value could not be formed.
	@see	#Status_Indicators(String)
*/
public static int Status_Conductor_Code
	(
	String	status
	)
	throws NumberFormatException
{
if (status != null &&
	status.length () != 0)
	{
	int
		index = status.indexOf ('(');
	if (index < 0)
		index = status.length ();
	if (index > 0)
		{
		try {return Integer.parseInt (status.substring (0, index));}
		catch (NumberFormatException exception) {}
		}
	}
throw new NumberFormatException
	("No Conductor procedure completion code present in \"" + status + "\".");
}

/**	Get a description String for a Conductor procedure completion code.
<p>
	If the code value is not a recognized value, the description will
	be "Unknown procedure completion code (<value>)."
<p>
	@param	code	The code value.
	@return	A String describing the meaning of the code value.
	@see	#Status_Conductor_Code(String)
*/
public static String Status_Conductor_Code_Description
	(
	int		code
	)
{
if (code < 0 && -(code + 1) < FAILURE_DESCRIPTION.length)
	return FAILURE_DESCRIPTION[-(code + 1)];
if (code == PROCEDURE_SUCCESS)
	return "Procedure success";
if (code == PROCEDURE_FAILURE)
	return "Procedure failure";
return "Unknown procedure completion code (" + String.valueOf (code) + ").";
}

/**	Get the procedure exit status value from a procedure status indicator
	String.
<p>
	@param	status	A procedure status indicator String.
	@return	The procedure exit status value.
	@throws	NumberFormatException	if a value could not be formed.
	@see	#Status_Indicators(String)
*/
public static int Status_Procedure_Exit_Value
	(
	String	status
	)
	throws NumberFormatException
{
if (status != null &&
	status.length () != 0)
	{
	int
		start = status.indexOf ('('),
		end;
	if (start >= 0)
		{
		end = status.indexOf (')', ++start);
		if (start < end)
			{
			try {return Integer.parseInt (status.substring (start, end));}
			catch (NumberFormatException exception) {}
			}
		}
	}
throw new NumberFormatException
	("No procedure exit status present in \"" + status + "\".");
}

/**	Assemble a properly formatted String for a Source table Status field
	value.
<p>
	This method is the converse of the {@link #Status_Indicators(String)
	Status_Indicators} method.
<p>
	@param	status	A Vector of procedure status indicator Strings.
	@return	A String suitably formatted for a Source table Status field
		value.
*/
public static String Status_Field_Value
	(
	Vector<String>	status
	)
{
String
	field_value = "";
for (int
		index = 0;
		index < status.size ();
		index++)
	{
	if (field_value.length () != 0)
		field_value += ",";
	field_value += status.get (index);
	}
return field_value;
}

/**	Assemble a properly formatted procedure status indicator String
	as used in a Sources table Status field value. 
<p>
	@param	conductor_status	A conductor procedure completion code.
	@param	procedure_status	A procedure exit status value.
	@return	A String suitably formatted for a Source table Status field
		value.
*/
public static String Status_Indicator
	(
	int		conductor_status,
	int		procedure_status
	)
{
return
	String.valueOf (conductor_status) + '(' +
	String.valueOf (procedure_status) + ')';
}

/**	Assemble a properly formatted procedure status indicator String
	as used in a Sources table Status field value. 
<p>
	@param	conductor_status	A conductor procedure completion code.
	@return	A String suitably formatted for a Source table Status field
		value.
*/
public static String Status_Indicator
	(
	int		conductor_status
	)
{return String.valueOf (conductor_status);}

/*==============================================================================
	Logging
*/
//	Management implementation.
/**	Register a Writer to receive processing log stream output.
<p>
	The Conductor writes its processing log reports, including the
	output from all pipeline procedures it runs, to all registered
	log Writers.
<p>
	@param	writer	A Writer object.
	@return	This Conductor Management object.
	@see	#Enable_Log_Writer(Writer, boolean)
	@see	#Remove_Log_Writer(Writer)
*/
public Management Add_Log_Writer
	(
	Writer	writer
	)
{
if ((DEBUG & DEBUG_MANAGEMENT) != 0)
	System.out.println
		(">-< Conductor.Add_Log_Writer:" + NL
		+ writer + NL
		+"    Styled_Writer: " + (writer instanceof Styled_Writer));
Logger.Add (writer);
return this;
}

//	Management implementation.
/**	Unregister a log Writer.
<p>
	@param	writer	A Writer object.
	@return	true	If the writer was registered and is now removed;
		false if it was not registered.
	@see	#Add_Log_Writer(Writer)
*/
public boolean Remove_Log_Writer
	(
	Writer	writer
	)
{
if ((DEBUG & DEBUG_MANAGEMENT) != 0)
	System.out.println
		(">-< Conductor.Remove_Log_Writer: " + writer);
return Logger.Remove (writer);
}

//	Management implementation.
/**	Enable or disable output to a {@link #Add_Log_Writer(Writer)
	registered log stream Writer}.
<p>
	@param	writer	A Writer that has been registered to receive
		Conductor log stream output. If the writer is not {@link
		#Add_Log_Writer(Writer) registered} to receive the Conductor log
		stream nothing is done.
	@param	enable	If false, Conductor log stream output to the Writer is
		suspended without having to unregister the Writer. If true, a
		Writer that has had its log stream output suspended will begin
		receiving it again.
	@return	This Conductor Management object.
*/
public Management Enable_Log_Writer
	(
	Writer	writer,
	boolean	enable
	)
{
Logger.Suspend (writer, ! enable);
return this;
}

/**	Logs a message to the Logger.
<p>
	@param	message	The message String to write to the Logger.
	@param	style	An AttributeSet style to apply to the message. This
		may be null to use the default text style.
	@throws	IOException	if the Log_File_Writer could not be written. If a
		Writer other than the Log_File_Writer throws an exception it is
		closed and {@link #Remove_Log_Writer(Writer) removed} from the
		Logger.
*/
protected void Log_Message
	(
	String			message,
	AttributeSet	style
	)
	throws IOException
{
try {Logger.Write (message, style);}
catch (Multiwriter_IOException exception)
	{
	//	Check if the log file threw an exception.
	boolean
		log_file_exception = false;
	Vector<Writer>
		writers = exception.Sources;
	Writer
		writer;
	int
		index = writers.size ();
	while (--index >= 0)
		{
		writer = writers.get (index);
		if (writer == Log_File_Writer)
			log_file_exception = true;
		else
			{
			try {writer.close ();}
			catch (IOException exeption) {}
			Remove_Log_Writer (writer);
			}
		}
	if (log_file_exception)
		{
		//	The local Log_File_Writer threw an exception.
		Close_Log_File ();
		throw (IOException)exception.Exceptions.get (index);
		}
	}
}

/**	Logs a message to the Logger.
<p>
	The default text style is used.
<p>
	@param	message	The message String to write to the Logger.
	@throws	IOException	if the Log_File_Writer could not be written.
	@see	#Log_Message(String, AttributeSet)
*/
protected void Log_Message
	(
	String	message
	)
	throws IOException
{Log_Message (message, null);}


private void Close_Log_File ()
{
if (Log_File_Writer != null)
	{
	//	Remove the log file writer from the Logger.
	Remove_Log_Writer (Log_File_Writer);
	try {Log_File_Writer.flush ();}
	catch (IOException exception) {}
	try {Log_File_Writer.close ();}
	catch (IOException exception) {}
	}
Log_File_Writer = null;
}


private void Stop_Logging ()
{
Close_Log_File ();
try {Logger.flush ();}
catch (IOException exception) {}
try {Logger.close ();}
catch (IOException exception) {}
Logger.Remove_All ();
}

/*==============================================================================
	Stage_Manager
*/
/**	Flag that determines if the Conductor requires a Stage_Manager.
<p>
	If true and a Stage_Manager connection can not be established the
	Conductor will throw an exception; otherwise the Conductor will
	proceed without a Stage_Manager.
<p>
	The initial value is false;
*/
public static boolean
	Require_Stage_Manager			= false;

private Local_Theater
	Theater_Management				= new Local_Theater (this);

private String
	Stage_Manager_Key				= null;


private String Connect_to_Stage_Manager ()
	throws IOException
{
if ((DEBUG & DEBUG_MANAGEMENT) != 0)
	System.out.println (">>> Connect_to_Stage_Manager");
String
	report = null;
try {Theater_Management.Open (Stage_Manager_Key);}
catch (IOException exception)
	{
	if ((DEBUG & DEBUG_MANAGEMENT) != 0)
		{
		System.out.println
			("    Exception: " + exception);
		if (exception instanceof ConnectException)
			System.out.println ("    ConnectException");
		if (exception instanceof Theater_Protocol_Exception)
			System.out.println ("    Theater_Protocol_Exception: Reason "
				+ ((Theater_Protocol_Exception)exception).Reason ());
		}
	if (Require_Stage_Manager ||
		(! (exception instanceof ConnectException) &&
		   (exception instanceof Theater_Protocol_Exception &&
				((Theater_Protocol_Exception)exception).Reason ()
					!= Theater_Protocol_Exception.TIMEOUT)))
		{
		Theater_Management.Auto_Open (false);
		if ((DEBUG & DEBUG_MANAGEMENT) != 0)
			System.out.println
				("    Exception being thrown");
		throw exception;
		}

	report =
	NL
	+"Note: A connection to the Stage_Manager could not be established." + NL
	+ exception.getMessage () + NL;
	}
if ((DEBUG & DEBUG_MANAGEMENT) != 0)
	System.out.println
		("    report -" + NL
		+ report + NL
		+ "<<< Connect_to_Stage_Manager");
return report;
}


private void Disconnect_from_Stage_Manager ()
{
if ((DEBUG & DEBUG_EXIT) != 0)
	System.out.println
		(">>> Conductor.Disconnect_from_Stage_Manager" + NL
		+"    Theater_Management.Auto_Open (false) ...");
//	Prevent auto-open when Done.
Theater_Management.Auto_Open (false);

if ((DEBUG & DEBUG_EXIT) != 0)
	System.out.println
		("    Stop_Logging ...");
Stop_Logging ();

if (The_Database != null)
	{
	if ((DEBUG & DEBUG_EXIT) != 0)
		System.out.println
			("    The_Database.Disconnect ...");
	try {The_Database.Disconnect ();}
	catch (Database_Exception exception) {}
	}

if ((DEBUG & DEBUG_EXIT) != 0)
	System.out.println
		("    Theater_Management.Close ...");
Theater_Management.Close ();
if ((DEBUG & DEBUG_EXIT) != 0)
	System.out.println
		("<<< Conductor.Disconnect_from_Stage_Manager");
}

//	Management implementation.
/**	Test if the Conductor is connected to a Stage_Manager.
<p>
	@return	true if the Conductor is connected to a Stage_Manager via
		an open Local_Theater; false otherwise.
	@see	Local_Theater
*/
public boolean Connected_to_Stage_Manager ()
{return Theater_Management.Opened ();}


private Message
	Conductor_Identity	= null;

//	Management implementation.
/**	Get the identity description Message for this Conductor.
<p>
	The identity Message contains the following parameters:
<p>
<dl>
<dt>{@link Message#ACTION_PARAMETER_NAME}
<dd>Indicates an identity Message with the {@link Message#IDENTITY_ACTION}
	value.

<dt>{@link Message#NAME_PARAMETER_NAME}
<dd>The identity name is {@link #CONDUCTOR_GROUP}.

<dt>{@link #HOSTNAME_PARAMETER}
<dd>The {@link Host#FULL_HOSTNAME} of the host system.

<dt>{@link #CONDUCTOR_ID_PARAMETER}
<dd>The Conductor ID is the {@link Host#SHORT_HOSTNAME} followed by a
	colon (':') and the system process ID of this Conductor. If the
	process ID can not be obtained only the short hostname is included.

<dt>{@link #CATALOG_PARAMETER}
<dd>The name of the Database catalog where the pipeline tables are located.

<dt>{@link #PIPELINE_PARAMETER}
<dd>The name of the Conductor pipeline being managed.

<dt>{@link #CONFIGURATION_SOURCE_PARAMETER}
<dd>The {@link Configuration#Source() source} of the Configuration that
	is being used.

<dt>{@link Message#CLASS_ID_PARAMETER_NAME}
<dd>The {@link #ID} of this Conductor class.

<dt>{@link #DATABASE_SERVER_PARAMETER}
<dd>Provides the name of the Configuration parameter group that contains
	the database access parameters.

<dt>A parameter group named the same as the value of the
	{@link #DATABASE_SERVER_PARAMETER}. This group contains the following
	parameters:
<p>
<dl>
<dt>{@link Database#TYPE}
<dd>The value is taken from the {@link Database#Configuration() database
	Configuration} parameter of the same name.

<dt>{@link Configuration#HOST}
<dd>The value is taken from the {@link Database#Configuration() database
	Configuration} parameter of the same name. However, if the value is
	"localhost" then the {@link Host#FULL_HOSTNAME} is used instead.

<dt>{@link Configuration#USER}
<dd>The value is taken from the {@link Database#Configuration() database
	Configuration} parameter of the same name.
</dl>
</dl>
<p>
	@return	A Message containing the idenity description parameters for
		this Conductor.
*/
public Message Identity ()
{
if (Conductor_Identity == null)
	{
	if ((DEBUG & DEBUG_MANAGEMENT) != 0)
		System.out.println (">>> Identity");
	Conductor_Identity = Message
		.Identity ()
		.Set (Message.NAME_PARAMETER_NAME, CONDUCTOR_GROUP)
		.Set (HOSTNAME_PARAMETER, Host.FULL_HOSTNAME)
		.Set (CONDUCTOR_ID_PARAMETER, Conductor_ID)
		.Set (CATALOG_PARAMETER, Catalog)
		.Set (PIPELINE_PARAMETER,
			Config_Value (PIPELINE_PARAMETER))
		.Set (CONFIGURATION_SOURCE_PARAMETER,
			Config_Value (CONFIGURATION_SOURCE_PARAMETER))
		.Set (Message.CLASS_ID_PARAMETER_NAME, ID);
	try
		{
		if ((DEBUG & DEBUG_MANAGEMENT) != 0)
			System.out.println
				("    Database Configuration -" + NL
				+ The_Database.Configuration ().Description ());
		String
			database_hostname = The_Database.Configuration ().Get_One
				(Database_Server_Name + Configuration.Path_Delimiter ()
					+ Configuration.HOST);
		if ("localhost".equals (database_hostname))
			database_hostname = Host.FULL_HOSTNAME;
		Conductor_Identity
			.Set (DATABASE_SERVER_PARAMETER, Database_Server_Name)
			.Add (new Parameter (Database_Server_Name)
				.Add (new Parameter (Database.TYPE)
					.Value (The_Database.Configuration ().Get_One
						(Database.TYPE)))
				.Add (new Parameter (Configuration.HOST)
					.Value (database_hostname))
				.Add (new Parameter (Configuration.USER)
					.Value (The_Database.Configuration ().Get_One
						(Configuration.USER))));
		}
	catch (PVL_Exception exception) {}
	if ((DEBUG & DEBUG_MANAGEMENT) != 0)
		System.out.println
			(Conductor_Identity.Description () + NL
			+ "<<< Identity");
	}
return Conductor_Identity;
}

/*==============================================================================
	Helpers
*/
private static String Error_Message
	(
	String	message
	)
{
String
	prefix =
		ID + NL +
		"Conductor ID: " + Conductor_ID;
if (message == null)
	message = prefix;
else if (message.indexOf (ID) < 0)
	message = prefix + NL + message;
return message;
}


private Database_Exception Database_Error
	(
	String	message,
	String	source_number,
	String	procedure_sequence
	)
{
String
	database_type = Config_Value (DATABASE_TYPE_PARAMETER),
	database_host = Config_Value (DATABASE_HOSTNAME_PARAMETER),
	report =
		"Problem with "
		+ ((database_type == null) ? "" : (database_type + ' ')) + "database "
		+ ((database_host == null) ? "" : ("on host " + database_host)) + NL
		+ "in pipeline " + Pipeline;
if (source_number != null)
	report += " for source file number " + source_number;
if (procedure_sequence != null)
	report += " at procedure sequence " + procedure_sequence;
report += "." + NL + message;
return new Database_Exception (Error_Message (report));
}


private Database_Exception Database_Error
	(
	String	message,
	String	source_number
	)
{return Database_Error (message, source_number, null);}

private Database_Exception Database_Error
	(
	String	message
	)
{return Database_Error (message, null, null);}


private static class Status_Report
{
public Conductor
	The_Conductor	= null;
public String
	Message			= null;
public int
	Status			= EXIT_SUCCESS;


public Status_Report
	(
	Conductor	conductor
	)
{The_Conductor = conductor;}


public Status_Report
	(
	int		status
	)
{Status = status;}


public Status_Report
	(
	Conductor	conductor,
	String		message,
	Exception	exception
	)
{this (conductor, message, exception, 0);}


public Status_Report
	(
	Conductor	conductor,
	String		message,
	Exception	exception,
	int			status
	)
{
The_Conductor = conductor;
Message = message;
Status = status;

if (exception != null)
	{
	if (message == null)
		message = "";
	else
		message += NL;

	String
		when = " occured " + ((conductor == null) ?
			"while constructing the Conductor." :
			"during pipeline processing.");

	if (exception instanceof Configuration_Exception)
		{
		Status = EXIT_CONFIGURATION_PROBLEM;
		Message = Error_Message (message
			+ "A Configuration problem" + when + NL
			+ exception.getMessage ());
		}
	else
	if (exception instanceof Database_Exception)
		{
		Status = EXIT_DATABASE_PROBLEM;
		Message = Error_Message (message
			+ "A database problem occured" + when + NL
			+ exception.getMessage ());
		}
	else
	if (exception instanceof IOException)
		{
		Status = EXIT_IO_FAILURE;
		Message = Error_Message (message
			+ "An I/O problem occured" + when + NL
			+ exception.getMessage ());
		}
	else
		{
		Status = EXIT_UNEXPECTED_EXCEPTION;
		StringWriter
			report_writer = new StringWriter ();
		exception.printStackTrace (new PrintWriter (report_writer, true));
		Message = Error_Message (message
			+ "An unexpected exception occured" + when + NL
			+ report_writer.toString ());
		}
	}
else
if (conductor != null &&
	conductor.Stop_on_Failure () > 0 &&
	conductor.Stop_on_Failure () == conductor.Sequential_Failures ())
	{
	if (message == null)
		message = "";
	else
		message += NL;
	Status = EXIT_TOO_MANY_FAILURES;
	Message = Error_Message (message
		+ "Too many sequential source processing failures ("
			+ conductor.Sequential_Failures + ").");
	}
}

}	//	Status_Report

/*==============================================================================
	Application main
*/
/**	Instantiate a Conductor application.
<p>
	@param	args	The Conductor {@link #Usage() command line} arguments.
*/
public static void main
	(
	String[]	args
	)
{
Configuration
	configuration = null;
configuration.Duplicate_Parameter_Action_Default
	= DEFAULT_DUPLICATE_PARAMETER_ACTION;

String
	database_server_name = null,
	pipeline = null,
	catalog = null;

boolean
	managed = false,
	wait_to_start = false;

if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println
		("*** " + ID + " ***");
for (int count = 0;
	 count < args.length;
	 count++)
	{
	if (args[count].length () > 1 &&
		args[count].charAt (0) == '-')
		{
		switch (args[count].toUpperCase ().charAt (1))
			{
			case 'P':	//	Pipeline.
				if (++count == args.length)
					{
					stdout.println
						("Missing pipeline name.");
					Usage ();
					}
				if (pipeline != null)
					{
					stdout.println
						("Multiple pipelines specified -" + NL
						+"    " + pipeline + " and " + args[count]);
					Usage ();
					}
				pipeline = args[count];
				break;

			case 'C':
				if (args[count].length () > 2 &&
				   (args[count].charAt (1) == 'A' ||
					args[count].charAt (2) == 'a'))
					{
					//	Catalog.
					if (++count == args.length)
						{
						stdout.println
							("Missing catalog name.");
						Usage ();
						}
					catalog = args[count];
					break;
					}

				//	Configuration.
				if (++count == args.length)
					{
					stdout.println
						("Missing configuration filename.");
					Usage ();
					}
				if (configuration != null)
					{
					stdout.println
						("Multiple configuration files specified -" + NL
						+"    " + configuration.Source () + " and "
						+ args[count]);
					Usage ();
					}
				try {configuration = new Configuration (args[count]);}
				catch (Configuration_Exception exception)
					{
					stdout.println
						("Unable to construct the configuration for \""
						+ args[count] + "\"." + NL
						+ exception.getMessage ());
					Exit (EXIT_CONFIGURATION_PROBLEM);
					}
				catch (IllegalArgumentException exception)
					{
					stdout.println
						("Unable to find the configuration file." + NL
						+ exception.getMessage ());
					Exit (EXIT_CONFIGURATION_PROBLEM);
					}
				break;

			case 'D':	//	Database
			case 'S':	//	Server
				if (++count == args.length)
					{
					stdout.println
						("Missing database server name.");
					Usage ();
					}
				if (database_server_name != null)
					{
					stdout.println
						("Multiple database servers specified -" + NL
						+"    " + database_server_name + " and " + args[count]);
					Usage ();
					}
				database_server_name = args[count];
				break;

			case 'M':	//	Managed/Monitored with GUI.
				managed = true;
				break;

			case 'W':	//	Wait for Management Start.
				wait_to_start = true;
				break;

			case 'V':	//	Version.
				stdout.println (ID);
				Exit (EXIT_SUCCESS);

			default:
				stdout.println
					("Unrecognized option: " + args[count]);

			case 'H':	//	Help
				Usage ();
			}
		}
	else
		{
		//	Pipeline.
		if (pipeline != null &&
			! pipeline.equals (args[count]))
			{
			stdout.println
				("Multiple pipelines specified -" + NL
				+"    " + pipeline + " and " + args[count]);
			Usage ();
			}
		pipeline = args[count];
		}
	}
if (pipeline == null)
	{
	stdout.println
		("No pipeline specified.");
	Usage ();
	}

if (catalog != null)
	{
	String
		name = Reference_Resolver.Catalog_Name (pipeline);
	if (name != null)
		{
		if (! name.equals (catalog))
			{
			stdout.println
				("Multiple database catalogs specified -" + NL
				+"    From the -catalog option: " + catalog + NL
				+"    From the pipeline name:   " + name);
			Exit (EXIT_COMMAND_LINE_SYNTAX);
			}
		}
	else
		pipeline = catalog + Reference_Resolver.COMPONENTS_DELIMITER + pipeline;
	}

if (configuration == null)
	{
	try {configuration = new Configuration ((String)null);}
	catch (Configuration_Exception exception)
		{
		stdout.println
			("Unable to construct the configuration for \""
			+ Configuration.Default_Source () + "\"." + NL
			+ exception.getMessage ());
		Exit (EXIT_CONFIGURATION_PROBLEM);
		}
	catch (IllegalArgumentException exception)
		{
		stdout.println
			("Unable to find the configuration file." + NL
			+ exception.getMessage ());
		Exit (EXIT_CONFIGURATION_PROBLEM);
		}
	}

//	Construct the Conductor.
Conductor
	conductor = null;
try {conductor = new Conductor (pipeline, configuration, database_server_name);}
catch (Exception exception)
	{
	Exit (new Status_Report
		(null,
		"Unable to construct the Conductor for pipeline "
			+ pipeline + " on " + Host.FULL_HOSTNAME + '.',
		exception));
	}

if (managed)
	{
	//	Give the Conductor to the GUI manager/monitor.
	new Manager (conductor, configuration);
	if (wait_to_start)
		//	Let the listener know that the Conductor is ready.
		stdout.println (">>> READY <<<");
	}
else
	{
	//	No local manager.
	stdout.println
		("Conductor 2.47 (2012/04/16 06:04:09) on "
			+ Conductor_ID);
	if ((DEBUG & DEBUG_SETUP) != 0)
		System.out.println
			("    Starting pipeline processing...");

	//	Semaphore thread to wait for pipeline processing to end.
	conductor.Main_Thread_Waiting = new Thread (new Runnable ()
		{public void run ()
		{try {synchronized (this) {wait ();}}
		 catch (InterruptedException except) {}}});
	conductor.Main_Thread_Waiting.start ();

	//	Let the listener know that the Conductor is ready.
	stdout.println (">>> READY <<<");

	if (! wait_to_start)
		conductor.Start ();

	//	Wait for pipeline processing to end.
	while (conductor.Main_Thread_Waiting.isAlive ())
		{
		try {conductor.Main_Thread_Waiting.join ();}
		catch (InterruptedException except) {}
		}

	Exit (new Status_Report
		(conductor, null, conductor.Processing_Exception ()));
	}
}


private static void Exit
	(
	Status_Report	status_report
	)
{
if ((DEBUG & DEBUG_EXIT) != 0)
	System.out.println
		(">>> Conductor.Exit");
int
	status = EXIT_SUCCESS;
if (status_report != null)
	{
	if (status_report.Message != null)
		stdout.println (status_report.Message);
	Send_Failure_Notification (status_report);

	if (status_report.The_Conductor != null)
		{
		//	Clean up.

		Processing_Changes
			processing_changes = new Processing_Changes ()
				.Exiting (true);
		if (status_report.Status != EXIT_SUCCESS &&
			status_report.Message != null)
			processing_changes.Error_Condition (status_report.Message);
		if ((DEBUG & DEBUG_EXIT) != 0)
			System.out.println
				("    Report_Processing_Event");
		status_report.The_Conductor
			.Report_Processing_Event (processing_changes);

		if ((DEBUG & DEBUG_EXIT) != 0)
			System.out.println
				("    Disconnect_from_Stage_Manager ...");
		status_report.The_Conductor.Disconnect_from_Stage_Manager ();
		}
	status = status_report.Status;
	}

//	Drain any pending output.
if ((DEBUG & DEBUG_EXIT) != 0)
	System.out.println
		("    Flush standard output");
System.out.flush ();
System.err.flush ();

if ((DEBUG & DEBUG_EXIT) != 0)
	System.out.println
		("    Exiting with status " + status);
System.exit (status);
}


private static void Exit
	(
	int		status
	)
{Exit (new Status_Report (status));}


private static void Exit ()
{Exit (null);}


private static void Send_Failure_Notification
	(
	Status_Report	status_report
	)
{
if ((DEBUG & DEBUG_NOTIFICATION) != 0)
	System.out.println
		(">>> Conductor.Send_Failure_Notification");
if (Notify_List != null &&
	status_report != null &&
	status_report.Status != EXIT_SUCCESS)
	{
	String
		message
			= "A failure condition has halted Conductor processing" + NL
			+ "of the " + Pipeline + " pipeline on " + Conductor_ID
			+ ((status_report.Message == null) ? "." :
				(" -" + NL
				+ NL
				+ status_report.Message));
	if ((DEBUG & DEBUG_NOTIFICATION) != 0)
		System.out.println
			("    Constructing Notify with Notify_List: " + Notify_List + NL
			+"    Message -" + NL
			+ message);
	Notify
		notifier = null;
	try {notifier = new Notify (Notify_List);}
	catch (NoClassDefFoundError error)
		{
		message = message.concat (NL + NL
			+ "!!! Email notification failed!" + NL
			+ "A class required by the Notify object could not be found.");
		if (error.getMessage () != null)
			message = message.concat (NL + error.getMessage ());
		if (status_report.The_Conductor != null)
			{
			try {status_report.The_Conductor
				.Log_Message (message, NOTICE_STYLE);}
			catch (IOException exception) {}
			}
		if ((DEBUG & DEBUG_NOTIFICATION) != 0)
			System.out.println (NL
				+ message + NL + NL
				+ "status_report.Th_Conductor != null? "
					+ (status_report.The_Conductor != null));
		}
	if (notifier != null)
		{
		if (notifier
			.Subject ("Conductor " + Pipeline + " pipeline halted")
			.Message (message)
			.Deliver ()
			!= Notify_List.size () &&
			status_report.The_Conductor != null)
			{
			message = message.concat (NL + NL
				+ "!!! Email notification problem!" + NL
				+ "Unable to deliver message to " + notifier.Undeliverable ());
			try {status_report.The_Conductor
				.Log_Message (message, NOTICE_STYLE);}
			catch (IOException exception) {}
			}
		}
	}
if ((DEBUG & DEBUG_NOTIFICATION) != 0)
	System.out.println
		("<<< Conductor.Send_Failure_Notification");
}

/**	Prints the command line usage syntax.
<p>
<blockquote><pre>
Usage: <b>Conductor</b> &lt;<i>Switches</i>&gt;
&nbsp;&nbsp;Switches -
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<U>P</U>ipeline</b>] &lt;<i>pipeline</i>&gt;
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<U>C</U>onfiguration</b> &lt;<i>source</i>&gt;]
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<U>D</U>atabase</b>|<b>-<U>S</U>erver</b> &lt;<i>server name</i>&gt;]
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<U>CA</U>talog</b> &lt;<i>catalog</i>&gt;]
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<U>M</U>onitor</b>]
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<U>W</U>ait-to-start</b>]
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<U>V</U>ersion</b>]
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<U>H</U>elp</b>]
</pre></blockquote>
<p>
	The <b>-<U>P</U>ipeline</b> switch specifies which pipeline the
	Conductor is to manage. A pipeline must be specified. A command line
	argument without a preceding switch name is assumed to be the
	pipeline name. The pipeline name has the form:
<p>
	<blockquote>
	[&lt;<i>catalog</i>&gt;<b>.</b>]&lt;<i>pipeline</i>&gt;
	</blockquote>
<p>
	where &lt;<i>pipeline</i>&gt; is the name of the pipeline and is
	used as the prefix of the Sources and Procedures table names:
<p>
	<blockquote>
	&lt;<i>pipeline</i>&gt;<b>_Sources</b><BR>
	&lt;<i>pipeline</i>&gt;<b>_Procedures</b>
	</blockquote>
<p>
	The <b>-<U>C</U>onfiguration</b> option is used to specify the
	filename, or URL (http or ftp), where the configuration parameters
	are to be found. If this option is not specified the
	"Conductor.conf" filename will be used. If the configuration file is
	not in the current working directory, it will be looked for in the
	user's home directory.
<p>
	The configuration file must contain the necessary database access
	information needed to identify and connect with the database server
	(as required by the {@link
	PIRL.Database.Database#Database(Configuration) Database} constructor
	and its {@link PIRL.Database.Database#Connect() Connect} method).
	The database "Type" parameter must be provided that specifies the
	type of database server (e.g. "MySQL") that will be accessed.
	Additional database access parameters typically provided are the
	server "Host" name and database "User" and "Password" access values.
	Depending on the type of database server and the driver and its
	Data_Port implementation (e.g. {@link
	PIRL.Database.MySQL_Data_Port}) there may be other required and
	optional parameters that can be included, such as a "Port" parameter
	to specify the database host system's network port to use for server
	communications.
<p>
	In addition to the "Catalog" parameter (described below), the
	"Log_Directory" parameter may optionally be located in the
	"/Conductor" group. The "Log_Directory" parameter specifies the
	directory where log files will be written. The value of this
	parameter may contain embedded references for the database {@link
	Reference_Resolver Reference_Resolver}. Other parameters that will be
	used if present in the Conductor configuration group are described in
	the Conductor control parameters section of the Conductor class
	description.
<p>
	The <b>-<U>S</U>erver</b> or <b>-<U>D</U>atabase</b> option may be
	used to specify the Group of database server access parameters in the
	configuration file. A configuration file may contain more than one
	Group of database server access parameters where the name of each
	such Group must be in the Server parameter list. By default the first
	name in the Server list is the Group of database access parameters
	that will be used.
<p>
	The <b>-<U>CA</U>talog</b> option may be used to specify the name of
	the database catalog containing the pipeline tables. However, it is
	ambiguous command line syntax to use this option and include a
	different &lt;<i>catalog</i>&gt; in the pipeline name. 
<p>
	The &lt;<i>catalog</i>&gt; name may alternatively be specified by a
	"Catalog" parameter in the configuration file. This parameter is first
	sought in the "/Conductor" parameters Group, then in the database server
	parameters Group or any parent of that group. It is necessary that a
	&lt;<i>catalog</i>&gt; name be found somewhere.
<p>
	The <b>-<U>M</U>anage</b> or <b>-<U>M</U>onitor</b> option may be
	used to run Conductor with a Manager GUI. In this mode Conductor will
	not proceed to process the pipeline immediately. Instead, the Manager
	will enable processing to be started, stopped, aborted, and the
	pipeline processing continuously monitored. By default the Conductor
	is run without a Manager.
<p>
	The <b>-<U>W</U>wait-to-start</b> option will cause the Conductor to
	remain in the wait state until it receives a message to start source
	record processing. If the Conductor is run with a Manager
	-wait-to-start is implicit. Remote management can be provided via a
	Stage_Manager and a Kapellmeister client. By default the Conductor
	will not wait-to-start unless it is run with a Manager.
<p>
	The <b>-<U>V</U>ersion</b> option will cause the Conductor version
	identification to be listed without running the Conductor.
<p>
	The <b>-<U>H</U>elp</b> option will list the brief command line
	syntax usage and then exit.
<p>
	<b>N.B.</b>: This method always results in a	System.exit with the
	{@link #EXIT_COMMAND_LINE_SYNTAX} status value.
*/
public static void Usage ()
{
System.out.println 
	("Usage: Conductor <Switches>" + NL
	+"  Switches -" + NL
	+"    [-Pipeline] <name>" + NL
	+"    [-Configuration <source>]" + NL
	+"      Default: Conductor.conf" + NL
	+"    [-Server <name>]" + NL
	+"      Default: First Server name in the Configuration" + NL
	+"    [-CAtalog <name>]" + NL
	+"      Default: The Catalog prefix of the Pipeline name" + NL
	+"        or Catalog name in the Configuration" + NL
	+"    [-Manager | Monitor]" + NL
	+"      Default: No Manager" + NL
	+"    [-Wait-to-start]" + NL
	+"      Default: Start immediately if no Manager" + NL
	+"    [-Version]" + NL
	+"    [-Help]" + NL
	);
System.exit (EXIT_COMMAND_LINE_SYNTAX);
}


}
