/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var isnan = require( './../../../../base/assert/is-nan' );
var evalrational = require( './../lib/evalrational.js' );


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof evalrational, 'function', 'main export is a function' );
	t.end();
});

tape( 'if provided two empty coefficient arrays, the function always returns `NaN`', function test( t ) {
	var v;
	var i;

	for ( i = 0; i < 100; i++ ) {
		v = evalrational( [], [], i );
		t.equal( isnan( v ), true, 'returns NaN' );
	}
	t.end();
});

tape( 'if provided only one coefficient for both arrays, the function always returns the ratio of the two coefficients', function test( t ) {
	var v;
	var i;

	for ( i = 0; i < 100; i++ ) {
		v = evalrational( [2.0], [4.0], i );
		t.equal( v, 0.5, 'returns coefficient ratio' );
	}
	t.end();
});

tape( 'if provided coefficient arrays of different lengths, the function always returns `NaN`', function test( t ) {
	var v;
	var i;

	for ( i = 0; i < 100; i++ ) {
		v = evalrational( [ 2.0, 1.0, 2.0 ], [ 3.0, 1.0 ], i );
		t.equal( isnan( v ), true, 'returns NaN' );
		v = evalrational( [ 0.5, 2.0 ], [ 2.0, 3.0, 1.0 ], i );
		t.equal( isnan( v ), true, 'returns NaN' );
	}
	t.end();
});

tape( 'if the value at which to evaluate a rational function is `0`, the function returns the ratio of the first coefficients', function test( t ) {
	var v;

	v = evalrational( [ 3.0, 2.0, 1.0 ], [ 0.5, 2.0, 1.0 ], 0.0 );
	t.equal( v, 6.0, 'returns coefficient ratio' );

	t.end();
});

tape( 'the function evaluates a rational function', function test( t ) {
	var P;
	var Q;
	var v;

	P = [ 2.0, 3.0, 2.0 ];
	Q = [ 1.0, 0.0, 0.0 ];
	v = evalrational( P, Q, 1.0 );
	t.equal( v, 7.0, 'returns 7' );

	P = [ -6.0, -5.0 ];
	Q = [ 3.0, 0.5 ];
	v = evalrational( P, Q, 6.0 );
	t.equal( v, -6.0, 'returns -6' );

	P = [ -19.0, 7.0, -4.0, 6.0 ];
	Q = [ 4.0, 5.0, 2.0, 1.0 ];
	v = evalrational( P, Q, 3.0 );
	t.equal( v, 2.0, 'returns 2' );

	P = [ 4.0, 2.0, 1.0 ];
	Q = [ 2.0, 0.0, 0.0 ];
	v = evalrational( P, Q, -2.0 );
	t.equal( v, 2.0, 'returns 2' );

	t.end();
});

tape( 'the function handles large `x` values', function test( t ) {
	var v1;
	var v2;
	var P;
	var Q;
	var x;
	var i;

	// 6x^5 + 5x^4 + 4x^3 + 3x^2 + 2x^1 + 1x^0
	P = [ 1.0, 2.0, 3.0, 4.0, 5.0, 6.0 ];

	// 1x^5 + 2x^4 + 3x^3 + 4x^2 + 5x^1 + 6x^0
	Q = [ 6.0, 5.0, 4.0, 3.0, 2.0, 1.0 ];

	x = 1.0e100;
	for ( i = 0; i < 1000; i++ ) {
		x *= 2.0;
		v1 = evalrational( P, Q, x );
		v2 = evalrational( P, Q, -x );
		t.equal( v1, v2, 'for x=+-'+x+', returns '+v1 );
	}
	t.end();
});

tape( 'the function returns analytically incorrect results for certain coefficient and `x` combinations', function test( t ) {
	var P;
	var Q;
	var v;

	// Case 1: large negative `x`

	// 1x^2 + 1e308x^1 + 0x^0
	P = [ 0.0, 1.0e308, 1.0 ];

	// 0x^2 + 0x^2 + 1x^0
	Q = [ 1.0, 0.0, 0.0 ];

	// (-1e308)^2 + (1e308)(-1e308) = +inf - inf => indeterminate => NaN
	v = evalrational( P, Q, -1.0e308 );
	t.notOk( isnan( v ), 'should be NaN' );

	// Case 2: large positive `x`

	// 1x^2 - 1e308x^1 + 0x^0
	P = [ 0.0, -1.0e308, 1.0 ];

	// 0x^2 + 0x^2 + 1x^0
	Q = [ 1.0, 0.0, 0.0 ];

	// (1e308)^2 - (1e308)(1e308) = +inf - inf => indeterminate => NaN
	v = evalrational( P, Q, 1.0e308 );
	t.notOk( isnan( v ), 'should be NaN' );

	t.end();
});
