/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*
* ## Notice
*
* The original C code, long comment, copyright, license, and constants are from [Cephes]{@link http://www.netlib.org/cephes}. The implementation follows the original, but has been modified for JavaScript.
*
* ```text
* Copyright 1984, 1995, 2000 by Stephen L. Moshier
*
* Some software in this archive may be from the book _Methods and Programs for Mathematical Functions_ (Prentice-Hall or Simon & Schuster International, 1989) or from the Cephes Mathematical Library, a commercial product. In either event, it is copyrighted by the author. What you see here may be used freely but it comes with no support or guarantee.
*
* Stephen L. Moshier
* moshier@na-net.ornl.gov
* ```
*/

'use strict';

// MODULES //

var abs = require( './../../../../base/special/abs' );
var exp = require( './../../../../base/special/exp' );
var ratval = require( './rational_pq.js' );


// VARIABLES //

// log(2**127)
var MAXLOG = 8.8029691931113054295988e+01;


// MAIN //

/**
* Computes the hyperbolic tangent of a number.
*
* ## Method
*
* For \\( |x| < 0.625 \\), we use a rational function of the form (Cody and Waite)
*
* ```tex
* x + x^3 \frac{\mathrm{P}(x)}{\mathrm{Q}(x)}
* ```
*
* Otherwise,
*
* ```tex
* \begin{align*}
* \operatorname{tanh}(x) &= \frac{\operatorname{sinh}(x)}{\operatorname{cosh(x)}} \\
* &= 1 - \frac{2}{e^{2x} + 1}
* \end{align*}
* ```
*
* ## Notes
*
* -   Relative error:
*
*     | arithmetic | domain | # trials | peak    | rms     |
*     |:----------:|:------:|:--------:|:-------:|:-------:|
*     | DEC        | -2,2   | 50000    | 3.3e-17 | 6.4e-18 |
*     | IEEE       | -2,2   | 30000    | 2.5e-16 | 5.8e-17 |
*
*
* @param {number} x - input value (in radians)
* @returns {number} hyperbolic tangent
*
* @example
* var v = tanh( 0.0 );
* // returns 0.0
*
* @example
* var v = tanh( 2.0 );
* // returns ~0.964
*
* @example
* var v = tanh( -2.0 );
* // returns ~-0.964
*
* @example
* var v = tanh( NaN );
* // returns NaN
*/
function tanh( x ) {
	var s;
	var z;
	z = abs( x );
	if ( z > 0.5*MAXLOG ) {
		return ( x < 0.0 ) ? -1.0 : 1.0;
	}
	if ( z >= 0.625 ) {
		s = exp( 2.0 * z );
		z = 1.0 - ( 2.0/(s+1.0) );
		if ( x < 0.0 ) {
			z = -z;
		}
	} else {
		if ( x === 0.0 ) {
			return x; // Handle `+-0`
		}
		s = x * x;
		z = x + ( x*s*ratval( s ) );
	}
	return z;
}


// EXPORTS //

module.exports = tanh;
