/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var isnan = require( './../../../../base/assert/is-nan' );
var PINF = require( '@stdlib/constants/float64/pinf' );
var NINF = require( '@stdlib/constants/float64/ninf' );
var EPS = require( '@stdlib/constants/float64/eps' );
var abs = require( './../../../../base/special/abs' );
var randu = require( '@stdlib/random/base/randu' );
var fresnel = require( './../lib' );


// FIXTURES //

var smallPositive = require( './fixtures/c/cephes/small.json' );
var mediumPositive = require( './fixtures/c/cephes/medium.json' );
var largePositive = require( './fixtures/c/cephes/large.json' );
var hugePositive = require( './fixtures/c/cephes/huge.json' );


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof fresnel, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function computes in-place the Fresnel integral (small positive values)', function test( t ) {
	var delta;
	var tol;
	var C;
	var S;
	var x;
	var y;
	var z;
	var i;

	z = [ 0.0, 0.0 ];
	x = smallPositive.x;
	S = smallPositive.S;
	C = smallPositive.C;

	for ( i = 0; i < x.length; i++ ) {
		y = fresnel( z, x[i] );
		t.equal( y, z, 'returns a reference to out' );
		if ( y[0] === S[i] ) {
			t.equal( y[0], S[i], 'x: '+x[i]+'. Expected: '+S[i] );
		} else {
			delta = abs( y[0] - S[i] );
			tol = 4.0 * EPS * abs( S[i] );
			t.ok( delta <= tol, 'within tolerance. x: '+x[i]+'. S: '+y[0]+'. Expected: '+S[i]+'. tol: '+tol+'. delta: '+delta+'.' );
		}
		if ( y[1] === C[i] ) {
			t.equal( y[1], C[i], 'x: '+x[i]+'. Expected: '+C[i] );
		} else {
			delta = abs( y[1] - C[i] );
			tol = 4.0 * EPS * abs( C[i] );
			t.ok( delta <= tol, 'within tolerance. x: '+x[i]+'. C: '+y[1]+'. Expected: '+C[i]+'. tol: '+tol+'. delta: '+delta+'.' );
		}
	}
	t.end();
});

tape( 'the function computes the Fresnel integral (small positive values)', function test( t ) {
	var delta;
	var tol;
	var C;
	var S;
	var x;
	var y;
	var i;

	x = smallPositive.x;
	S = smallPositive.S;
	C = smallPositive.C;

	for ( i = 0; i < x.length; i++ ) {
		y = fresnel( x[i] );
		if ( y[0] === S[i] ) {
			t.equal( y[0], S[i], 'x: '+x[i]+'. Expected: '+S[i] );
		} else {
			delta = abs( y[0] - S[i] );
			tol = 4.0 * EPS * abs( S[i] );
			t.ok( delta <= tol, 'within tolerance. x: '+x[i]+'. S: '+y[0]+'. Expected: '+S[i]+'. tol: '+tol+'. delta: '+delta+'.' );
		}
		if ( y[1] === C[i] ) {
			t.equal( y[1], C[i], 'x: '+x[i]+'. Expected: '+C[i] );
		} else {
			delta = abs( y[1] - C[i] );
			tol = 4.0 * EPS * abs( C[i] );
			t.ok( delta <= tol, 'within tolerance. x: '+x[i]+'. C: '+y[1]+'. Expected: '+C[i]+'. tol: '+tol+'. delta: '+delta+'.' );
		}
	}
	t.end();
});

tape( 'the function computes the Fresnel integral (medium positive values)', function test( t ) {
	var delta;
	var tol;
	var C;
	var S;
	var x;
	var y;
	var i;

	x = mediumPositive.x;
	S = mediumPositive.S;
	C = mediumPositive.C;

	for ( i = 0; i < x.length; i++ ) {
		y = fresnel( x[i] );
		if ( y[0] === S[i] ) {
			t.equal( y[0], S[i], 'x: '+x[i]+'. Expected: '+S[i] );
		} else {
			delta = abs( y[0] - S[i] );
			tol = 1.5 * EPS * abs( S[i] );
			t.ok( delta <= tol, 'within tolerance. x: '+x[i]+'. S: '+y[0]+'. Expected: '+S[i]+'. tol: '+tol+'. delta: '+delta+'.' );
		}
		if ( y[1] === C[i] ) {
			t.equal( y[1], C[i], 'x: '+x[i]+'. Expected: '+C[i] );
		} else {
			delta = abs( y[1] - C[i] );
			tol = 1.5 * EPS * abs( C[i] );
			t.ok( delta <= tol, 'within tolerance. x: '+x[i]+'. C: '+y[1]+'. Expected: '+C[i]+'. tol: '+tol+'. delta: '+delta+'.' );
		}
	}
	t.end();
});

tape( 'the function computes the Fresnel integral (large positive values)', function test( t ) {
	var delta;
	var tol;
	var C;
	var S;
	var x;
	var y;
	var i;

	x = largePositive.x;
	S = largePositive.S;
	C = largePositive.C;

	for ( i = 0; i < x.length; i++ ) {
		y = fresnel( x[i] );
		if ( y[0] === S[i] ) {
			t.equal( y[0], S[i], 'x: '+x[i]+'. Expected: '+S[i] );
		} else {
			delta = abs( y[0] - S[i] );
			tol = 1.5 * EPS * abs( S[i] );
			t.ok( delta <= tol, 'within tolerance. x: '+x[i]+'. S: '+y[0]+'. Expected: '+S[i]+'. tol: '+tol+'. delta: '+delta+'.' );
		}
		if ( y[1] === C[i] ) {
			t.equal( y[1], C[i], 'x: '+x[i]+'. Expected: '+C[i] );
		} else {
			delta = abs( y[1] - C[i] );
			tol = 1.5 * EPS * abs( C[i] );
			t.ok( delta <= tol, 'within tolerance. x: '+x[i]+'. C: '+y[1]+'. Expected: '+C[i]+'. tol: '+tol+'. delta: '+delta+'.' );
		}
	}
	t.end();
});

tape( 'the function computes the Fresnel integral (very large positive values)', function test( t ) {
	var delta;
	var tol;
	var C;
	var S;
	var x;
	var y;
	var i;

	x = hugePositive.x;
	S = hugePositive.S;
	C = hugePositive.C;

	for ( i = 0; i < x.length; i++ ) {
		y = fresnel( x[i] );
		if ( y[0] === S[i] ) {
			t.equal( y[0], S[i], 'x: '+x[i]+'. Expected: '+S[i] );
		} else {
			delta = abs( y[0] - S[i] );
			tol = 1.5 * EPS * abs( S[i] );
			t.ok( delta <= tol, 'within tolerance. x: '+x[i]+'. S: '+y[0]+'. Expected: '+S[i]+'. tol: '+tol+'. delta: '+delta+'.' );
		}
		if ( y[1] === C[i] ) {
			t.equal( y[1], C[i], 'x: '+x[i]+'. Expected: '+C[i] );
		} else {
			delta = abs( y[1] - C[i] );
			tol = 1.5 * EPS * abs( C[i] );
			t.ok( delta <= tol, 'within tolerance. x: '+x[i]+'. C: '+y[1]+'. Expected: '+C[i]+'. tol: '+tol+'. delta: '+delta+'.' );
		}
	}
	t.end();
});

tape( 'the function is an odd function of x', function test( t ) {
	var yn;
	var yp;
	var x;
	var i;

	for ( i = 0; i < 500; i++ ) {
		x = randu() * 10000.0;
		yn = fresnel( -x );
		yp = fresnel( x );

		t.equal( yp[0], -yn[0], 'S(x) = -S(-x)' );
		t.equal( yp[1], -yn[1], 'C(x) = -C(-x)' );
	}
	t.end();
});

tape( 'the function returns `[0.5, 0.5]` if provided `+infinity`', function test( t ) {
	var v = fresnel( PINF );
	t.equal( v[0], 0.5, 'returns 0.5' );
	t.equal( v[1], 0.5, 'returns 0.5' );
	t.end();
});

tape( 'the function returns `[-0.5, -0.5]` if provided `-infinity`', function test( t ) {
	var v = fresnel( NINF );
	t.equal( v[0], -0.5, 'returns -0.5' );
	t.equal( v[1], -0.5, 'returns -0.5' );
	t.end();
});

tape( 'the function returns `[NaN, NaN]` if provided a `NaN`', function test( t ) {
	var v = fresnel( NaN );
	t.equal( isnan( v[0] ), true, 'returns NaN' );
	t.equal( isnan( v[1] ), true, 'returns NaN' );
	t.end();
});
