# -*- coding: utf-8 -*-
import sys
from textwrap import dedent

import pytest

import manimpango


def pytest_report_header(config):
    return (
        f"ManimPango version {manimpango.__version__}\n"
        f"Pango version {manimpango.pango_version()}\n"
        f"Cairo version {manimpango.cairo_version()}"
    )


@pytest.fixture
def setup_fontconfig(tmpdir, monkeypatch):
    if sys.platform.startswith("win32") and "GCC" not in sys.version:
        default_fontconfig_configuration = dedent(
            """\
        <?xml version="1.0"?>
        <!DOCTYPE fontconfig SYSTEM "urn:fontconfig:fonts.dtd">
        <!-- /etc/fonts/fonts.conf file to configure system font access -->
        <fontconfig>
            <description>Default configuration file</description>

        <!--
            DO NOT EDIT THIS FILE.
            IT WILL BE REPLACED WHEN FONTCONFIG IS UPDATED.
            LOCAL CHANGES BELONG IN 'local.conf'.

            The intent of this standard configuration file is to be adequate for
            most environments.  If you have a reasonably normal environment and
            have found problems with this configuration, they are probably
            things that others will also want fixed.  Please submit any
            problems to the fontconfig bugzilla system located at fontconfig.org

            Note that the normal 'make install' procedure for fontconfig is to
            replace any existing fonts.conf file with the new version.  Place
            any local customizations in local.conf which this file references.

            Keith Packard
        -->

        <!-- Font directory list -->

            <dir>WINDOWSFONTDIR</dir>

            <dir prefix="xdg">fonts</dir>
            <!-- the following element will be removed in the future -->
            <dir>~/.fonts</dir>

        <!--
        Accept deprecated 'mono' alias, replacing it with 'monospace'
        -->
            <match target="pattern">
                <test qual="any" name="family">
                    <string>mono</string>
                </test>
                <edit name="family" mode="assign" binding="same">
                    <string>monospace</string>
                </edit>
            </match>

        <!--
        Accept alternate 'sans serif' spelling, replacing it with 'sans-serif'
        -->
            <match target="pattern">
                <test qual="any" name="family">
                    <string>sans serif</string>
                </test>
                <edit name="family" mode="assign" binding="same">
                    <string>sans-serif</string>
                </edit>
            </match>

        <!--
        Accept deprecated 'sans' alias, replacing it with 'sans-serif'
        -->
            <match target="pattern">
                <test qual="any" name="family">
                    <string>sans</string>
                </test>
                <edit name="family" mode="assign" binding="same">
                    <string>sans-serif</string>
                </edit>
            </match>
        <!--
        Accept alternate 'system ui' spelling, replacing it with 'system-ui'
        -->
            <match target="pattern">
                <test qual="any" name="family">
                    <string>system ui</string>
                </test>
                <edit name="family" mode="assign" binding="same">
                    <string>system-ui</string>
                </edit>
            </match>

        <!--
        Load local system customization file
        -->
            <include ignore_missing="yes">conf.d</include>

        <!-- Font cache directory list -->

            <cachedir>LOCAL_APPDATA_FONTCONFIG_CACHE</cachedir>
            <cachedir prefix="xdg">fontconfig</cachedir>
            <!-- the following element will be removed in the future -->
            <cachedir>~/.fontconfig</cachedir>

            <config>
        <!--
        Rescan configuration every 30 seconds when FcFontSetList is called
        -->
                <rescan>
                    <int>30</int>
                </rescan>
            </config>

        </fontconfig>
        """
        )
        with open(tmpdir / "fonts.conf", "w") as f:
            f.write(default_fontconfig_configuration)
        monkeypatch.setenv("FONTCONFIG_PATH", str(tmpdir))
    monkeypatch.setenv("PANGOCAIRO_BACKEND", "fc")
