// license:BSD-3-Clause
// copyright-holders:AJR
/***************************************************************************

    Cherry MX 1500 101/102-key keyboards

    These AT and PS/2-compatible keyboards have a built-in smartcard reader
    programmed to read medical cards. The electrical interface is nominally
    ISO 7816-2 compliant, but none of the three supported synchronous
    protocols adheres to the ISO 7816-3 standard. (One of these formats
    appears to be simply a 24C04 I²C EEPROM.) The three LEDs at the top
    right of the keyboard (Error, Data, Power) are related to the smartcard
    reader, as are several undocumented keyboard commands. All this
    additional functionality is currently unemulated.

***************************************************************************/

#include "emu.h"
#include "cherry_mx1500.h"

#include "cpu/mcs51/i8052.h"


// device type definition
DEFINE_DEVICE_TYPE(CHERRY_G80_1500, cherry_g80_1500_device, "g80_1500", "Cherry G80-1500 Multi-Function Keyboard")

cherry_g80_1500_device::cherry_g80_1500_device(const machine_config &mconfig, const char *tag, device_t *owner, u32 clock)
	: device_t(mconfig, CHERRY_G80_1500, tag, owner, clock)
	, device_pc_kbd_interface(mconfig, *this)
	, m_mcu(*this, "mcu")
	, m_keys(*this, "ROW%u", 0U)
	, m_leds(*this, "led%u", 0U)
	, m_p1(0xff)
{
}

void cherry_g80_1500_device::device_start()
{
	m_leds.resolve();

	set_pc_kbdc_device();

	save_item(NAME(m_p1));
}


static INPUT_PORTS_START(g80_1500)
	PORT_START("CONFIG")
	PORT_CONFNAME(3, 0, "Layout")
	PORT_CONFSETTING(0, "US-International (G80-1500 HAU)")
	PORT_CONFSETTING(1, "German (G80-1500 HAD)")
	PORT_CONFSETTING(2, "French (G80-1500 HAF)")
	//PORT_CONFSETTING(3, "UK-English (G80-1500 HAG)")

	PORT_START("ROW0")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(F7)) PORT_CODE(KEYCODE_F7)
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(ASTERISK)) PORT_CODE(KEYCODE_ASTERISK)
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(PLUS_PAD)) PORT_CODE(KEYCODE_PLUS_PAD)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(',') PORT_CHAR('<') PORT_CODE(KEYCODE_COMMA) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(',') PORT_CHAR(';') PORT_CODE(KEYCODE_COMMA) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(';') PORT_CHAR('.') PORT_CODE(KEYCODE_COMMA) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("L") PORT_CHAR('l') PORT_CHAR('L') PORT_CODE(KEYCODE_L)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("O") PORT_CHAR('o') PORT_CHAR('O') PORT_CODE(KEYCODE_O)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('9') PORT_CHAR('(') PORT_CODE(KEYCODE_9) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('9') PORT_CHAR(')') PORT_CHAR(']') PORT_CODE(KEYCODE_9) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(U'ç') PORT_CHAR('9') PORT_CHAR('^') PORT_CODE(KEYCODE_9) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(ENTER_PAD)) PORT_CODE(KEYCODE_ENTER_PAD)

	PORT_START("ROW1")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(F6)) PORT_CODE(KEYCODE_F6)
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(MINUS_PAD)) PORT_CODE(KEYCODE_MINUS_PAD)
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(3_PAD)) PORT_CODE(KEYCODE_3_PAD)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("M") PORT_CHAR('m') PORT_CHAR('M') PORT_CODE(KEYCODE_M) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME(u8"M  μ") PORT_CHAR('m') PORT_CHAR('M') PORT_CHAR(U'μ') PORT_CODE(KEYCODE_M) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(',') PORT_CHAR('?') PORT_CODE(KEYCODE_M) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("K") PORT_CHAR('k') PORT_CHAR('K') PORT_CODE(KEYCODE_K)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("I") PORT_CHAR('i') PORT_CHAR('I') PORT_CODE(KEYCODE_I)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('8') PORT_CHAR('*') PORT_CODE(KEYCODE_8) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('8') PORT_CHAR('(') PORT_CHAR('[') PORT_CODE(KEYCODE_8) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('_') PORT_CHAR('8') PORT_CHAR('\\') PORT_CODE(KEYCODE_8) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(DEL_PAD)) PORT_CODE(KEYCODE_DEL_PAD)

	PORT_START("ROW2")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(F10)) PORT_CODE(KEYCODE_F10)
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(PGUP)) PORT_CODE(KEYCODE_PGUP)
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(PGDN)) PORT_CODE(KEYCODE_PGDN)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(RSHIFT)) PORT_CODE(KEYCODE_RSHIFT)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_UNUSED) PORT_CONDITION("CONFIG", 3, EQUALS, 0) // INT 2 key
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('#') PORT_CHAR('\'') PORT_CODE(KEYCODE_BACKSLASH) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('*') PORT_CHAR(U'μ') PORT_CODE(KEYCODE_BACKSLASH) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(']') PORT_CHAR('}') PORT_CODE(KEYCODE_CLOSEBRACE) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('+') PORT_CHAR('*') PORT_CHAR('~') PORT_CODE(KEYCODE_CLOSEBRACE) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('$') PORT_CHAR(U'£') PORT_CHAR(U'¤') PORT_CODE(KEYCODE_CLOSEBRACE) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('=') PORT_CHAR('+') PORT_CODE(KEYCODE_EQUALS) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(U'´') PORT_CHAR('`') PORT_CODE(KEYCODE_EQUALS) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('=') PORT_CHAR('+') PORT_CHAR('}') PORT_CODE(KEYCODE_EQUALS) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(DOWN)) PORT_CODE(KEYCODE_DOWN)

	PORT_START("ROW3")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(F9)) PORT_CODE(KEYCODE_F9)
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(HOME)) PORT_CODE(KEYCODE_HOME)
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(END)) PORT_CODE(KEYCODE_END)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('/') PORT_CHAR('?') PORT_CODE(KEYCODE_SLASH) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('-') PORT_CHAR('_') PORT_CODE(KEYCODE_SLASH) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('!') PORT_CHAR(U'§') PORT_CODE(KEYCODE_SLASH) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('\'') PORT_CHAR('"') PORT_CODE(KEYCODE_QUOTE) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(U'Ä',U'ä') PORT_CODE(KEYCODE_QUOTE) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(U'ù') PORT_CHAR('%') PORT_CODE(KEYCODE_QUOTE) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('[') PORT_CHAR('{') PORT_CODE(KEYCODE_OPENBRACE) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(U'Ü',U'ü') PORT_CODE(KEYCODE_OPENBRACE) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME(u8"^  ¨") PORT_CODE(KEYCODE_OPENBRACE) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('-') PORT_CHAR('_') PORT_CODE(KEYCODE_MINUS) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(U'ß') PORT_CHAR('?') PORT_CHAR('\\') PORT_CODE(KEYCODE_MINUS) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(')') PORT_CHAR(U'°') PORT_CHAR(']') PORT_CODE(KEYCODE_MINUS) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(LEFT)) PORT_CODE(KEYCODE_LEFT)

	PORT_START("ROW4")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(F8)) PORT_CODE(KEYCODE_F8)
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(SLASH_PAD)) PORT_CODE(KEYCODE_SLASH_PAD)
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(1_PAD)) PORT_CODE(KEYCODE_1_PAD)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('.') PORT_CHAR('>') PORT_CODE(KEYCODE_STOP) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('.') PORT_CHAR(':') PORT_CODE(KEYCODE_STOP) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(':') PORT_CHAR('/') PORT_CODE(KEYCODE_STOP) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(';') PORT_CHAR(':') PORT_CODE(KEYCODE_COLON) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME(u8"Ö") PORT_CHAR(U'ö') PORT_CHAR(U'Ö') PORT_CODE(KEYCODE_COLON) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("M") PORT_CHAR('m') PORT_CHAR('M') PORT_CODE(KEYCODE_COLON) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("P") PORT_CHAR('p') PORT_CHAR('P') PORT_CODE(KEYCODE_P)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('0') PORT_CHAR(')') PORT_CODE(KEYCODE_0) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('0') PORT_CHAR('=') PORT_CHAR('}') PORT_CODE(KEYCODE_0) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(U'à') PORT_CHAR('0') PORT_CHAR('@') PORT_CODE(KEYCODE_0) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(RCONTROL)) PORT_CODE(KEYCODE_RCONTROL)

	PORT_START("ROW5")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(F5)) PORT_CODE(KEYCODE_F5)
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(9_PAD)) PORT_CODE(KEYCODE_9_PAD)
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(6_PAD)) PORT_CODE(KEYCODE_6_PAD)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("N") PORT_CHAR('n') PORT_CHAR('N') PORT_CODE(KEYCODE_N)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("J") PORT_CHAR('j') PORT_CHAR('J') PORT_CODE(KEYCODE_J)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("U") PORT_CHAR('u') PORT_CHAR('U') PORT_CODE(KEYCODE_U)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('7') PORT_CHAR('&') PORT_CODE(KEYCODE_7) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('7') PORT_CHAR('/') PORT_CHAR('{') PORT_CODE(KEYCODE_7) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(U'è') PORT_CHAR('7') PORT_CHAR('`') PORT_CODE(KEYCODE_7) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Esc") PORT_CHAR(0x1b) PORT_CODE(KEYCODE_ESC)

	PORT_START("ROW6")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(F2)) PORT_CODE(KEYCODE_F2)
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(7_PAD)) PORT_CODE(KEYCODE_7_PAD)
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(UP)) PORT_CODE(KEYCODE_UP)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("C") PORT_CHAR('c') PORT_CHAR('C') PORT_CODE(KEYCODE_C)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("F") PORT_CHAR('f') PORT_CHAR('F') PORT_CODE(KEYCODE_F)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("R") PORT_CHAR('r') PORT_CHAR('R') PORT_CODE(KEYCODE_R)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('4') PORT_CHAR('$') PORT_CODE(KEYCODE_4) PORT_CONDITION("CONFIG", 3, NOTEQUALS, 2)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('\'') PORT_CHAR('4') PORT_CHAR('{') PORT_CODE(KEYCODE_4) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(F1)) PORT_CODE(KEYCODE_F1)

	PORT_START("ROW7")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(PRTSCR)) PORT_CODE(KEYCODE_PRTSCR)
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(8_PAD)) PORT_CODE(KEYCODE_8_PAD)
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(2_PAD)) PORT_CODE(KEYCODE_2_PAD)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("X") PORT_CHAR('x') PORT_CHAR('X') PORT_CODE(KEYCODE_X)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("D") PORT_CHAR('d') PORT_CHAR('D') PORT_CODE(KEYCODE_D)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("E") PORT_CHAR('e') PORT_CHAR('E') PORT_CODE(KEYCODE_E)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('3') PORT_CHAR('#') PORT_CODE(KEYCODE_3) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('3') PORT_CHAR(U'§') PORT_CHAR(U'³') PORT_CODE(KEYCODE_3) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('"') PORT_CHAR('3') PORT_CHAR('#') PORT_CODE(KEYCODE_3) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(0_PAD)) PORT_CODE(KEYCODE_0_PAD)

	PORT_START("ROW8")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(SCRLOCK)) PORT_CODE(KEYCODE_SCRLOCK)
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(PAUSE)) PORT_CODE(KEYCODE_PAUSE)
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(5_PAD)) PORT_CODE(KEYCODE_5_PAD)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Z") PORT_CHAR('z') PORT_CHAR('Z') PORT_CODE(KEYCODE_Z) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Y") PORT_CHAR('y') PORT_CHAR('Y') PORT_CODE(KEYCODE_Z) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("W") PORT_CHAR('w') PORT_CHAR('W') PORT_CODE(KEYCODE_Z) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("S") PORT_CHAR('s') PORT_CHAR('S') PORT_CODE(KEYCODE_S)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("W") PORT_CHAR('w') PORT_CHAR('W') PORT_CODE(KEYCODE_W) PORT_CONDITION("CONFIG", 3, NOTEQUALS, 2)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Z") PORT_CHAR('z') PORT_CHAR('Z') PORT_CODE(KEYCODE_W) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('2') PORT_CHAR('@') PORT_CODE(KEYCODE_2) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('2') PORT_CHAR('"') PORT_CHAR(U'²') PORT_CODE(KEYCODE_2) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(U'é') PORT_CHAR('2') PORT_CHAR('~') PORT_CODE(KEYCODE_2) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(RIGHT)) PORT_CODE(KEYCODE_RIGHT)

	PORT_START("ROW9")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(F12)) PORT_CODE(KEYCODE_F12)
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(INSERT)) PORT_CODE(KEYCODE_INSERT)
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(DEL)) PORT_CODE(KEYCODE_DEL)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_UNUSED) PORT_CONDITION("CONFIG", 3, EQUALS, 0) // INT 1 key
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('<') PORT_CHAR('>') PORT_CHAR('|') PORT_CODE(KEYCODE_BACKSLASH2) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('<') PORT_CHAR('>') PORT_CODE(KEYCODE_BACKSLASH2) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("A") PORT_CHAR('a') PORT_CHAR('A') PORT_CODE(KEYCODE_A) PORT_CONDITION("CONFIG", 3, NOTEQUALS, 2)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Q") PORT_CHAR('q') PORT_CHAR('Q') PORT_CODE(KEYCODE_A) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Q") PORT_CHAR('q') PORT_CHAR('Q') PORT_CODE(KEYCODE_Q) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Q  @") PORT_CHAR('q') PORT_CHAR('Q') PORT_CHAR('@') PORT_CODE(KEYCODE_Q) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("A") PORT_CHAR('a') PORT_CHAR('A') PORT_CODE(KEYCODE_Q) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('1') PORT_CHAR('!') PORT_CODE(KEYCODE_1) PORT_CONDITION("CONFIG", 3, NOTEQUALS, 2)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('&') PORT_CHAR('1') PORT_CODE(KEYCODE_1) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(LCONTROL)) PORT_CODE(KEYCODE_LCONTROL)

	PORT_START("ROW10")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(F11)) PORT_CODE(KEYCODE_F11)
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(0x08) PORT_CODE(KEYCODE_BACKSPACE)
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(0x0d) PORT_CODE(KEYCODE_ENTER)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_SHIFT_1) PORT_CODE(KEYCODE_LSHIFT)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(CAPSLOCK)) PORT_CODE(KEYCODE_CAPSLOCK)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(0x09) PORT_CODE(KEYCODE_TAB)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('`') PORT_CHAR('~') PORT_CODE(KEYCODE_TILDE) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('^') PORT_CHAR(U'°') PORT_CODE(KEYCODE_TILDE) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(U'²') PORT_CODE(KEYCODE_TILDE) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(' ') PORT_CODE(KEYCODE_SPACE)

	PORT_START("ROW11")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(F3)) PORT_CODE(KEYCODE_F3)
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(NUMLOCK)) PORT_CODE(KEYCODE_NUMLOCK)
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(4_PAD)) PORT_CODE(KEYCODE_4_PAD)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("V") PORT_CHAR('v') PORT_CHAR('V') PORT_CODE(KEYCODE_V)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("G") PORT_CHAR('g') PORT_CHAR('G') PORT_CODE(KEYCODE_G)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("T") PORT_CHAR('t') PORT_CHAR('T') PORT_CODE(KEYCODE_T)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('5') PORT_CHAR('%') PORT_CODE(KEYCODE_5) PORT_CONDITION("CONFIG", 3, NOTEQUALS, 2)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('(') PORT_CHAR('5') PORT_CHAR('[') PORT_CODE(KEYCODE_5) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(RALT)) PORT_CODE(KEYCODE_RALT) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Alt Gr") PORT_CHAR(UCHAR_SHIFT_2) PORT_CODE(KEYCODE_RALT) PORT_CONDITION("CONFIG", 3, NOTEQUALS, 0)

	PORT_START("ROW12")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(F4)) PORT_CODE(KEYCODE_F4)
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('\\') PORT_CHAR('|') PORT_CODE(KEYCODE_BACKSLASH) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_UNUSED) PORT_CONDITION("CONFIG", 3, NOTEQUALS, 0)
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("B") PORT_CHAR('b') PORT_CHAR('B') PORT_CODE(KEYCODE_B)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_UNKNOWN) // smartcard-related
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("H") PORT_CHAR('h') PORT_CHAR('H') PORT_CODE(KEYCODE_H)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Y") PORT_CHAR('y') PORT_CHAR('Y') PORT_CODE(KEYCODE_Y) PORT_CONDITION("CONFIG", 3, NOTEQUALS, 1)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Z") PORT_CHAR('z') PORT_CHAR('Z') PORT_CODE(KEYCODE_Y) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('6') PORT_CHAR('^') PORT_CODE(KEYCODE_6) PORT_CONDITION("CONFIG", 3, EQUALS, 0)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('6') PORT_CHAR('&') PORT_CODE(KEYCODE_6) PORT_CONDITION("CONFIG", 3, EQUALS, 1)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('-') PORT_CHAR('6') PORT_CHAR('|') PORT_CODE(KEYCODE_6) PORT_CONDITION("CONFIG", 3, EQUALS, 2)
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(UCHAR_MAMEKEY(LALT)) PORT_CODE(KEYCODE_LALT)
INPUT_PORTS_END

ioport_constructor cherry_g80_1500_device::device_input_ports() const
{
	return INPUT_PORTS_NAME(g80_1500);
}


void cherry_g80_1500_device::data_write(int state)
{
	m_mcu->set_input_line(MCS51_INT1_LINE, state ? CLEAR_LINE : ASSERT_LINE);
}

void cherry_g80_1500_device::mcu_p1_w(u8 data)
{
	// P1.7 is clock output to smartcard (TODO)
	m_p1 = data;
}

u8 cherry_g80_1500_device::mcu_p3_r()
{
	// P3.0 (RXD) must be high to enable keycode output
	// P3.4 (T0) is smartcard-related (TODO)
	return m_pc_kbdc->clock_signal() ? 0xff : 0xfe;
}

void cherry_g80_1500_device::mcu_p3_w(u8 data)
{
	// P3.1 (TXD) is serial data output to PC
	m_pc_kbdc->data_write_from_kb(BIT(data, 1));

	// P3.2 (INT0) is clock output to PC
	m_pc_kbdc->clock_write_from_kb(BIT(data, 2));

	// P3.5 (T1) is serial data output to smartcard (TODO)
}

u8 cherry_g80_1500_device::matrix_r(offs_t offset)
{
	u16 strobe = u16(m_p1) << 8 | offset >> 8;

	u8 data = 0xff;
	for (int i = 0; i < 13; i++)
	{
		if (BIT(strobe, 14))
			data &= m_keys[i]->read();

		// Skip over P1.1
		if (i == 4)
			strobe <<= 2;
		else
			strobe <<= 1;
	}

	return ~data;
}

void cherry_g80_1500_device::outputs_w(offs_t offset, u8 data)
{
	// Bit 0 = Scroll Lock key LED
	// Bit 1 = Caps Lock key LED
	// Bit 2 = Num Lock key LED
	for (int i = 0; i < 3; i++)
		m_leds[i] = BIT(data, i);

	// TODO: upper bits = other LEDs and smartcard-related signals
}


void cherry_g80_1500_device::prog_map(address_map &map)
{
	map(0x0000, 0x7fff).rom().region("eprom", 0);
}

void cherry_g80_1500_device::ext_map(address_map &map)
{
	map(0x0000, 0xffff).rw(FUNC(cherry_g80_1500_device::matrix_r), FUNC(cherry_g80_1500_device::outputs_w));
}

void cherry_g80_1500_device::device_add_mconfig(machine_config &config)
{
	i8032_device &mcu(I8032(config, m_mcu, 7'392'000)); // exact type and clock unknown (might actually use a more conventional 7.3728 MHz XTAL, as on G80-1000)
	mcu.set_addrmap(AS_PROGRAM, &cherry_g80_1500_device::prog_map);
	mcu.set_addrmap(AS_DATA, &cherry_g80_1500_device::ext_map);
	mcu.port_out_cb<1>().set(FUNC(cherry_g80_1500_device::mcu_p1_w));
	mcu.port_in_cb<3>().set(FUNC(cherry_g80_1500_device::mcu_p3_r));
	mcu.port_out_cb<3>().set(FUNC(cherry_g80_1500_device::mcu_p3_w));
}

ROM_START(g80_1500)
	ROM_REGION(0x8000, "eprom", 0)
	ROM_LOAD("nm27c256q_635-0945 index 05.bin", 0x0000, 0x8000, CRC(223714f4) SHA1(009fc939f86bf70d288523446e292095d9706ab2))
ROM_END

const tiny_rom_entry *cherry_g80_1500_device::device_rom_region() const
{
	return ROM_NAME(g80_1500);
}
