/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_utils.c
    \brief  the utilities for the LCMAPS
    \author Martijn Steenbakkers for the EU DataGrid.
*/

/*****************************************************************************
                            Include header files
******************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <stdarg.h>
#include <grp.h>
#include <pwd.h>
#include "lcmaps_defines.h"
#include "lcmaps_types.h"
#include "lcmaps_log.h"
#include "_lcmaps_utils.h"
#include "lcmaps_config.h"


/******************************************************************************
Function:       lcmaps_genfilename() (copied from GLOBUS gatekeeper.c)
Description:    generate an absolute file name given a starting prefix,
                a relative or absolute path, and a suffix
                Only use prefix if path is relative.
Parameters:
Returns:        a pointer to a string which could be freeded.
******************************************************************************/
/*!
    \fn lcmaps_genfilename(
        char * prefixp,
        char * pathp,
        char * suffixp
        )
    \brief Generate an absolute file name.
    
    Given a starting prefix, a relative or absolute path, and a suffix an
    absolute file name is generated. Uses the prefix only if
    the path is relative. (Copied (and modified) from GLOBUS gatekeeper.c)
    
    \param prefixp string containing the prefix to be prepended.
    \param pathp   relative/absolute path to file name.
    \param suffixp string containing the suffix to be appended.

    \return pointer to a string containing the absolute path to the file, which has to
    be freed.
*/
char * lcmaps_genfilename(
        char * prefixp,
        char * pathp,
        char * suffixp
)
{
    char * newfilename;
    int    prefixl, pathl, suffixl;
    char * prefix,  * path, * suffix;

    prefix = (prefixp) ? prefixp : "";
    path   = (pathp) ? pathp : "";
    suffix  = (suffixp) ? suffixp : "";

    prefixl = strlen(prefix);
    pathl   =  strlen(path);
    suffixl  =  strlen(suffix); 

    newfilename = (char *) calloc(1, (prefixl + pathl + suffixl + 3));
    if (newfilename) 
    {
        if (*path != '/')
        {
            strcat(newfilename, prefix);
            if ((prefixl != 0) && (prefix[prefixl-1] != '/'))
            {
                strcat(newfilename, "/");
            }
        }
        strcat(newfilename, path);
        if ((pathl  != 0) &&
            (suffixl != 0) && 
            (path[pathl-1] != '/') && 
             suffix[0] != '/')
        {
            strcat(newfilename, "/");
        }
        strcat(newfilename, suffix);
    }
    return newfilename;
}

/******************************************************************************
Function:   lcmaps_tokenize() (in modified form from globus_gatekeeper_utils.c)

Description:
    Breakup the command in to args, pointing the args array
    at the tokens. Replace white space at the end of each
    token with a null. A token maybe in quotes. 

Parameters:
    command: The command line to be parsed.
    args:    A pointer to an array of pointers to be filled it
    n:       Size of the array, on input, and set to size used on output. 
    sep:     string of seperating characters

Returns:
    0 on success. 
    -1 on to malloc
    -2 on to many args
    -3 on quote not matched
******************************************************************************/
/*!
    \fn lcmaps_tokenize(
        const char * command, 
        char ** args,
        int * n,
        char * sep
        )
    \brief Break the argument string up into tokens

    Breakup the command in to arguments, pointing the args array at the tokens.
    Replace white space at the end of each token with a null. A token maybe in quotes.
    (Copied (and modified) from GLOBUS gatekeeper.c)
    \param command the command line to be parsed
    \param args    pointer to an array of pointers to be filled
    \param n       size of the array, on input, and set to size used on output
    \param sep     string of separating characters
    \retval 0  succes
    \retval -1 malloc error
    \retval -2 too many args
    \retval -3 quote not matched
    \retval -4 invalid input
    \internal
*/
int lcmaps_tokenize(
        const char * command, 
        char ** args,
        int * n,
        char * sep
    )
{
    int maxargs;
    int i;
    const char * cp;
    const char * pp;
    const char * qp;
    char ** arg;

    if (args==NULL || n==NULL || sep==NULL)
	return -4;

    if (command==NULL)	{
	args[0] = NULL;
	*n = 0;
	return 0;
    }

    arg = args;
/*    i = *n - 1; */
    i = 0;
    maxargs = *n;

    cp = command;
    while (*cp)
    {
    /* skip leading sep characters */
        while (*cp && strchr(sep, *cp))
        {
            cp++;
        }
        pp = NULL;
        if (*cp == '\"')
        {
            cp++;
            pp = cp;
            if ((qp = strchr(cp,'\"')) == NULL)
            {
                *n = i;
                return -3;
            }
            cp = qp + 1;
        }
        else if (*cp)
        {
            pp = cp;
            if ((qp = strpbrk(cp,sep)) == NULL)
            {
                qp = strchr(cp,'\0');
            }
            cp = qp;
        }
        else
        {
            continue;
        }
        if (pp)
        {
            /*
             * fill at most maxargs-1 arguments; let the last one point to NULL
             */
            i++;
            if (i >= maxargs)
            {
                i--;
                *n = i;
                return(-2); /* too many args */
            }
            *arg = (char*)malloc((qp - pp) + 1);
            if (*arg == NULL)
            {
                i--;
                *n = i;
                return -1;
            }
            memcpy(*arg,pp,qp - pp);
            *(*arg + (qp - pp)) = '\0';
            arg++;
        }
    }
    *arg = NULL;
    *n = i;
    return(0);
}
/******************************************************************************
Function:   lcmaps_get_gidlist()

Description:
    Finds the list of gids for user in the group file (/etc/group)
    Returns a list of gid_t which should be freed by calling program.

Parameters:
    username:   the name of the user
    ngroups:    ptr to int which will be filled with the number of gids.
    group_list: ptr to an array of gid_t.

Returns:
    0 on success. 
    -1 on realloc failure
    -2 on getgrent failure
    1 on failure
******************************************************************************/
/*!
    \fn lcmaps_get_gidlist(
        const char * username,
        int * ngroups,
        gid_t ** group_list
    )
    \brief Finds the list of gids for user in the group file (/etc/group)

    Returns a list of gid_t which should be freed by calling program.
    \param username the name of the user
    \param ngroups ptr to int which will be filled with the number of gids
    \param group_list ptr to an array of gid_t
    \retval 0 success
    \retval -1 realloc failure
    \retval -2 getgrent failure
    \retval 1  failure
*/
int lcmaps_get_gidlist(
        const char * username,
        int * ngroups,
        gid_t ** group_list
    )
{
#ifdef HAVE_GETGROUPLIST
    /* New method of obtaining secondary Group IDs from the system. Using
     * getgrouplist() is faster, because it performs a smarter search for
     * secondary Group IDs */
    int ng = 0;
    gid_t *groups = NULL;

    if (!ngroups || !group_list) {
        return 1;
    }

    struct passwd *pw = getpwnam(username);
    if (pw == NULL)
        return 1;

    if (getgrouplist(username, pw->pw_gid, NULL, &ng) < 0) {
        groups = (gid_t *) malloc(ng * sizeof (gid_t));
        if (groups == NULL) {
            lcmaps_log(LOG_ERR,"lcmaps_get_gidlist(): cannot malloc\n");
            return -1;
        }
        getgrouplist(username, pw->pw_gid, groups, &ng);
    }

    *group_list = groups;
    *ngroups    = ng;
    return 0;
#else /* HAVE_GETGROUPLIST */
    /* This is the old way of doing it. It was reliable, but suffers an
     * increased load on LDAP based system due to the lookup of all the entries
     * in the database, each time you pass here */

    struct group        * group_info = NULL;
    gid_t               * groups = NULL;
    gid_t               * newgroups = NULL;
    int                   i = 0;

    /* rewind the file pointer to the beginning of the /etc/group file */
    setgrent();

    lcmaps_log_debug(2,"lcmaps_get_gidlist(): looping through group file\n");
    *ngroups = 0;
    while ( ( group_info = getgrent() ) )
    {
        char ** pgr_mem = group_info->gr_mem;
        char *  gr_mem = NULL;

        /* lcmaps_log_debug(4,"lcmaps_get_gidlist(): group %s\n", group_info->gr_name); */
        while ( (gr_mem = *pgr_mem) )
        {
            /* lcmaps_log_debug(4,"lcmaps_get_gidlist(): group member %s\n", gr_mem); */
            if (strncmp(username, gr_mem, strlen(username))==0)
            {
                /* lcmaps_log_debug(2,"lcmaps_get_gidlist(): found group %s for %s\n", group_info->gr_name, username); */

                (*ngroups)++;
                newgroups = (gid_t *) realloc(groups, ((*ngroups) * sizeof(gid_t)));
                if (newgroups == NULL)
                {
                    lcmaps_log(LOG_ERR,"lcmaps_get_gidlist(): cannot realloc\n");
                    free(groups);
                    return -1;
                }
                groups=newgroups;
                groups[(*ngroups)-1] = group_info->gr_gid;
            }
            ++pgr_mem;
        }
    }
    if (errno==ENOMEM)
    {
        lcmaps_log(LOG_ERR,"lcmaps_get_gidlist(): Cannot read the group file, %s\n", strerror(errno));
        free(groups);
        groups=NULL;
        /* Close the group file */
        endgrent();
        return -2;
    }
    *group_list=groups;
    lcmaps_log_debug(4,"lcmaps_get_gidlist(): %d groups found for %s\n", *ngroups, username);
    for (i = 0; i < *ngroups; i++)
    {
        lcmaps_log_debug(4,"lcmaps_get_gidlist(): group nr %d ==> gid_t %d\n", i+1, groups[i]);
    }
    /* Close the group file */
    endgrent();
    return 0;
#endif
}
/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/src/pluginmanager/lcmaps_utils.c,v $
    $Date: 2012-01-26 12:07:08 +0100 (Thu, 26 Jan 2012) $
    $Revision: 15876 $
    $Author: msalle $
******************************************************************************/
