import operator

from juju.errors import ProviderError
from collections import namedtuple

# Just a generic base instance type, with required attrs.
InstanceType = namedtuple("InstanceType", "arch cpu mem")


class TypeSolver(object):

    def __init__(self, instance_types):
        self._instance_types = instance_types

    def run(self, constraints):
        instance_type = constraints["instance-type"]
        if instance_type is not None:
            if not instance_type in self._instance_types:
                raise ProviderError(
                    "Invalid instance type %s" % instance_type)
            return instance_type

        return self._solve(constraints)

    def _solve(self, constraints):
        possible_types = list(self._instance_types)
        for f in self._get_filters(constraints):
            possible_types = filter(f, possible_types)

        if not possible_types:
            raise ProviderError(
                "No instance type satisfies %s" % dict(constraints))
        return sorted(
            possible_types,
            key=lambda x: (self._instance_types[x].cpu,
                           self._instance_types[x].mem))[0]

    def _get_filters(self, constraints):
        filters = [
            self._filter_func("cpu", constraints, operator.ge),
            self._filter_func("mem", constraints, operator.ge),
            self._filter_func("arch", constraints, operator.eq)
            ]
        return filters

    def _filter_func(self, name, constraints, op):

        desired = constraints[name]
        if not desired:
            return lambda _: True

        def f(type_):
            value = getattr(self._instance_types[type_], name)
            return op(value, desired)
        return f
