/*
 * Jalview - A Sequence Alignment Editor and Viewer (2.11.5.0)
 * Copyright (C) 2025 The Jalview Authors
 * 
 * This file is part of Jalview.
 * 
 * Jalview is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 *  
 * Jalview is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 * PURPOSE.  See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Jalview.  If not, see <http://www.gnu.org/licenses/>.
 * The Jalview Authors are detailed in the 'AUTHORS' file.
 */
package jalview.analysis.scoremodels;

import jalview.analysis.AlignmentAnnotationUtils;
import jalview.analysis.AlignmentUtils;
import jalview.api.AlignmentViewPanel;
import jalview.api.FeatureRenderer;
import jalview.api.analysis.ScoreModelI;
import jalview.api.analysis.SimilarityParamsI;
import jalview.datamodel.AlignmentAnnotation;
import jalview.datamodel.AlignmentView;
import jalview.datamodel.SeqCigar;
import jalview.datamodel.SequenceI;
import jalview.math.Matrix;
import jalview.math.MatrixI;
import jalview.util.Constants;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/* This class contains methods to calculate distance score between 
 * secondary structure annotations of the sequences. 
 */
public class SecondaryStructureDistanceModel extends DistanceScoreModel
{
  private static final String NAME = "Secondary Structure Similarity";

  private ScoreMatrix ssRateMatrix;

  private String description;

  FeatureRenderer fr;

  /**
   * Constructor
   */
  public SecondaryStructureDistanceModel()
  {

  }

  @Override
  public ScoreModelI getInstance(AlignmentViewPanel view)
  {
    SecondaryStructureDistanceModel instance;
    try
    {
      instance = this.getClass().getDeclaredConstructor().newInstance();
      instance.configureFromAlignmentView(view);
      return instance;
    } catch (InstantiationException | IllegalAccessException e)
    {
      jalview.bin.Console.errPrintln("Error in " + getClass().getName()
              + ".getInstance(): " + e.getMessage());
      return null;
    } catch (ReflectiveOperationException roe)
    {
      return null;
    }
  }

  boolean configureFromAlignmentView(AlignmentViewPanel view)

  {
    fr = view.cloneFeatureRenderer();
    return true;
  }

  ArrayList<AlignmentAnnotation> ssForSeqs = null;

  @Override
  public SequenceI[] expandSeqData(SequenceI[] sequences,
          AlignmentView seqData, SimilarityParamsI scoreParams,
          List<String> labels, ArrayList<AlignmentAnnotation> ssAnnotationForSeqs,
          HashMap<Integer, String> annotationDetails)
  {
    ssForSeqs = new ArrayList<AlignmentAnnotation>();
    List<SequenceI> newSequences = new ArrayList<SequenceI>();
    List<SeqCigar> newCigs = new ArrayList<SeqCigar>();
    int sq = 0;

    AlignmentAnnotation[] alignAnnotList = fr.getViewport().getAlignment()
            .getAlignmentAnnotation();

    String ssSource = scoreParams.getSecondaryStructureSource();
    if (ssSource == null || ssSource == "")
    {
      ssSource = Constants.SS_ALL_PROVIDERS;
    }

    /*
     * Add secondary structure annotations that are added to the annotation track
     * to the map
     */
    Map<SequenceI, ArrayList<AlignmentAnnotation>> ssAlignmentAnnotationForSequences = AlignmentUtils
            .getSequenceAssociatedAlignmentAnnotations(alignAnnotList,
                    ssSource);

    for (SeqCigar scig : seqData.getSequences())
    {
      // get the next sequence that should be bound to this scig: may be null
      SequenceI alSeq = sequences[sq++];
      List<AlignmentAnnotation> ssec = ssAlignmentAnnotationForSequences
              .get(scig.getRefSeq());
      if (ssec == null)
      {
        // not defined
        newSequences.add(alSeq);
        if (alSeq != null)
        {
          //labels.add("No Secondary Structure");
          labels.add(Constants.STRUCTURE_PROVIDERS.get("None"));
        }
        SeqCigar newSeqCigar = scig; // new SeqCigar(scig);
        newCigs.add(newSeqCigar);
        ssForSeqs.add(null);
      }
      else
      {
        for (int i = 0; i < ssec.size(); i++)
        {
          if (alSeq != null)
          {
        	// Add annotationDetails if the annotation has  
        	// ANNOTATION_DETAILS property value (additional metadata)
            
            if (ssec.get(i).hasAnnotationDetailsProperty())
            {
              // using key = labels.size() gives the position of the node
              annotationDetails.put(labels.size(), ssec.get(i).getAnnotationDetailsProperty());
            }
            
            String provider = AlignmentAnnotationUtils
                    .extractSSSourceFromAnnotationDescription(ssec.get(i));
            labels.add(provider);
          }
          newSequences.add(alSeq);
          SeqCigar newSeqCigar = scig; // new SeqCigar(scig);
          newCigs.add(newSeqCigar);
          ssForSeqs.add(ssec.get(i));
        }
      }
    }
    ssAnnotationForSeqs.addAll(ssForSeqs);
    seqData.setSequences(newCigs.toArray(new SeqCigar[0]));
    return newSequences.toArray(new SequenceI[0]);

  }

  /**
   * Calculates distance score [i][j] between each pair of protein sequences
   * based on their secondary structure annotations (H, E, C). The final score
   * is normalised by the number of alignment columns processed, providing an
   * average similarity score.
   * <p>
   * The parameters argument can include settings for handling gap-residue
   * aligned positions and may determine if the score calculation is based on
   * the longer or shorter sequence in each pair. This can be important for
   * handling partial alignments or sequences of significantly different
   * lengths.
   * 
   * @param seqData
   *          The aligned sequence data including secondary structure
   *          annotations.
   * @param params
   *          Additional parameters for customising the scoring process, such as
   *          gap handling and sequence length consideration.
   */
  @Override
  public MatrixI findDistances(AlignmentView seqData,
          SimilarityParamsI params)
  {
    if (ssForSeqs == null
            || ssForSeqs.size() != seqData.getSequences().length)
    {
      // expandSeqData needs to be called to initialise the hash
      SequenceI[] sequences = new SequenceI[seqData.getSequences().length];
      // we throw away the new labels in this case..
      expandSeqData(sequences, seqData, params, new ArrayList<String>(), 
              new ArrayList<AlignmentAnnotation>(), new HashMap<Integer, String>());
    }
    SeqCigar[] seqs = seqData.getSequences();
    int noseqs = seqs.length; // no of sequences
    int cpwidth = 0;
    double[][] similarities = new double[noseqs][noseqs]; // matrix to store
                                                          // similarity score
    // secondary structure source parameter selected by the user from the drop
    // down.
    String ssSource = params.getSecondaryStructureSource();
    if (ssSource == null || ssSource == "")
    {
      ssSource = Constants.SS_ALL_PROVIDERS;
    }
    ssRateMatrix = ScoreModels.getInstance().getSecondaryStructureMatrix();

    // need to get real position for view position
    int[] viscont = seqData.getVisibleContigs();

    /*
     * scan each column, compute and add to each similarity[i, j]
     * the number of secondary structure annotation that seqi 
     * and seqj do not share
     */
    for (int vc = 0; vc < viscont.length; vc += 2)
    {
      // Iterates for each column position
      for (int cpos = viscont[vc]; cpos <= viscont[vc + 1]; cpos++)
      {
        cpwidth++; // used to normalise the similarity score

        /*
         * get set of sequences without gap in the current column
         */
        Set<SeqCigar> seqsWithoutGapAtCol = findSeqsWithoutGapAtColumn(seqs,
                cpos);

        /*
         * calculate similarity score for each secondary structure annotation on i'th and j'th
         * sequence and add this measure to the similarities matrix 
         * for [i, j] for j > i
         */
        for (int i = 0; i < (noseqs - 1); i++)
        {
          AlignmentAnnotation aa_i = ssForSeqs.get(i);
          boolean undefinedSS1 = aa_i == null;
          // check if the sequence contains gap in the current column
          boolean gap1 = !seqsWithoutGapAtCol.contains(seqs[i]);
          // secondary structure is fetched only if the current column is not
          // gap for the sequence
          char ss1 = '*';
          if (!gap1 && !undefinedSS1)
          {
            // fetch the position in sequence for the column and finds the
            // corresponding secondary structure annotation
            // TO DO - consider based on priority and displayed
            int seqPosition_i = seqs[i].findPosition(cpos);
            if (aa_i != null)
              ss1 = AlignmentUtils.findSSAnnotationForGivenSeqposition(aa_i,
                      seqPosition_i);
          }
          // Iterates for each sequences
          for (int j = i + 1; j < noseqs; j++)
          {

            // check if ss is defined
            AlignmentAnnotation aa_j = ssForSeqs.get(j);
            boolean undefinedSS2 = aa_j == null;

            // Set similarity to max score if both SS are not defined
            if (undefinedSS1 && undefinedSS2)
            {
              similarities[i][j] += ssRateMatrix.getMaximumScore();
              continue;
            }

            // Set similarity to minimum score if either one SS is not defined
            else if (undefinedSS1 || undefinedSS2)
            {
              similarities[i][j] += ssRateMatrix.getMinimumScore();
              continue;
            }

            boolean gap2 = !seqsWithoutGapAtCol.contains(seqs[j]);

            // Variable to store secondary structure at the current column
            char ss2 = '*';

            if (!gap2 && !undefinedSS2)
            {
              int seqPosition = seqs[j].findPosition(cpos);

              if (aa_j != null)
                ss2 = AlignmentUtils.findSSAnnotationForGivenSeqposition(
                        aa_j, seqPosition);
            }

            if ((!gap1 && !gap2) || params.includeGaps())
            {
              // Calculate similarity score based on the substitution matrix
              double similarityScore = ssRateMatrix.getPairwiseScore(ss1,
                      ss2);
              similarities[i][j] += similarityScore;
            }
          }
        }
      }
    }

    /*
     * normalise the similarity scores (summed over columns) by the
     * number of visible columns used in the calculation
     * and fill in the bottom half of the matrix
     */
    // TODO JAL-2424 cpwidth may be out by 1 - affects scores but not tree shape

    for (int i = 0; i < noseqs; i++)
    {
      for (int j = i + 1; j < noseqs; j++)
      {
        similarities[i][j] /= cpwidth;
        similarities[j][i] = similarities[i][j];
      }
    }
    return SimilarityScoreModel
            .similarityToDistance(new Matrix(similarities));

  }

  /**
   * Builds and returns a set containing sequences (SeqCigar) which do not have
   * a gap at the given column position.
   * 
   * @param seqs
   * @param columnPosition
   *          (0..)
   * @return
   */
  private Set<SeqCigar> findSeqsWithoutGapAtColumn(SeqCigar[] seqs,
          int columnPosition)
  {
    Set<SeqCigar> seqsWithoutGapAtCol = new HashSet<>();
    for (SeqCigar seq : seqs)
    {
      int spos = seq.findPosition(columnPosition);
      if (spos != -1)
      {
        /*
         * position is not a gap
         */
        seqsWithoutGapAtCol.add(seq);
      }
    }
    return seqsWithoutGapAtCol;
  }

  @Override
  public String getName()
  {
    return NAME;
  }

  @Override
  public String getDescription()
  {
    return description;
  }

  @Override
  public boolean isDNA()
  {
    return false;
  }

  @Override
  public boolean isProtein()
  {
    return false;
  }

  @Override
  public boolean isSecondaryStructure()
  {
    return true;
  }

  @Override
  public String toString()
  {
    return "Score between sequences based on similarity between binary "
            + "vectors marking secondary structure displayed at each column";
  }
}