/* GSequencer - Advanced GTK Sequencer
 * Copyright (C) 2005-2024 Joël Krähemann
 *
 * This file is part of GSequencer.
 *
 * GSequencer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GSequencer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GSequencer.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ags/audio/ags_seq_synth_util.h>

#include <ags/audio/ags_audio_buffer_util.h>

#if defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
#include <Accelerate/Accelerate.h>
#endif

#include <math.h>
#include <complex.h>

/**
 * SECTION:ags_seq_synth_util
 * @short_description: frequency modulation synth util
 * @title: AgsSeqSynthUtil
 * @section_id:
 * @include: ags/audio/ags_seq_synth_util.h
 *
 * Utility functions to compute FM synths with sequencer and vibrato.
 */

GType
ags_seq_synth_util_get_type(void)
{
  static gsize g_define_type_id__static = 0;

  if(g_once_init_enter(&g_define_type_id__static)){
    GType ags_type_seq_synth_util = 0;

    ags_type_seq_synth_util =
      g_boxed_type_register_static("AgsSeqSynthUtil",
				   (GBoxedCopyFunc) ags_seq_synth_util_copy,
				   (GBoxedFreeFunc) ags_seq_synth_util_free);

    g_once_init_leave(&g_define_type_id__static, ags_type_seq_synth_util);
  }

  return(g_define_type_id__static);
}

/**
 * ags_seq_synth_util_alloc:
 *
 * Allocate #AgsSeqSynthUtil-struct
 *
 * Returns: a new #AgsSeqSynthUtil-struct
 *
 * Since: 7.5.0
 */
AgsSeqSynthUtil*
ags_seq_synth_util_alloc()
{
  AgsSeqSynthUtil *ptr;

  ptr = (AgsSeqSynthUtil *) g_new(AgsSeqSynthUtil,
				  1);

  ptr[0] = AGS_SEQ_SYNTH_UTIL_INITIALIZER;

  return(ptr);
}

/**
 * ags_seq_synth_util_copy:
 * @ptr: the original #AgsSeqSynthUtil-struct
 *
 * Create a copy of @ptr.
 *
 * Returns: a pointer of the new #AgsSeqSynthUtil-struct
 *
 * Since: 7.5.0
 */
gpointer
ags_seq_synth_util_copy(AgsSeqSynthUtil *ptr)
{
  AgsSeqSynthUtil *new_ptr;

  g_return_val_if_fail(ptr != NULL, NULL);

  new_ptr = (AgsSeqSynthUtil *) g_new(AgsSeqSynthUtil,
				      1);
  
  new_ptr->source = ptr->source;
  new_ptr->source_stride = ptr->source_stride;

  new_ptr->buffer_length = ptr->buffer_length;
  new_ptr->format = ptr->format;
  new_ptr->samplerate = ptr->samplerate;

  new_ptr->synth_oscillator_mode = ptr->synth_oscillator_mode;

  new_ptr->frequency = ptr->frequency;
  new_ptr->phase = ptr->phase;
  new_ptr->volume = ptr->volume;

  new_ptr->lfo_oscillator_mode = ptr->lfo_oscillator_mode;

  new_ptr->lfo_frequency = ptr->lfo_frequency;
  new_ptr->lfo_depth = ptr->lfo_depth;
  new_ptr->tuning = ptr->tuning;

  new_ptr->vibrato_enabled = ptr->vibrato_enabled;

  new_ptr->vibrato_gain = ptr->vibrato_gain;
  new_ptr->vibrato_lfo_depth = ptr->vibrato_lfo_depth;
  new_ptr->vibrato_lfo_freq = ptr->vibrato_lfo_freq;
  new_ptr->vibrato_tuning = ptr->vibrato_tuning;

  new_ptr->frame_count = ptr->frame_count;
  new_ptr->offset = ptr->offset;

  new_ptr->note_256th_mode = ptr->note_256th_mode;

  new_ptr->offset_256th = ptr->offset_256th;
  
  return(new_ptr);
}

/**
 * ags_seq_synth_util_free:
 * @ptr: the #AgsSeqSynthUtil-struct
 *
 * Free the memory of @ptr.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_free(AgsSeqSynthUtil *ptr)
{
  g_return_if_fail(ptr != NULL);
  
  g_free(ptr);
}

/**
 * ags_seq_synth_util_get_source:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get source buffer of @seq_synth_util.
 * 
 * Returns: the source buffer
 * 
 * Since: 7.5.0
 */
gpointer
ags_seq_synth_util_get_source(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(NULL);
  }

  return(seq_synth_util->source);
}

/**
 * ags_seq_synth_util_set_source:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @source: (transfer none): the source buffer
 *
 * Set @source buffer of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_source(AgsSeqSynthUtil *seq_synth_util,
			      gpointer source)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->source = source;
}

/**
 * ags_seq_synth_util_get_source_stride:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get source stride of @seq_synth_util.
 * 
 * Returns: the source buffer stride
 * 
 * Since: 7.5.0
 */
guint
ags_seq_synth_util_get_source_stride(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0);
  }

  return(seq_synth_util->source_stride);
}

/**
 * ags_seq_synth_util_set_source_stride:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @source_stride: the source buffer stride
 *
 * Set @source stride of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_source_stride(AgsSeqSynthUtil *seq_synth_util,
				     guint source_stride)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->source_stride = source_stride;
}

/**
 * ags_seq_synth_util_get_buffer_length:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get buffer length of @seq_synth_util.
 * 
 * Returns: the buffer length
 * 
 * Since: 7.5.0
 */
guint
ags_seq_synth_util_get_buffer_length(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0);
  }

  return(seq_synth_util->buffer_length);
}

/**
 * ags_seq_synth_util_set_buffer_length:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @buffer_length: the buffer length
 *
 * Set @buffer_length of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_buffer_length(AgsSeqSynthUtil *seq_synth_util,
				     guint buffer_length)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->buffer_length = buffer_length;
}

/**
 * ags_seq_synth_util_get_format:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get audio buffer util format of @seq_synth_util.
 * 
 * Returns: the audio buffer util format
 * 
 * Since: 7.5.0
 */
AgsSoundcardFormat
ags_seq_synth_util_get_format(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0);
  }

  return(seq_synth_util->format);
}

/**
 * ags_seq_synth_util_set_format:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @format: the audio buffer util format
 *
 * Set @format of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_format(AgsSeqSynthUtil *seq_synth_util,
			      AgsSoundcardFormat format)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->format = format;
}

/**
 * ags_seq_synth_util_get_samplerate:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get samplerate of @seq_synth_util.
 * 
 * Returns: the samplerate
 * 
 * Since: 7.5.0
 */
guint
ags_seq_synth_util_get_samplerate(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0);
  }

  return(seq_synth_util->samplerate);
}

/**
 * ags_seq_synth_util_set_samplerate:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @samplerate: the samplerate
 *
 * Set @samplerate of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_samplerate(AgsSeqSynthUtil *seq_synth_util,
				  guint samplerate)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->samplerate = samplerate;
}

/**
 * ags_seq_synth_util_get_synth_oscillator_mode:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get synth oscillator mode of @seq_synth_util.
 * 
 * Returns: the synth oscillator mode
 * 
 * Since: 7.5.0
 */
guint
ags_seq_synth_util_get_synth_oscillator_mode(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0);
  }

  return(seq_synth_util->synth_oscillator_mode);
}

/**
 * ags_seq_synth_util_set_synth_oscillator_mode:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @synth_oscillator_mode: the synth oscillator mode
 *
 * Set @synth_oscillator_mode of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_synth_oscillator_mode(AgsSeqSynthUtil *seq_synth_util,
					     guint synth_oscillator_mode)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->synth_oscillator_mode = synth_oscillator_mode;
}

/**
 * ags_seq_synth_util_get_frequency:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get frequency of @seq_synth_util.
 * 
 * Returns: the frequency
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_frequency(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(1.0);
  }

  return(seq_synth_util->frequency);
}

/**
 * ags_seq_synth_util_set_frequency:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @frequency: the frequency
 *
 * Set @frequency of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_frequency(AgsSeqSynthUtil *seq_synth_util,
				 gdouble frequency)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->frequency = frequency;
}

/**
 * ags_seq_synth_util_get_phase:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get phase of @seq_synth_util.
 * 
 * Returns: the phase
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_phase(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(1.0);
  }

  return(seq_synth_util->phase);
}

/**
 * ags_seq_synth_util_set_phase:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @phase: the phase
 *
 * Set @phase of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_phase(AgsSeqSynthUtil *seq_synth_util,
			     gdouble phase)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->phase = phase;
}

/**
 * ags_seq_synth_util_get_volume:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get volume of @seq_synth_util.
 * 
 * Returns: the volume
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_volume(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(1.0);
  }

  return(seq_synth_util->volume);
}

/**
 * ags_seq_synth_util_set_volume:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @volume: the volume
 *
 * Set @volume of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_volume(AgsSeqSynthUtil *seq_synth_util,
			      gdouble volume)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->volume = volume;
}

/**
 * ags_seq_synth_util_get_seq_tuning_0:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq tuning 0 of @seq_synth_util.
 * 
 * Returns: the seq tuning 0
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_seq_tuning_0(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->seq_tuning_0);
}

/**
 * ags_seq_synth_util_set_seq_tuning_0:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_tuning_0: the seq tuning 0
 *
 * Set @seq_tuning_0 of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_tuning_0(AgsSeqSynthUtil *seq_synth_util,
				    gdouble seq_tuning_0)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_tuning_0 = seq_tuning_0;
}

/**
 * ags_seq_synth_util_get_seq_tuning_1:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq tuning 1 of @seq_synth_util.
 * 
 * Returns: the seq tuning 1
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_seq_tuning_1(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->seq_tuning_1);
}

/**
 * ags_seq_synth_util_set_seq_tuning_1:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_tuning_1: the seq tuning 1
 *
 * Set @seq_tuning_1 of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_tuning_1(AgsSeqSynthUtil *seq_synth_util,
				    gdouble seq_tuning_1)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_tuning_1 = seq_tuning_1;
}

/**
 * ags_seq_synth_util_get_seq_tuning_2:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq tuning 2 of @seq_synth_util.
 * 
 * Returns: the seq tuning 2
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_seq_tuning_2(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->seq_tuning_2);
}

/**
 * ags_seq_synth_util_set_seq_tuning_2:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_tuning_2: the seq tuning 2
 *
 * Set @seq_tuning_2 of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_tuning_2(AgsSeqSynthUtil *seq_synth_util,
				    gdouble seq_tuning_2)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_tuning_2 = seq_tuning_2;
}

/**
 * ags_seq_synth_util_get_seq_tuning_3:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq tuning 3 of @seq_synth_util.
 * 
 * Returns: the seq tuning 3
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_seq_tuning_3(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->seq_tuning_3);
}

/**
 * ags_seq_synth_util_set_seq_tuning_3:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_tuning_3: the seq tuning 3
 *
 * Set @seq_tuning_3 of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_tuning_3(AgsSeqSynthUtil *seq_synth_util,
				    gdouble seq_tuning_3)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_tuning_3 = seq_tuning_3;
}

/**
 * ags_seq_synth_util_get_seq_tuning_4:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq tuning 4 of @seq_synth_util.
 * 
 * Returns: the seq tuning 4
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_seq_tuning_4(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->seq_tuning_4);
}

/**
 * ags_seq_synth_util_set_seq_tuning_4:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_tuning_4: the seq tuning 4
 *
 * Set @seq_tuning_4 of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_tuning_4(AgsSeqSynthUtil *seq_synth_util,
				    gdouble seq_tuning_4)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_tuning_4 = seq_tuning_4;
}

/**
 * ags_seq_synth_util_get_seq_tuning_5:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq tuning 5 of @seq_synth_util.
 * 
 * Returns: the seq tuning 5
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_seq_tuning_5(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->seq_tuning_5);
}

/**
 * ags_seq_synth_util_set_seq_tuning_5:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_tuning_5: the seq tuning 5
 *
 * Set @seq_tuning_5 of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_tuning_5(AgsSeqSynthUtil *seq_synth_util,
				    gdouble seq_tuning_5)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_tuning_5 = seq_tuning_5;
}

/**
 * ags_seq_synth_util_get_seq_tuning_6:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq tuning 6 of @seq_synth_util.
 * 
 * Returns: the seq tuning 6
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_seq_tuning_6(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->seq_tuning_6);
}

/**
 * ags_seq_synth_util_set_seq_tuning_6:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_tuning_6: the seq tuning 6
 *
 * Set @seq_tuning_6 of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_tuning_6(AgsSeqSynthUtil *seq_synth_util,
				    gdouble seq_tuning_6)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_tuning_6 = seq_tuning_6;
}

/**
 * ags_seq_synth_util_get_seq_tuning_7:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq tuning 7 of @seq_synth_util.
 * 
 * Returns: the seq tuning 7
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_seq_tuning_7(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->seq_tuning_7);
}

/**
 * ags_seq_synth_util_set_seq_tuning_7:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_tuning_7: the seq tuning 7
 *
 * Set @seq_tuning_7 of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_tuning_7(AgsSeqSynthUtil *seq_synth_util,
				    gdouble seq_tuning_7)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_tuning_7 = seq_tuning_7;
}

/**
 * ags_seq_synth_util_get_seq_tuning_pingpong:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq tuning pingpong of @seq_synth_util.
 * 
 * Returns: the seq tuning pingpong
 * 
 * Since: 7.5.0
 */
gboolean
ags_seq_synth_util_get_seq_tuning_pingpong(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(FALSE);
  }

  return(seq_synth_util->seq_tuning_pingpong);
}

/**
 * ags_seq_synth_util_set_seq_tuning_pingpong:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_tuning_pingpong: the seq tuning pingpong
 *
 * Set @seq_tuning_pingpong of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_tuning_pingpong(AgsSeqSynthUtil *seq_synth_util,
					   gboolean seq_tuning_pingpong)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_tuning_pingpong = seq_tuning_pingpong;
}

/**
 * ags_seq_synth_util_get_seq_tuning_lfo_frequency:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq tuning lfo_frequency of @seq_synth_util.
 * 
 * Returns: the seq tuning lfo_frequency
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_seq_tuning_lfo_frequency(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->seq_tuning_lfo_frequency);
}

/**
 * ags_seq_synth_util_set_seq_tuning_lfo_frequency:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_tuning_lfo_frequency: the seq tuning lfo_frequency
 *
 * Set @seq_tuning_lfo_frequency of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_tuning_lfo_frequency(AgsSeqSynthUtil *seq_synth_util,
						gdouble seq_tuning_lfo_frequency)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_tuning_lfo_frequency = seq_tuning_lfo_frequency;
}

/**
 * ags_seq_synth_util_get_seq_volume_0:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq volume 0 of @seq_synth_util.
 * 
 * Returns: the seq volume 0
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_seq_volume_0(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->seq_volume_0);
}

/**
 * ags_seq_synth_util_set_seq_volume_0:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_volume_0: the seq volume 0
 *
 * Set @seq_volume_0 of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_volume_0(AgsSeqSynthUtil *seq_synth_util,
				    gdouble seq_volume_0)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_volume_0 = seq_volume_0;
}

/**
 * ags_seq_synth_util_get_seq_volume_1:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq volume 1 of @seq_synth_util.
 * 
 * Returns: the seq volume 1
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_seq_volume_1(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->seq_volume_1);
}

/**
 * ags_seq_synth_util_set_seq_volume_1:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_volume_1: the seq volume 1
 *
 * Set @seq_volume_1 of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_volume_1(AgsSeqSynthUtil *seq_synth_util,
				    gdouble seq_volume_1)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_volume_1 = seq_volume_1;
}

/**
 * ags_seq_synth_util_get_seq_volume_2:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq volume 2 of @seq_synth_util.
 * 
 * Returns: the seq volume 2
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_seq_volume_2(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->seq_volume_2);
}

/**
 * ags_seq_synth_util_set_seq_volume_2:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_volume_2: the seq volume 2
 *
 * Set @seq_volume_2 of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_volume_2(AgsSeqSynthUtil *seq_synth_util,
				    gdouble seq_volume_2)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_volume_2 = seq_volume_2;
}

/**
 * ags_seq_synth_util_get_seq_volume_3:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq volume 3 of @seq_synth_util.
 * 
 * Returns: the seq volume 3
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_seq_volume_3(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->seq_volume_3);
}

/**
 * ags_seq_synth_util_set_seq_volume_3:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_volume_3: the seq volume 3
 *
 * Set @seq_volume_3 of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_volume_3(AgsSeqSynthUtil *seq_synth_util,
				    gdouble seq_volume_3)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_volume_3 = seq_volume_3;
}

/**
 * ags_seq_synth_util_get_seq_volume_4:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq volume 4 of @seq_synth_util.
 * 
 * Returns: the seq volume 4
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_seq_volume_4(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->seq_volume_4);
}

/**
 * ags_seq_synth_util_set_seq_volume_4:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_volume_4: the seq volume 4
 *
 * Set @seq_volume_4 of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_volume_4(AgsSeqSynthUtil *seq_synth_util,
				    gdouble seq_volume_4)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_volume_4 = seq_volume_4;
}

/**
 * ags_seq_synth_util_get_seq_volume_5:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq volume 5 of @seq_synth_util.
 * 
 * Returns: the seq volume 5
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_seq_volume_5(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->seq_volume_5);
}

/**
 * ags_seq_synth_util_set_seq_volume_5:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_volume_5: the seq volume 5
 *
 * Set @seq_volume_5 of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_volume_5(AgsSeqSynthUtil *seq_synth_util,
				    gdouble seq_volume_5)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_volume_5 = seq_volume_5;
}

/**
 * ags_seq_synth_util_get_seq_volume_6:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq volume 6 of @seq_synth_util.
 * 
 * Returns: the seq volume 6
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_seq_volume_6(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->seq_volume_6);
}

/**
 * ags_seq_synth_util_set_seq_volume_6:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_volume_6: the seq volume 6
 *
 * Set @seq_volume_6 of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_volume_6(AgsSeqSynthUtil *seq_synth_util,
				    gdouble seq_volume_6)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_volume_6 = seq_volume_6;
}

/**
 * ags_seq_synth_util_get_seq_volume_7:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq volume 7 of @seq_synth_util.
 * 
 * Returns: the seq volume 7
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_seq_volume_7(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->seq_volume_7);
}

/**
 * ags_seq_synth_util_set_seq_volume_7:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_volume_7: the seq volume 7
 *
 * Set @seq_volume_7 of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_volume_7(AgsSeqSynthUtil *seq_synth_util,
				    gdouble seq_volume_7)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_volume_7 = seq_volume_7;
}

/**
 * ags_seq_synth_util_get_seq_volume_pingpong:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq volume pingpong of @seq_synth_util.
 * 
 * Returns: the seq volume pingpong
 * 
 * Since: 7.5.0
 */
gboolean
ags_seq_synth_util_get_seq_volume_pingpong(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(FALSE);
  }

  return(seq_synth_util->seq_volume_pingpong);
}

/**
 * ags_seq_synth_util_set_seq_volume_pingpong:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_volume_pingpong: the seq volume pingpong
 *
 * Set @seq_volume_pingpong of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_volume_pingpong(AgsSeqSynthUtil *seq_synth_util,
					   gboolean seq_volume_pingpong)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_volume_pingpong = seq_volume_pingpong;
}

/**
 * ags_seq_synth_util_get_seq_volume_lfo_frequency:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get seq volume lfo_frequency of @seq_synth_util.
 * 
 * Returns: the seq volume lfo_frequency
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_seq_volume_lfo_frequency(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->seq_volume_lfo_frequency);
}

/**
 * ags_seq_synth_util_set_seq_volume_lfo_frequency:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @seq_volume_lfo_frequency: the seq volume lfo_frequency
 *
 * Set @seq_volume_lfo_frequency of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_seq_volume_lfo_frequency(AgsSeqSynthUtil *seq_synth_util,
						gdouble seq_volume_lfo_frequency)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->seq_volume_lfo_frequency = seq_volume_lfo_frequency;
}

/**
 * ags_seq_synth_util_get_lfo_oscillator_mode:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get LFO oscillator mode of @seq_synth_util.
 * 
 * Returns: the LFO oscillator mode
 * 
 * Since: 7.5.0
 */
guint
ags_seq_synth_util_get_lfo_oscillator_mode(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0);
  }

  return(seq_synth_util->lfo_oscillator_mode);
}

/**
 * ags_seq_synth_util_set_lfo_oscillator_mode:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @lfo_oscillator_mode: the LFO oscillator mode
 *
 * Set @lfo_oscillator_mode of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_lfo_oscillator_mode(AgsSeqSynthUtil *seq_synth_util,
					   guint lfo_oscillator_mode)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->lfo_oscillator_mode = lfo_oscillator_mode;
}

/**
 * ags_seq_synth_util_get_lfo_frequency:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get LFO frequency of @seq_synth_util.
 * 
 * Returns: the LFO frequency
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_lfo_frequency(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->lfo_frequency);
}

/**
 * ags_seq_synth_util_set_lfo_frequency:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @lfo_frequency: the LFO frequency
 *
 * Set @lfo_frequency of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_lfo_frequency(AgsSeqSynthUtil *seq_synth_util,
				     gdouble lfo_frequency)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->lfo_frequency = lfo_frequency;
}

/**
 * ags_seq_synth_util_get_lfo_depth:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get LFO depth of @seq_synth_util.
 * 
 * Returns: the LFO depth
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_lfo_depth(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(1.0);
  }

  return(seq_synth_util->lfo_depth);
}

/**
 * ags_seq_synth_util_set_lfo_depth:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @lfo_depth: the LFO depth
 *
 * Set @lfo_depth of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_lfo_depth(AgsSeqSynthUtil *seq_synth_util,
				 gdouble lfo_depth)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->lfo_depth = lfo_depth;
}

/**
 * ags_seq_synth_util_get_tuning:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get tuning of @seq_synth_util.
 * 
 * Returns: the tuning
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_tuning(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(1.0);
  }

  return(seq_synth_util->tuning);
}

/**
 * ags_seq_synth_util_set_tuning:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @tuning: the tuning
 *
 * Set @tuning of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_tuning(AgsSeqSynthUtil *seq_synth_util,
			      gdouble tuning)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->tuning = tuning;
}

/**
 * ags_seq_synth_util_get_vibrato_enabled:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get vibrato enabled of @seq_synth_util.
 * 
 * Returns: %TRUE if the vibrato is enabled, otherwise %FALSE
 * 
 * Since: 7.5.0
 */
gboolean
ags_seq_synth_util_get_vibrato_enabled(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(FALSE);
  }

  return(seq_synth_util->vibrato_enabled);
}

/**
 * ags_seq_synth_util_set_vibrato_enabled:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @vibrato_enabled: the vibrato enabled
 *
 * Set @vibrato_enabled of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_vibrato_enabled(AgsSeqSynthUtil *seq_synth_util,
				       gboolean vibrato_enabled)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->vibrato_enabled = vibrato_enabled;
}

/**
 * ags_seq_synth_util_get_vibrato_gain:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get vibrato gain of @seq_synth_util.
 * 
 * Returns: the vibrato gain
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_vibrato_gain(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->vibrato_gain);
}

/**
 * ags_seq_synth_util_set_vibrato_gain:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @vibrato_gain: the vibrato gain
 *
 * Set @vibrato_gain of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_vibrato_gain(AgsSeqSynthUtil *seq_synth_util,
				    gdouble vibrato_gain)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->vibrato_gain = vibrato_gain;
}

/**
 * ags_seq_synth_util_get_vibrato_lfo_depth:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get vibrato lfo_depth of @seq_synth_util.
 * 
 * Returns: the vibrato lfo_depth
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_vibrato_lfo_depth(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->vibrato_lfo_depth);
}

/**
 * ags_seq_synth_util_set_vibrato_lfo_depth:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @vibrato_lfo_depth: the vibrato lfo_depth
 *
 * Set @vibrato_lfo_depth of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_vibrato_lfo_depth(AgsSeqSynthUtil *seq_synth_util,
					 gdouble vibrato_lfo_depth)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->vibrato_lfo_depth = vibrato_lfo_depth;
}

/**
 * ags_seq_synth_util_get_vibrato_lfo_freq:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get vibrato lfo_freq of @seq_synth_util.
 * 
 * Returns: the vibrato lfo_freq
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_vibrato_lfo_freq(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->vibrato_lfo_freq);
}

/**
 * ags_seq_synth_util_set_vibrato_lfo_freq:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @vibrato_lfo_freq: the vibrato lfo_freq
 *
 * Set @vibrato_lfo_freq of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_vibrato_lfo_freq(AgsSeqSynthUtil *seq_synth_util,
					gdouble vibrato_lfo_freq)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->vibrato_lfo_freq = vibrato_lfo_freq;
}

/**
 * ags_seq_synth_util_get_vibrato_tuning:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get vibrato tuning of @seq_synth_util.
 * 
 * Returns: the vibrato tuning
 * 
 * Since: 7.5.0
 */
gdouble
ags_seq_synth_util_get_vibrato_tuning(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0.0);
  }

  return(seq_synth_util->vibrato_tuning);
}

/**
 * ags_seq_synth_util_set_vibrato_tuning:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @vibrato_tuning: the vibrato tuning
 *
 * Set @vibrato_tuning of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_vibrato_tuning(AgsSeqSynthUtil *seq_synth_util,
				      gdouble vibrato_tuning)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->vibrato_tuning = vibrato_tuning;
}

/**
 * ags_seq_synth_util_get_frame_count:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get frame count of @seq_synth_util.
 * 
 * Returns: the frame count
 * 
 * Since: 7.5.0
 */
guint
ags_seq_synth_util_get_frame_count(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0);
  }

  return(seq_synth_util->frame_count);
}

/**
 * ags_seq_synth_util_set_frame_count:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @frame_count: the frame count
 *
 * Set @frame_count of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_frame_count(AgsSeqSynthUtil *seq_synth_util,
				   guint frame_count)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->frame_count = frame_count;
}

/**
 * ags_seq_synth_util_get_offset:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get offset of @seq_synth_util.
 * 
 * Returns: the offset
 * 
 * Since: 7.5.0
 */
guint
ags_seq_synth_util_get_offset(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0);
  }

  return(seq_synth_util->offset);
}

/**
 * ags_seq_synth_util_set_offset:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @offset: the offset
 *
 * Set @offset of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_offset(AgsSeqSynthUtil *seq_synth_util,
			      guint offset)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->offset = offset;
}

/**
 * ags_seq_synth_util_get_note_256th_mode:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get note 256th mode of @seq_synth_util.
 * 
 * Returns: %TRUE if note 256th mode, otherwise %FALSE
 * 
 * Since: 7.5.0
 */
gboolean
ags_seq_synth_util_get_note_256th_mode(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0);
  }

  return(seq_synth_util->note_256th_mode);
}

/**
 * ags_seq_synth_util_set_note_256th_mode:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @note_256th_mode: the note 256th mode
 *
 * Set @note_256th_mode of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_note_256th_mode(AgsSeqSynthUtil *seq_synth_util,
				       gboolean note_256th_mode)
{
  if(seq_synth_util == NULL){
    return;
  }

  seq_synth_util->note_256th_mode = note_256th_mode;
}

/**
 * ags_seq_synth_util_get_offset_256th:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Get offset as note 256th of @seq_synth_util.
 * 
 * Returns: the offset as note 256th
 * 
 * Since: 7.5.0
 */
guint
ags_seq_synth_util_get_offset_256th(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL){
    return(0);
  }

  return(seq_synth_util->offset_256th);
}

/**
 * ags_seq_synth_util_set_offset_256th:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @offset_256th: the offset as note 256th
 *
 * Set @offset_256th of @seq_synth_util.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_set_offset_256th(AgsSeqSynthUtil *seq_synth_util,
				    guint offset_256th)
{
  if(seq_synth_util == NULL){
    return;
  }
  
  seq_synth_util->offset_256th = offset_256th;
}

/**
 * ags_seq_synth_util_get_tuning_by_offset:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @offset: the offset
 *
 * Get tuning by offset of @seq_synth_util.
 * 
 * Returns: the tuning
 *
 * Since: 7.5.0 
 */
gdouble
ags_seq_synth_util_get_tuning_by_offset(AgsSeqSynthUtil *seq_synth_util,
					guint offset)
{
  guint samplerate;
  gdouble tuning_a, tuning_b;
  gdouble tuning;
  gdouble seq_tuning_lfo_frequency;
  gboolean seq_tuning_pingpong;
  guint seq_offset;
  guint seq_tuning_offset, seq_tuning_offset_end;
  
  samplerate = seq_synth_util->samplerate;

  tuning_a = 0.0;
  tuning_b = 0.0;
  
  seq_tuning_lfo_frequency = seq_synth_util->seq_tuning_lfo_frequency;
  
  seq_tuning_pingpong = seq_synth_util->seq_tuning_pingpong;

  seq_offset = 0;

  if(seq_tuning_lfo_frequency != 0.0){
    seq_offset = (guint) floor(offset / (samplerate / seq_tuning_lfo_frequency));
  }
  
  seq_tuning_offset = 0;
  seq_tuning_offset_end = 0;
  
  if(seq_tuning_lfo_frequency != 0.0){
    seq_tuning_offset_end = (guint) floor((double) samplerate / seq_tuning_lfo_frequency);
  }

  if(seq_tuning_offset_end != 0){
    seq_tuning_offset = offset % seq_tuning_offset_end;
  }
  
  switch(seq_offset % 8){
  case 0:
    {
      if(!seq_tuning_pingpong){
	tuning_a = seq_synth_util->seq_tuning_0;
	tuning_b = seq_synth_util->seq_tuning_1;
      }else{
	if((guint) floor(seq_offset / 8.0) % 2 == 1){
	  tuning_a = seq_synth_util->seq_tuning_0;
	  tuning_b = seq_synth_util->seq_tuning_1;
	}else{
	  tuning_a = seq_synth_util->seq_tuning_0;
	  tuning_b = seq_synth_util->seq_tuning_1;
	}
      }
    }
    break;
  case 1:
    {
      if(!seq_tuning_pingpong){
	tuning_a = seq_synth_util->seq_tuning_1;
	tuning_b = seq_synth_util->seq_tuning_2;
      }else{
	if((guint) floor(seq_offset / 8.0) % 2 == 1){
	  tuning_a = seq_synth_util->seq_tuning_1;
	  tuning_b = seq_synth_util->seq_tuning_0;
	}else{
	  tuning_a = seq_synth_util->seq_tuning_1;
	  tuning_b = seq_synth_util->seq_tuning_2;
	}
      }
    }
    break;
  case 2:
    {
      if(!seq_tuning_pingpong){
	tuning_a = seq_synth_util->seq_tuning_2;
	tuning_b = seq_synth_util->seq_tuning_3;
      }else{
	if((guint) floor(seq_offset / 8.0) % 2 == 1){
	  tuning_a = seq_synth_util->seq_tuning_2;
	  tuning_b = seq_synth_util->seq_tuning_1;
	}else{
	  tuning_a = seq_synth_util->seq_tuning_2;
	  tuning_b = seq_synth_util->seq_tuning_3;
	}
      }
    }
    break;
  case 3:
    {
      if(!seq_tuning_pingpong){
	tuning_a = seq_synth_util->seq_tuning_3;
	tuning_b = seq_synth_util->seq_tuning_4;
      }else{
	if((guint) floor(seq_offset / 8.0) % 2 == 1){
	  tuning_a = seq_synth_util->seq_tuning_3;
	  tuning_b = seq_synth_util->seq_tuning_2;
	}else{
	  tuning_a = seq_synth_util->seq_tuning_3;
	  tuning_b = seq_synth_util->seq_tuning_4;
	}
      }
    }
    break;
  case 4:
    {
      if(!seq_tuning_pingpong){
	tuning_a = seq_synth_util->seq_tuning_4;
	tuning_b = seq_synth_util->seq_tuning_5;
      }else{
	if((guint) floor(seq_offset / 8.0) % 2 == 1){
	  tuning_a = seq_synth_util->seq_tuning_4;
	  tuning_b = seq_synth_util->seq_tuning_3;
	}else{
	  tuning_a = seq_synth_util->seq_tuning_4;
	  tuning_b = seq_synth_util->seq_tuning_5;
	}
      }
    }
    break;
  case 5:
    {
      if(!seq_tuning_pingpong){
	tuning_a = seq_synth_util->seq_tuning_5;
	tuning_b = seq_synth_util->seq_tuning_6;
      }else{
	if((guint) floor(seq_offset / 8.0) % 2 == 1){
	  tuning_a = seq_synth_util->seq_tuning_5;
	  tuning_b = seq_synth_util->seq_tuning_4;
	}else{
	  tuning_a = seq_synth_util->seq_tuning_5;
	  tuning_b = seq_synth_util->seq_tuning_6;
	}
      }
    }
    break;
  case 6:
    {
      if(!seq_tuning_pingpong){
	tuning_a = seq_synth_util->seq_tuning_6;
	tuning_b = seq_synth_util->seq_tuning_7;
      }else{
	if((guint) floor(seq_offset / 8.0) % 2 == 1){
	  tuning_a = seq_synth_util->seq_tuning_6;
	  tuning_b = seq_synth_util->seq_tuning_5;
	}else{
	  tuning_a = seq_synth_util->seq_tuning_6;
	  tuning_b = seq_synth_util->seq_tuning_7;
	}
      }
    }
    break;
  case 7:
    {
      if(!seq_tuning_pingpong){
	tuning_a = seq_synth_util->seq_tuning_7;
	tuning_b = seq_synth_util->seq_tuning_0;
      }else{
	if((guint) floor(seq_offset / 8.0) % 2 == 1){
	  tuning_a = seq_synth_util->seq_tuning_7;
	  tuning_b = seq_synth_util->seq_tuning_6;
	}else{
	  tuning_a = seq_synth_util->seq_tuning_7;
	  tuning_b = seq_synth_util->seq_tuning_0;
	}
      }
    }
    break;
  }

  tuning = 0.0;
  
  if(seq_tuning_offset_end != 0){
    tuning = ((seq_tuning_offset) * (tuning_a / seq_tuning_offset_end)) + ((seq_tuning_offset_end - seq_tuning_offset) * (tuning_b / seq_tuning_offset_end));
  }
  
  return(tuning);
}

/**
 * ags_seq_synth_util_get_volume_by_offset:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * @offset: the offset
 *
 * Get volume by offset of @seq_synth_util.
 * 
 * Returns: the volume
 *
 * Since: 7.5.0 
 */
gdouble
ags_seq_synth_util_get_volume_by_offset(AgsSeqSynthUtil *seq_synth_util,
					guint offset)
{
  guint samplerate;
  gdouble volume_a, volume_b;
  gdouble volume;
  gdouble seq_volume_lfo_frequency;
  gboolean seq_volume_pingpong;
  guint seq_offset;
  guint seq_volume_offset, seq_volume_offset_end;
  
  samplerate = seq_synth_util->samplerate;

  volume_a = 1.0;
  volume_b = 1.0;
  
  seq_volume_lfo_frequency = seq_synth_util->seq_volume_lfo_frequency;
  
  seq_volume_pingpong = seq_synth_util->seq_volume_pingpong;

  seq_offset = 0;
  
  if(seq_volume_lfo_frequency != 0.0){
    seq_offset = (guint) floor(offset / (samplerate / seq_volume_lfo_frequency));
  }
  
  seq_volume_offset = 0;
  seq_volume_offset_end = 0;

  if(seq_volume_lfo_frequency != 0.0){
    seq_volume_offset_end = (guint) floor((double) samplerate / seq_volume_lfo_frequency);
  }

  if(seq_volume_offset_end != 0){
    seq_volume_offset = offset % seq_volume_offset_end;
  }
  
  switch(seq_offset % 8){
  case 0:
    {
      if(!seq_volume_pingpong){
	volume_a = seq_synth_util->seq_volume_0;
	volume_b = seq_synth_util->seq_volume_1;
      }else{
	if((guint) floor(seq_offset / 8.0) % 2 == 1){
	  volume_a = seq_synth_util->seq_volume_0;
	  volume_b = seq_synth_util->seq_volume_1;
	}else{
	  volume_a = seq_synth_util->seq_volume_0;
	  volume_b = seq_synth_util->seq_volume_1;
	}
      }
    }
    break;
  case 1:
    {
      if(!seq_volume_pingpong){
	volume_a = seq_synth_util->seq_volume_1;
	volume_b = seq_synth_util->seq_volume_2;
      }else{
	if((guint) floor(seq_offset / 8.0) % 2 == 1){
	  volume_a = seq_synth_util->seq_volume_1;
	  volume_b = seq_synth_util->seq_volume_0;
	}else{
	  volume_a = seq_synth_util->seq_volume_1;
	  volume_b = seq_synth_util->seq_volume_2;
	}
      }
    }
    break;
  case 2:
    {
      if(!seq_volume_pingpong){
	volume_a = seq_synth_util->seq_volume_2;
	volume_b = seq_synth_util->seq_volume_3;
      }else{
	if((guint) floor(seq_offset / 8.0) % 2 == 1){
	  volume_a = seq_synth_util->seq_volume_2;
	  volume_b = seq_synth_util->seq_volume_1;
	}else{
	  volume_a = seq_synth_util->seq_volume_2;
	  volume_b = seq_synth_util->seq_volume_3;
	}
      }
    }
    break;
  case 3:
    {
      if(!seq_volume_pingpong){
	volume_a = seq_synth_util->seq_volume_3;
	volume_b = seq_synth_util->seq_volume_4;
      }else{
	if((guint) floor(seq_offset / 8.0) % 2 == 1){
	  volume_a = seq_synth_util->seq_volume_3;
	  volume_b = seq_synth_util->seq_volume_2;
	}else{
	  volume_a = seq_synth_util->seq_volume_3;
	  volume_b = seq_synth_util->seq_volume_4;
	}
      }
    }
    break;
  case 4:
    {
      if(!seq_volume_pingpong){
	volume_a = seq_synth_util->seq_volume_4;
	volume_b = seq_synth_util->seq_volume_5;
      }else{
	if((guint) floor(seq_offset / 8.0) % 2 == 1){
	  volume_a = seq_synth_util->seq_volume_4;
	  volume_b = seq_synth_util->seq_volume_3;
	}else{
	  volume_a = seq_synth_util->seq_volume_4;
	  volume_b = seq_synth_util->seq_volume_5;
	}
      }
    }
    break;
  case 5:
    {
      if(!seq_volume_pingpong){
	volume_a = seq_synth_util->seq_volume_5;
	volume_b = seq_synth_util->seq_volume_6;
      }else{
	if((guint) floor(seq_offset / 8.0) % 2 == 1){
	  volume_a = seq_synth_util->seq_volume_5;
	  volume_b = seq_synth_util->seq_volume_4;
	}else{
	  volume_a = seq_synth_util->seq_volume_5;
	  volume_b = seq_synth_util->seq_volume_6;
	}
      }
    }
    break;
  case 6:
    {
      if(!seq_volume_pingpong){
	volume_a = seq_synth_util->seq_volume_6;
	volume_b = seq_synth_util->seq_volume_7;
      }else{
	if((guint) floor(seq_offset / 8.0) % 2 == 1){
	  volume_a = seq_synth_util->seq_volume_6;
	  volume_b = seq_synth_util->seq_volume_5;
	}else{
	  volume_a = seq_synth_util->seq_volume_6;
	  volume_b = seq_synth_util->seq_volume_7;
	}
      }
    }
    break;
  case 7:
    {
      if(!seq_volume_pingpong){
	volume_a = seq_synth_util->seq_volume_7;
	volume_b = seq_synth_util->seq_volume_0;
      }else{
	if((guint) floor(seq_offset / 8.0) % 2 == 1){
	  volume_a = seq_synth_util->seq_volume_7;
	  volume_b = seq_synth_util->seq_volume_6;
	}else{
	  volume_a = seq_synth_util->seq_volume_7;
	  volume_b = seq_synth_util->seq_volume_0;
	}
      }
    }
    break;
  }

  volume = 1.0;

  if(seq_volume_offset_end != 0){
    volume = ((seq_volume_offset) * (volume_a / seq_volume_offset_end)) + ((seq_volume_offset_end - seq_volume_offset) * (volume_b / seq_volume_offset_end));
  }
  
  return(volume);
}

/**
 * ags_seq_synth_util_compute_sin_s8:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Compute SEQ sine synth of signed 8 bit data.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_compute_sin_s8(AgsSeqSynthUtil *seq_synth_util)
{
  gint8 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 127.0;
  
  if(seq_synth_util == NULL ||
     seq_synth_util->source == NULL){
    return;
  }

  source = seq_synth_util->source;

  samplerate = seq_synth_util->samplerate;

  frequency = seq_synth_util->frequency;
  phase = seq_synth_util->phase;
  volume = seq_synth_util->volume;

  lfo_frequency = seq_synth_util->lfo_frequency;
  lfo_depth = seq_synth_util->lfo_depth;
  tuning = seq_synth_util->tuning;

  vibrato_gain = seq_synth_util->vibrato_gain;
  vibrato_lfo_depth = seq_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = seq_synth_util->vibrato_lfo_freq;
  vibrato_tuning = seq_synth_util->vibrato_tuning;

  if(seq_synth_util->vibrato_enabled == FALSE){
    vibrato_gain = 0.0;
  }
  
  offset = seq_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sine, v_seq_volume;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride)
    };

    v_seq_volume = (ags_v8double) {
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)
    };

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 1 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 2 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 3 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 4 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 5 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 6 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 7 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate)	
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    default:
      v_sine = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sine *= v_seq_volume;
    v_sine *= scale;
    v_sine *= volume;

    v_buffer += v_sine;

    *(source) = (gint8) v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint8) v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint8) v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint8) v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint8) v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint8) v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint8) v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint8) v_buffer[7];

    source += seq_synth_util->source_stride;    
      
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8], tmp2_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride)};

    double v_seq_volume[] = {
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)};
    
    double v_sine[8];

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    default:
      v_sine[0] = 0.0;
      v_sine[1] = 0.0;
      v_sine[2] = 0.0;
      v_sine[3] = 0.0;
      v_sine[4] = 0.0;
      v_sine[5] = 0.0;
      v_sine[6] = 0.0;
      v_sine[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};

    vDSP_vmulD(v_sine, 1, v_seq_volume, 1, tmp0_ret_v_buffer, 1, 8);
    
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_scale, 0, tmp1_ret_v_buffer, 1, 8);

    vDSP_vmulD(tmp1_ret_v_buffer, 1, v_volume, 0, tmp2_ret_v_buffer, 1, 8);

    vDSP_vaddD(v_buffer, 1, tmp2_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint8) ret_v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint8) ret_v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint8) ret_v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint8) ret_v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint8) ret_v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint8) ret_v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint8) ret_v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint8) ret_v_buffer[7];

    source += seq_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }
    
    source += seq_synth_util->source_stride;

    i += 8;
  }
#endif

  for(; i < seq_synth_util->buffer_length;){
    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }
    
    source += seq_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_seq_synth_util_compute_sin_s16:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Compute SEQ sine synth of signed 16 bit data.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_compute_sin_s16(AgsSeqSynthUtil *seq_synth_util)
{
  gint16 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 32767.0;
  
  if(seq_synth_util == NULL ||
     seq_synth_util->source == NULL){
    return;
  }

  source = seq_synth_util->source;

  samplerate = seq_synth_util->samplerate;

  frequency = seq_synth_util->frequency;
  phase = seq_synth_util->phase;
  volume = seq_synth_util->volume;

  lfo_frequency = seq_synth_util->lfo_frequency;
  lfo_depth = seq_synth_util->lfo_depth;
  tuning = seq_synth_util->tuning;

  vibrato_gain = seq_synth_util->vibrato_gain;
  vibrato_lfo_depth = seq_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = seq_synth_util->vibrato_lfo_freq;
  vibrato_tuning = seq_synth_util->vibrato_tuning;

  if(seq_synth_util->vibrato_enabled == FALSE){
    vibrato_gain = 0.0;
  }
  
  offset = seq_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sine, v_seq_volume;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride)
    };
    
    v_seq_volume = (ags_v8double) {
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)
    };

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 1 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 2 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 3 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 4 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 5 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 6 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 7 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate)	
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    default:
      v_sine = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sine *= v_seq_volume;
    v_sine *= scale;
    v_sine *= volume;

    v_buffer += v_sine;

    *(source) = (gint16) v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint16) v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint16) v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint16) v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint16) v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint16) v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint16) v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint16) v_buffer[7];

    source += seq_synth_util->source_stride;

    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8], tmp2_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride)};

    double v_seq_volume[] = {
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)};

    double v_sine[8];

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    default:
      v_sine[0] = 0.0;
      v_sine[1] = 0.0;
      v_sine[2] = 0.0;
      v_sine[3] = 0.0;
      v_sine[4] = 0.0;
      v_sine[5] = 0.0;
      v_sine[6] = 0.0;
      v_sine[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sine, 1, v_seq_volume, 1, tmp0_ret_v_buffer, 1, 8);
    
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_scale, 0, tmp1_ret_v_buffer, 1, 8);

    vDSP_vmulD(tmp1_ret_v_buffer, 1, v_volume, 0, tmp2_ret_v_buffer, 1, 8);
    
    vDSP_vaddD(v_buffer, 1, tmp2_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint16) ret_v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint16) ret_v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint16) ret_v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint16) ret_v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint16) ret_v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint16) ret_v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint16) ret_v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint16) ret_v_buffer[7];

    source += seq_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }

    source += seq_synth_util->source_stride;
    i += 8;
  }
#endif

  for(; i < seq_synth_util->buffer_length;){
    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint16) ((gint32) (source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint16) ((gint32) (source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint16) ((gint32) (source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint16) ((gint32) (source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint16) ((gint32) (source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }

    source += seq_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_seq_synth_util_compute_sin_s24:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Compute SEQ sine synth of signed 24 bit data.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_compute_sin_s24(AgsSeqSynthUtil *seq_synth_util)
{
  gint32 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 8388607.0;
  
  if(seq_synth_util == NULL ||
     seq_synth_util->source == NULL){
    return;
  }

  source = seq_synth_util->source;

  samplerate = seq_synth_util->samplerate;

  frequency = seq_synth_util->frequency;
  phase = seq_synth_util->phase;
  volume = seq_synth_util->volume;

  lfo_frequency = seq_synth_util->lfo_frequency;
  lfo_depth = seq_synth_util->lfo_depth;
  tuning = seq_synth_util->tuning;

  vibrato_gain = seq_synth_util->vibrato_gain;
  vibrato_lfo_depth = seq_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = seq_synth_util->vibrato_lfo_freq;
  vibrato_tuning = seq_synth_util->vibrato_tuning;

  if(seq_synth_util->vibrato_enabled == FALSE){
    vibrato_gain = 0.0;
  }
  
  offset = seq_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sine, v_seq_volume;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride)
    };
    
    v_seq_volume = (ags_v8double) {
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)
    };

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate)
	};      
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 1 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 2 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 3 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 4 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 5 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 6 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 7 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate)	
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    default:
      v_sine = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sine *= v_seq_volume;
    v_sine *= scale;
    v_sine *= volume;

    v_buffer += v_sine;

    *(source) = (gint32) v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[7];

    source += seq_synth_util->source_stride;
    
    i += 8; 
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8], tmp2_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride)};

    double v_seq_volume[] = {
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)};

    double v_sine[8];

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    default:
      v_sine[0] = 0.0;
      v_sine[1] = 0.0;
      v_sine[2] = 0.0;
      v_sine[3] = 0.0;
      v_sine[4] = 0.0;
      v_sine[5] = 0.0;
      v_sine[6] = 0.0;
      v_sine[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sine, 1, v_seq_volume, 1, tmp0_ret_v_buffer, 1, 8);
    
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_scale, 0, tmp1_ret_v_buffer, 1, 8);

    vDSP_vmulD(tmp1_ret_v_buffer, 1, v_volume, 0, tmp2_ret_v_buffer, 1, 8);

    vDSP_vaddD(v_buffer, 1, tmp2_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint32) ret_v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[7];

    source += seq_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }

    source += seq_synth_util->source_stride;
    i += 8;
  }
#endif

  for(; i < seq_synth_util->buffer_length;){
    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }
    
    source += seq_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_seq_synth_util_compute_sin_s32:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Compute SEQ sine synth of signed 32 bit data.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_compute_sin_s32(AgsSeqSynthUtil *seq_synth_util)
{
  gint32 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 214748363.0;
  
  if(seq_synth_util == NULL ||
     seq_synth_util->source == NULL){
    return;
  }

  source = seq_synth_util->source;

  samplerate = seq_synth_util->samplerate;

  frequency = seq_synth_util->frequency;
  phase = seq_synth_util->phase;
  volume = seq_synth_util->volume;

  lfo_frequency = seq_synth_util->lfo_frequency;
  lfo_depth = seq_synth_util->lfo_depth;
  tuning = seq_synth_util->tuning;

  vibrato_gain = seq_synth_util->vibrato_gain;
  vibrato_lfo_depth = seq_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = seq_synth_util->vibrato_lfo_freq;
  vibrato_tuning = seq_synth_util->vibrato_tuning;

  if(seq_synth_util->vibrato_enabled == FALSE){
    vibrato_gain = 0.0;
  }
  
  offset = seq_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sine, v_seq_volume;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride)
    };
    
    v_seq_volume = (ags_v8double) {
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)
    };

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 1 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 2 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 3 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 4 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 5 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 6 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 7 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate)	
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)
	};    
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};    
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    default:
      v_sine = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sine *= v_seq_volume;
    v_sine *= scale;
    v_sine *= volume;

    v_buffer += v_sine;

    *(source) = (gint32) v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[7];

    source += seq_synth_util->source_stride;    
      
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8], tmp2_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride)};

    double v_seq_volume[] = {
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)};

    double v_sine[8];

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    default:
      v_sine[0] = 0.0;
      v_sine[1] = 0.0;
      v_sine[2] = 0.0;
      v_sine[3] = 0.0;
      v_sine[4] = 0.0;
      v_sine[5] = 0.0;
      v_sine[6] = 0.0;
      v_sine[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sine, 1, v_seq_volume, 1, tmp0_ret_v_buffer, 1, 8);
    
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_scale, 0, tmp1_ret_v_buffer, 1, 8);

    vDSP_vmulD(tmp1_ret_v_buffer, 1, v_volume, 0, tmp2_ret_v_buffer, 1, 8);

    vDSP_vaddD(v_buffer, 1, tmp2_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint32) ret_v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[7];

    source += seq_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }

    source += seq_synth_util->source_stride;
    i += 8;
  }
#endif

  for(; i < seq_synth_util->buffer_length;){
    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint32) ((gint64) (source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint32) ((gint64) (source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint32) ((gint64) (source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint32) ((gint64) (source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint32) ((gint64) (source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }
    
    source += seq_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_seq_synth_util_compute_sin_s64:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Compute SEQ sine synth of signed 64 bit data.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_compute_sin_s64(AgsSeqSynthUtil *seq_synth_util)
{
  gint64 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 9223372036854775807.0;
  
  if(seq_synth_util == NULL ||
     seq_synth_util->source == NULL){
    return;
  }

  source = seq_synth_util->source;

  samplerate = seq_synth_util->samplerate;

  frequency = seq_synth_util->frequency;
  phase = seq_synth_util->phase;
  volume = seq_synth_util->volume;

  lfo_frequency = seq_synth_util->lfo_frequency;
  lfo_depth = seq_synth_util->lfo_depth;
  tuning = seq_synth_util->tuning;

  vibrato_gain = seq_synth_util->vibrato_gain;
  vibrato_lfo_depth = seq_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = seq_synth_util->vibrato_lfo_freq;
  vibrato_tuning = seq_synth_util->vibrato_tuning;

  if(seq_synth_util->vibrato_enabled == FALSE){
    vibrato_gain = 0.0;
  }
  
  offset = seq_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sine, v_seq_volume;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride)
    };
    
    v_seq_volume = (ags_v8double) {
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)
    };

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate)
	};      
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 1 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 2 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 3 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 4 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 5 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 6 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 7 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate)	
	};    
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};    
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    default:
      v_sine = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sine *= v_seq_volume;
    v_sine *= scale;
    v_sine *= volume;

    v_buffer += v_sine;

    *(source) = (gint64) v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint64) v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint64) v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint64) v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint64) v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint64) v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint64) v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint64) v_buffer[7];

    source += seq_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8], tmp2_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride)};

    double v_seq_volume[] = {
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)};

    double v_sine[8];

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    default:
      v_sine[0] = 0.0;
      v_sine[1] = 0.0;
      v_sine[2] = 0.0;
      v_sine[3] = 0.0;
      v_sine[4] = 0.0;
      v_sine[5] = 0.0;
      v_sine[6] = 0.0;
      v_sine[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sine, 1, v_seq_volume, 1, tmp0_ret_v_buffer, 1, 8);
    
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_scale, 0, tmp1_ret_v_buffer, 1, 8);

    vDSP_vmulD(tmp1_ret_v_buffer, 1, v_volume, 0, tmp2_ret_v_buffer, 1, 8);

    vDSP_vaddD(v_buffer, 1, tmp2_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint64) ret_v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint64) ret_v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint64) ret_v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint64) ret_v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint64) ret_v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint64) ret_v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint64) ret_v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint64) ret_v_buffer[7];

    source += seq_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }

    source += seq_synth_util->source_stride;
    i += 8;
  }
#endif

  for(; i < seq_synth_util->buffer_length;){
    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint64) ((gint64) (source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint64) ((gint64) (source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint64) ((gint64) (source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint64) ((gint64) (source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint64) ((gint64) (source)[0] + (gint64) (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }

    source += seq_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_seq_synth_util_compute_sin_float:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Compute SEQ sine synth of floating point data.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_compute_sin_float(AgsSeqSynthUtil *seq_synth_util)
{
  gfloat *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  if(seq_synth_util == NULL ||
     seq_synth_util->source == NULL){
    return;
  }

  source = seq_synth_util->source;

  samplerate = seq_synth_util->samplerate;

  frequency = seq_synth_util->frequency;
  phase = seq_synth_util->phase;
  volume = seq_synth_util->volume;

  lfo_frequency = seq_synth_util->lfo_frequency;
  lfo_depth = seq_synth_util->lfo_depth;
  tuning = seq_synth_util->tuning;

  vibrato_gain = seq_synth_util->vibrato_gain;
  vibrato_lfo_depth = seq_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = seq_synth_util->vibrato_lfo_freq;
  vibrato_tuning = seq_synth_util->vibrato_tuning;

  if(seq_synth_util->vibrato_enabled == FALSE){
    vibrato_gain = 0.0;
  }
  
  offset = seq_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sine, v_seq_volume;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride)
    };
    
    v_seq_volume = (ags_v8double) {
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)
    };

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate)
	};      
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 1 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 2 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 3 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 4 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 5 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 6 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 7 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate)	
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    default:
      v_sine = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sine *= v_seq_volume;
    v_sine *= volume;

    v_buffer += v_sine;

    *(source) = (gfloat) v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gfloat) v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gfloat) v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gfloat) v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gfloat) v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gfloat) v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gfloat) v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gfloat) v_buffer[7];

    source += seq_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride)};

    double v_seq_volume[] = {
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)};

    double v_sine[8];

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    default:
      v_sine[0] = 0.0;
      v_sine[1] = 0.0;
      v_sine[2] = 0.0;
      v_sine[3] = 0.0;
      v_sine[4] = 0.0;
      v_sine[5] = 0.0;
      v_sine[6] = 0.0;
      v_sine[7] = 0.0;
    };

    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sine, 1, v_seq_volume, 1, tmp0_ret_v_buffer, 1, 8);
    
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_volume, 0, tmp1_ret_v_buffer, 1, 8);

    vDSP_vaddD(v_buffer, 1, tmp1_ret_v_buffer, 1, ret_v_buffer, 1, 8);
        
    *(source) = (gfloat) ret_v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gfloat) ret_v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gfloat) ret_v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gfloat) ret_v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gfloat) ret_v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gfloat) ret_v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gfloat) ret_v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gfloat) ret_v_buffer[7];

    source += seq_synth_util->source_stride;
    
    i += 8;
  }
#else
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	*(source) = (gfloat) ((tmp_source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gfloat) ((tmp_source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gfloat) ((tmp_source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gfloat) ((tmp_source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gfloat) ((tmp_source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gfloat) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    }

    source += seq_synth_util->source_stride;

    i += 8;
  }
#endif

  for(; i < seq_synth_util->buffer_length;){
    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	*(source) = (gfloat) ((source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gfloat) ((source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gfloat) ((source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gfloat) ((source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gfloat) ((source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    }

    source += seq_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_seq_synth_util_compute_sin_double:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Compute SEQ sine synth of double precision floating point data.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_compute_sin_double(AgsSeqSynthUtil *seq_synth_util)
{
  gdouble *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  if(seq_synth_util == NULL ||
     seq_synth_util->source == NULL){
    return;
  }

  source = seq_synth_util->source;

  samplerate = seq_synth_util->samplerate;

  frequency = seq_synth_util->frequency;
  phase = seq_synth_util->phase;
  volume = seq_synth_util->volume;

  lfo_frequency = seq_synth_util->lfo_frequency;
  lfo_depth = seq_synth_util->lfo_depth;
  tuning = seq_synth_util->tuning;

  vibrato_gain = seq_synth_util->vibrato_gain;
  vibrato_lfo_depth = seq_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = seq_synth_util->vibrato_lfo_freq;
  vibrato_tuning = seq_synth_util->vibrato_tuning;

  if(seq_synth_util->vibrato_enabled == FALSE){
    vibrato_gain = 0.0;
  }
  
  offset = seq_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sine, v_seq_volume;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride)
    };
    
    v_seq_volume = (ags_v8double) {
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)
    };

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate)
	};      
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 1 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 2 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 3 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 4 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 5 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 6 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 7 + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate)	
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    default:
      v_sine = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sine *= v_seq_volume;
    v_sine *= volume;

    v_buffer += v_sine;

    *(source) = (gdouble) v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gdouble) v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gdouble) v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gdouble) v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gdouble) v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gdouble) v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gdouble) v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gdouble) v_buffer[7];

    source += seq_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride)};

    double v_seq_volume[] = {
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)};

    double v_sine[8];

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    default:
      v_sine[0] = 0.0;
      v_sine[1] = 0.0;
      v_sine[2] = 0.0;
      v_sine[3] = 0.0;
      v_sine[4] = 0.0;
      v_sine[5] = 0.0;
      v_sine[6] = 0.0;
      v_sine[7] = 0.0;
    };

    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sine, 1, v_seq_volume, 1, tmp0_ret_v_buffer, 1, 8);
    
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_volume, 0, tmp1_ret_v_buffer, 1, 8);

    vDSP_vaddD(v_buffer, 1, tmp1_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gdouble) ret_v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gdouble) ret_v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gdouble) ret_v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gdouble) ret_v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gdouble) ret_v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gdouble) ret_v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gdouble) ret_v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gdouble) ret_v_buffer[7];

    source += seq_synth_util->source_stride;
    
    i += 8;
  }
#else
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	*(source) = (gdouble) ((tmp_source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gdouble) ((tmp_source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gdouble) ((tmp_source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gdouble) ((tmp_source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gdouble) ((tmp_source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += seq_synth_util->source_stride) = (gdouble) ((tmp_source += seq_synth_util->source_stride)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    }

    source += seq_synth_util->source_stride;

    i += 8;
  }
#endif

  for(; i < seq_synth_util->buffer_length;){
    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	*(source) = (gdouble) ((source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gdouble) ((source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gdouble) ((source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gdouble) ((source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gdouble) ((source)[0] + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    }

    source += seq_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_seq_synth_util_compute_sin_complex:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Compute SEQ sine synth of complex data.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_compute_sin_complex(AgsSeqSynthUtil *seq_synth_util)
{
  AgsComplex *source;
  
  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i;

  if(seq_synth_util == NULL ||
     seq_synth_util->source == NULL){
    return;
  }

  source = seq_synth_util->source;

  samplerate = seq_synth_util->samplerate;

  frequency = seq_synth_util->frequency;
  phase = seq_synth_util->phase;
  volume = seq_synth_util->volume;

  lfo_frequency = seq_synth_util->lfo_frequency;
  lfo_depth = seq_synth_util->lfo_depth;
  tuning = seq_synth_util->tuning;

  vibrato_gain = seq_synth_util->vibrato_gain;
  vibrato_lfo_depth = seq_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = seq_synth_util->vibrato_lfo_freq;
  vibrato_tuning = seq_synth_util->vibrato_tuning;

  if(seq_synth_util->vibrato_enabled == FALSE){
    vibrato_gain = 0.0;
  }
  
  offset = seq_synth_util->offset;
  
  i = 0;
  
  for(; i < seq_synth_util->buffer_length;){
    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	ags_complex_set(source,
			(ags_complex_get(source) + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume)));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	ags_complex_set(source,
			(ags_complex_get(source) + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume)));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	ags_complex_set(source,
			(ags_complex_get(source) + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate) * volume)));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	ags_complex_set(source,
			(ags_complex_get(source) + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate) * volume)));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	ags_complex_set(source,
			(ags_complex_get(source) + (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate) * volume)));
      }
      break;
    }

    source += seq_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_seq_synth_util_compute_sin:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Compute SEQ sine synth.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_compute_sin(AgsSeqSynthUtil *seq_synth_util)
{
  if(seq_synth_util == NULL ||
     seq_synth_util->source == NULL){
    return;
  }

  switch(seq_synth_util->format){
  case AGS_SOUNDCARD_SIGNED_8_BIT:
    {
      ags_seq_synth_util_compute_sin_s8(seq_synth_util);
    }
    break;
  case AGS_SOUNDCARD_SIGNED_16_BIT:
    {
      ags_seq_synth_util_compute_sin_s16(seq_synth_util);
    }
    break;
  case AGS_SOUNDCARD_SIGNED_24_BIT:
    {
      ags_seq_synth_util_compute_sin_s24(seq_synth_util);
    }
    break;
  case AGS_SOUNDCARD_SIGNED_32_BIT:
    {
      ags_seq_synth_util_compute_sin_s32(seq_synth_util);
    }
    break;
  case AGS_SOUNDCARD_SIGNED_64_BIT:
    {
      ags_seq_synth_util_compute_sin_s64(seq_synth_util);
    }
    break;
  case AGS_SOUNDCARD_FLOAT:
    {
      ags_seq_synth_util_compute_sin_float(seq_synth_util);
    }
    break;
  case AGS_SOUNDCARD_DOUBLE:
    {
      ags_seq_synth_util_compute_sin_double(seq_synth_util);
    }
    break;
  case AGS_SOUNDCARD_COMPLEX:
    {
      ags_seq_synth_util_compute_sin_complex(seq_synth_util);
    }
    break;
  }
}


/**
 * ags_seq_synth_util_compute_sawtooth_s8:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Compute SEQ sawtooth synth of signed 8 bit data.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_compute_sawtooth_s8(AgsSeqSynthUtil *seq_synth_util)
{
  gint8 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 127.0;
  
  if(seq_synth_util == NULL ||
     seq_synth_util->source == NULL){
    return;
  }

  source = seq_synth_util->source;

  samplerate = seq_synth_util->samplerate;

  frequency = seq_synth_util->frequency;
  phase = seq_synth_util->phase;
  volume = seq_synth_util->volume;

  lfo_frequency = seq_synth_util->lfo_frequency;
  lfo_depth = seq_synth_util->lfo_depth;
  tuning = seq_synth_util->tuning;

  vibrato_gain = seq_synth_util->vibrato_gain;
  vibrato_lfo_depth = seq_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = seq_synth_util->vibrato_lfo_freq;
  vibrato_tuning = seq_synth_util->vibrato_tuning;

  if(seq_synth_util->vibrato_enabled == FALSE){
    vibrato_gain = 0.0;
  }
  
  offset = seq_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sawtooth, v_seq_volume;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride)
    };

    v_seq_volume = (ags_v8double) {
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)
    };
    
    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0)
	};      
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};    
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    default:
      v_sawtooth = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sawtooth *= v_seq_volume;
    v_sawtooth *= scale;
    v_sawtooth *= volume;

    v_buffer += v_sawtooth;

    *(source) = (gint8) v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint8) v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint8) v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint8) v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint8) v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint8) v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint8) v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint8) v_buffer[7];

    source += seq_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8], tmp2_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride)};

    double v_seq_volume[] = {
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)};

    double v_sawtooth[8];

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    default:
      v_sawtooth[0] = 0.0;
      v_sawtooth[1] = 0.0;
      v_sawtooth[2] = 0.0;
      v_sawtooth[3] = 0.0;
      v_sawtooth[4] = 0.0;
      v_sawtooth[5] = 0.0;
      v_sawtooth[6] = 0.0;
      v_sawtooth[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sawtooth, 1, v_seq_volume, 1, tmp0_ret_v_buffer, 1, 8);
    
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_scale, 0, tmp1_ret_v_buffer, 1, 8);

    vDSP_vmulD(tmp1_ret_v_buffer, 1, v_volume, 0, tmp2_ret_v_buffer, 1, 8);

    vDSP_vaddD(v_buffer, 1, tmp2_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint8) ret_v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint8) ret_v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint8) ret_v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint8) ret_v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint8) ret_v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint8) ret_v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint8) ret_v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint8) ret_v_buffer[7];

    source += seq_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) scale * ((sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate) - 1.0) * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + (i)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += seq_synth_util->source_stride)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }
    
    source += seq_synth_util->source_stride;

    i += 8;
  }
#endif

  for(; i < seq_synth_util->buffer_length;){
    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate) - 1.0) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }

    source += seq_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_seq_synth_util_compute_sawtooth_s16:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Compute SEQ sawtooth synth of signed 16 bit data.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_compute_sawtooth_s16(AgsSeqSynthUtil *seq_synth_util)
{
  gint16 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 32767.0;
  
  if(seq_synth_util == NULL ||
     seq_synth_util->source == NULL){
    return;
  }

  source = seq_synth_util->source;

  samplerate = seq_synth_util->samplerate;

  frequency = seq_synth_util->frequency;
  phase = seq_synth_util->phase;
  volume = seq_synth_util->volume;

  lfo_frequency = seq_synth_util->lfo_frequency;
  lfo_depth = seq_synth_util->lfo_depth;
  tuning = seq_synth_util->tuning;

  vibrato_gain = seq_synth_util->vibrato_gain;
  vibrato_lfo_depth = seq_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = seq_synth_util->vibrato_lfo_freq;
  vibrato_tuning = seq_synth_util->vibrato_tuning;

  if(seq_synth_util->vibrato_enabled == FALSE){
    vibrato_gain = 0.0;
  }
  
  offset = seq_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sawtooth, v_seq_volume;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride)
    };

    v_seq_volume = (ags_v8double) {
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)
    };
    
    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    default:
      v_sawtooth = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sawtooth *= v_seq_volume;
    v_sawtooth *= scale;
    v_sawtooth *= volume;

    v_buffer += v_sawtooth;

    *(source) = (gint16) v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint16) v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint16) v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint16) v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint16) v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint16) v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint16) v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint16) v_buffer[7];

    source += seq_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8], tmp2_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride)};

    double v_seq_volume[] = {
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)};

    double v_sawtooth[8];

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
 	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    default:
      v_sawtooth[0] = 0.0;
      v_sawtooth[1] = 0.0;
      v_sawtooth[2] = 0.0;
      v_sawtooth[3] = 0.0;
      v_sawtooth[4] = 0.0;
      v_sawtooth[5] = 0.0;
      v_sawtooth[6] = 0.0;
      v_sawtooth[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sawtooth, 1, v_seq_volume, 1, tmp0_ret_v_buffer, 1, 8);
    
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_scale, 0, tmp1_ret_v_buffer, 1, 8);

    vDSP_vmulD(tmp1_ret_v_buffer, 1, v_volume, 0, tmp2_ret_v_buffer, 1, 8);

    vDSP_vaddD(v_buffer, 1, tmp2_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint16) ret_v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint16) ret_v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint16) ret_v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint16) ret_v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint16) ret_v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint16) ret_v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint16) ret_v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint16) ret_v_buffer[7];

    source += seq_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	*(source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + (i)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	*(source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + (i)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }
    
    source += seq_synth_util->source_stride;

    i += 8;
  }
#endif

  for(; i < seq_synth_util->buffer_length;){
    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint16) ((gint32) (source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	*(source) = (gint16) ((gint32) (source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + (i)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint16) ((gint32) (source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	*(source) = (gint16) ((gint32) (source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + (i)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint16) ((gint32) (source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }    

    source += seq_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_seq_synth_util_compute_sawtooth_s24:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Compute SEQ sawtooth synth of signed 24 bit data.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_compute_sawtooth_s24(AgsSeqSynthUtil *seq_synth_util)
{
  gint32 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 8388607.0;
  
  if(seq_synth_util == NULL ||
     seq_synth_util->source == NULL){
    return;
  }

  source = seq_synth_util->source;

  samplerate = seq_synth_util->samplerate;

  frequency = seq_synth_util->frequency;
  phase = seq_synth_util->phase;
  volume = seq_synth_util->volume;

  lfo_frequency = seq_synth_util->lfo_frequency;
  lfo_depth = seq_synth_util->lfo_depth;
  tuning = seq_synth_util->tuning;

  vibrato_gain = seq_synth_util->vibrato_gain;
  vibrato_lfo_depth = seq_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = seq_synth_util->vibrato_lfo_freq;
  vibrato_tuning = seq_synth_util->vibrato_tuning;

  if(seq_synth_util->vibrato_enabled == FALSE){
    vibrato_gain = 0.0;
  }
  
  offset = seq_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sawtooth, v_seq_volume;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride)
    };
    
    v_seq_volume = (ags_v8double) {
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)
    };

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    default:
      v_sawtooth = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sawtooth *= v_seq_volume;
    v_sawtooth *= scale;
    v_sawtooth *= volume;

    v_buffer += v_sawtooth;

    *(source) = (gint32) v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[7];

    source += seq_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8], tmp2_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride)};

    double v_seq_volume[] = {
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)};

    double v_sawtooth[8];

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    default:
      v_sawtooth[0] = 0.0;
      v_sawtooth[1] = 0.0;
      v_sawtooth[2] = 0.0;
      v_sawtooth[3] = 0.0;
      v_sawtooth[4] = 0.0;
      v_sawtooth[5] = 0.0;
      v_sawtooth[6] = 0.0;
      v_sawtooth[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sawtooth, 1, v_seq_volume, 1, tmp0_ret_v_buffer, 1, 8);
    
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_scale, 0, tmp1_ret_v_buffer, 1, 8);

    vDSP_vmulD(tmp1_ret_v_buffer, 1, v_volume, 0, tmp2_ret_v_buffer, 1, 8);

    vDSP_vaddD(v_buffer, 1, tmp2_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint32) ret_v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[7];

    source += seq_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	*(source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + (i)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + (i)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += seq_synth_util->source_stride)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }
    
    source += seq_synth_util->source_stride;
    i += 8;
  }
#endif

  for(; i < seq_synth_util->buffer_length;){
    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	*(source) = (gint32) ((gint32) (source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + (i)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) scale * (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }

    source += seq_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_seq_synth_util_compute_sawtooth_s32:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Compute SEQ sawtooth synth of signed 32 bit data.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_compute_sawtooth_s32(AgsSeqSynthUtil *seq_synth_util)
{
  gint32 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 214748363.0;

  if(seq_synth_util == NULL ||
     seq_synth_util->source == NULL){
    return;
  }

  source = seq_synth_util->source;

  samplerate = seq_synth_util->samplerate;

  frequency = seq_synth_util->frequency;
  phase = seq_synth_util->phase;
  volume = seq_synth_util->volume;

  lfo_frequency = seq_synth_util->lfo_frequency;
  lfo_depth = seq_synth_util->lfo_depth;
  tuning = seq_synth_util->tuning;

  vibrato_gain = seq_synth_util->vibrato_gain;
  vibrato_lfo_depth = seq_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = seq_synth_util->vibrato_lfo_freq;
  vibrato_tuning = seq_synth_util->vibrato_tuning;

  if(seq_synth_util->vibrato_enabled == FALSE){
    vibrato_gain = 0.0;
  }
  
  offset = seq_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sawtooth, v_seq_volume;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride)
    };
    
    v_seq_volume = (ags_v8double) {
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)
    };

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    default:
      v_sawtooth = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }
    
    v_sawtooth *= v_seq_volume;
    v_sawtooth *= scale;
    v_sawtooth *= volume;

    v_buffer += v_sawtooth;
    
    *(source) = (gint32) v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint32) v_buffer[7];
    
    source += seq_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8], tmp2_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride)};

    double v_seq_volume[] = {
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)};

    double v_sawtooth[8];

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    default:
      v_sawtooth[0] = 0.0;
      v_sawtooth[1] = 0.0;
      v_sawtooth[2] = 0.0;
      v_sawtooth[3] = 0.0;
      v_sawtooth[4] = 0.0;
      v_sawtooth[5] = 0.0;
      v_sawtooth[6] = 0.0;
      v_sawtooth[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};

    vDSP_vmulD(v_sawtooth, 1, v_seq_volume, 1, tmp0_ret_v_buffer, 1, 8);
    
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_scale, 0, tmp1_ret_v_buffer, 1, 8);

    vDSP_vmulD(tmp1_ret_v_buffer, 1, v_volume, 0, tmp2_ret_v_buffer, 1, 8);

    vDSP_vaddD(v_buffer, 1, tmp2_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint32) ret_v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint32) ret_v_buffer[7];

    source += seq_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	*(source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + (i)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	*(source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + (i)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }
    
    source += seq_synth_util->source_stride;

    i += 8;
  }
#endif

  for(; i < seq_synth_util->buffer_length;){
    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint32) ((gint64) (source)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	*(source) = (gint32) ((gint64) (source)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + (i)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint32) ((gint64) (source)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	*(source) = (gint32) ((gint64) (source)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + (i)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint32) ((gint64) (source)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }
  
    source += seq_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_seq_synth_util_compute_sawtooth_s64:
 * @seq_synth_util: the #AgsSeqSynthUtil-struct
 * 
 * Compute SEQ sawtooth synth of signed 64 bit data.
 *
 * Since: 7.5.0
 */
void
ags_seq_synth_util_compute_sawtooth_s64(AgsSeqSynthUtil *seq_synth_util)
{
  gint64 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 9223372036854775807.0;
  
  if(seq_synth_util == NULL ||
     seq_synth_util->source == NULL){
    return;
  }

  source = seq_synth_util->source;

  samplerate = seq_synth_util->samplerate;

  frequency = seq_synth_util->frequency;
  phase = seq_synth_util->phase;
  volume = seq_synth_util->volume;

  lfo_frequency = seq_synth_util->lfo_frequency;
  lfo_depth = seq_synth_util->lfo_depth;
  tuning = seq_synth_util->tuning;

  vibrato_gain = seq_synth_util->vibrato_gain;
  vibrato_lfo_depth = seq_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = seq_synth_util->vibrato_lfo_freq;
  vibrato_tuning = seq_synth_util->vibrato_tuning;

  if(seq_synth_util->vibrato_enabled == FALSE){
    vibrato_gain = 0.0;
  }
  
  offset = seq_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sawtooth, v_seq_volume;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride),
      (gdouble) *(tmp_source += seq_synth_util->source_stride)
    };
    
    v_seq_volume = (ags_v8double) {
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)
    };

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0)
	};      
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    default:
      v_sawtooth = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sawtooth *= v_seq_volume;
    v_sawtooth *= scale;
    v_sawtooth *= volume;

    v_buffer += v_sawtooth;

    *(source) = (gint64) v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint64) v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint64) v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint64) v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint64) v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint64) v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint64) v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint64) v_buffer[7];

    source += seq_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8], tmp2_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride),
      (double) *(tmp_source += seq_synth_util->source_stride)};

    double v_seq_volume[] = {
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6),
      (double) ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7)};

    double v_sawtooth[8];

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    default:
      v_sawtooth[0] = 0.0;
      v_sawtooth[1] = 0.0;
      v_sawtooth[2] = 0.0;
      v_sawtooth[3] = 0.0;
      v_sawtooth[4] = 0.0;
      v_sawtooth[5] = 0.0;
      v_sawtooth[6] = 0.0;
      v_sawtooth[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sawtooth, 1, v_seq_volume, 1, tmp0_ret_v_buffer, 1, 8);
    
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_scale, 0, tmp1_ret_v_buffer, 1, 8);

    vDSP_vmulD(tmp1_ret_v_buffer, 1, v_volume, 0, tmp2_ret_v_buffer, 1, 8);

    vDSP_vaddD(v_buffer, 1, tmp2_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint64) ret_v_buffer[0];
    *(source += seq_synth_util->source_stride) = (gint64) ret_v_buffer[1];
    *(source += seq_synth_util->source_stride) = (gint64) ret_v_buffer[2];
    *(source += seq_synth_util->source_stride) = (gint64) ret_v_buffer[3];
    *(source += seq_synth_util->source_stride) = (gint64) ret_v_buffer[4];
    *(source += seq_synth_util->source_stride) = (gint64) ret_v_buffer[5];
    *(source += seq_synth_util->source_stride) = (gint64) ret_v_buffer[6];
    *(source += seq_synth_util->source_stride) = (gint64) ret_v_buffer[7];

    source += seq_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = seq_synth_util->buffer_length - (seq_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	*(source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	*(source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 1) * ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 1)) + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 2) * ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 2)) + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 3) * ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 3)) + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 4) * ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 4)) + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 5) * ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 5)) + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 6) * ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 6)) + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += seq_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += seq_synth_util->source_stride)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i + 7) * ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i + 7)) + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }
    
    source += seq_synth_util->source_stride;
    
    i += 8;
  }
#endif

  for(; i < seq_synth_util->buffer_length;){
    switch(seq_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint64) ((gint64) (source)[0] + (gint64) scale * (ags_seq_synth_util_get_volume_by_offset(seq_synth_util, offset + i) * ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2(((((tuning + ags_seq_synth_util_get_tuning_by_offset(seq_synth_util, offset + i)) + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	*(source) = (gint64) ((gint64) (source)[0] + (gint64) scale * (ags_seq_synth_u