/* $Id: gmt_core_module.c 15186 2015-11-06 21:01:51Z pwessel $
 *
 * Copyright (c) 2012-2015
 * by P. Wessel, W. H. F. Smith, R. Scharroo, J. Luis, and F. Wobbe
 * See LICENSE.TXT file for copying and redistribution conditions.
 */

/* gmt_core_module.c populates the external array of GMT core
 * module parameters such as name, group and purpose strings.
 * This file also contains the following convenience function to
 * display all module purposes:
 *
 *   void gmt_core_module_show_all (struct GMTAPI_CTRL *API);
 *
 * Developers of external APIs for accessing GMT modules may use this
 * function to retrieve option keys needed for module arg processing:
 *
 *   char * gmt_core_module_info (void *API, const char *module);
 *
 * DO NOT edit this file directly! Regenerate thee file by running
 * 	gmt_make_module_src.sh core
 */
#include "gmt_dev.h"
#ifndef BUILD_SHARED_LIBS
#include "gmt_core_module.h"
#endif

static inline struct GMTAPI_CTRL * gmt_get_api_ptr (struct GMTAPI_CTRL *ptr) {return (ptr);}

/* Sorted array with information for all GMT core modules */

/* name, library, and purpose for each module */
struct Gmt_moduleinfo {
	const char *name;             /* Program name */
	const char *component;        /* Component (core, supplement, custom) */
	const char *purpose;          /* Program purpose */
	const char *keys;             /* Program option info for external APIs */
#ifndef BUILD_SHARED_LIBS
	/* gmt module function pointer: */
	int (*p_func)(void*, int, void*);
#endif
};

struct Gmt_moduleinfo g_core_module[] = {
#ifdef BUILD_SHARED_LIBS
	{"blockmean", "core", "Block average (x,y,z) data tables by L2 norm", "<DI,>DO,RG-"},
	{"blockmedian", "core", "Block average (x,y,z) data tables by L1 norm (spatial median)", "<DI,>DO,RG-"},
	{"blockmode", "core", "Block average (x,y,z) data tables by mode estimation", "<DI,>DO,RG-"},
	{"filter1d", "core", "Time domain filtering of 1-D data tables", "<DI,>DO"},
	{"fitcircle", "core", "Find mean position and great [or small] circle fit to points on sphere", "<DI,>TO,>DF"},
	{"gmt2kml", "core", "Convert GMT data tables to KML files for Google Earth", "<DI,>TO"},
	{"gmtconnect", "core", "Connect individual lines whose end points match within tolerance", "<DI,>DO,CDo,LTo,QTo"},
	{"gmtconvert", "core", "Convert, paste, or extract columns from data tables", "<DI,>DO"},
	{"gmtdefaults", "core", "List current GMT default parameters", ""},
	{"gmtget", "core", "Get individual GMT default parameters", ">TO"},
	{"gmtinfo", "core", "Get information about data tables", "<DI,>TO,>DC"},
	{"gmtlogo", "core", "Plot the GMT logo on maps", ""},
	{"gmtmath", "core", "Reverse Polish Notation (RPN) calculator for data tables", "<DI,ADi,>DO"},
	{"gmtread", "core", "Read GMT objects into external API", "<?I,>?O,-T-"},
	{"gmtregress", "core", "Linear regression of 1-D data sets", "<DI,>DO"},
	{"gmtselect", "core", "Select data table subsets based on multiple spatial criteria", "<DI,CDi,FDi,LDi,>DO,RG-"},
	{"gmtset", "core", "Change individual GMT default parameters", ""},
	{"gmtsimplify", "core", "Line reduction using the Douglas-Peucker algorithm", "<DI,>DO"},
	{"gmtspatial", "core", "Geospatial operations on lines and polygons", "<DI,DDi,NDi,TDi,>DO"},
	{"gmtvector", "core", "Operations on Cartesian vectors in 2-D and 3-D", "<DI,ADi,>DO"},
	{"gmtwhich", "core", "Find full path to specified files", "<TI,>TO"},
	{"gmtwrite", "core", "Write GMT objects from external API", "<?I,>?O,-T-"},
	{"grd2cpt", "core", "Make linear or histogram-equalized color palette table from grid", "<GI,>CO,RG-"},
	{"grd2rgb", "core", "Write r/g/b grid files from a grid file, a raw RGB file, or SUN rasterfile", "<GI,RG-"},
	{"grd2xyz", "core", "Convert grid file to data table", "<GI,>DO,RG-"},
	{"grdblend", "core", "Blend several partially over-lapping grids into one larger grid", "<TI,GGO,RG-"},
	{"grdclip", "core", "Clip the range of grid values", "<GI,GGO,RG-"},
	{"grdcontour", "core", "Make contour map using a grid", "<GI,CCi,DDD,>XO,RG-"},
	{"grdconvert", "core", "Convert between different grid formats", "<GI,>GO,RG-"},
	{"grdcut", "core", "Extract subregion from a grid", "<GI,GGO,RG-"},
	{"grdedit", "core", "Modify header or content of a grid", "<GI,NDi,GGO,RG-"},
	{"grdfft", "core", "Mathematical operations on grids in the wavenumber (or frequency) domain", "<GI,GGO,RG-,GDE"},
	{"grdfilter", "core", "Filter a grid in the space (or time) domain", "<GI,GGO,RG-"},
	{"grdgradient", "core", "Compute directional gradients from a grid", "<GI,GGO,SGo,RG-"},
	{"grdhisteq", "core", "Perform histogram equalization for a grid", "<GI,GGO,DTo,RG-"},
	{"grdimage", "core", "Project grids or images and plot them on maps", "<GI,CCi,IGi,>XO,RG-,AIA"},
	{"grdinfo", "core", "Extract information from grids", "<GI,>TO,>DC,RG-"},
	{"grdlandmask", "core", "Create a \"wet-dry\" mask grid from shoreline data base", "GGO,RG-"},
	{"grdmask", "core", "Create mask grid from polygons or point coverage", "<DI,GGORGi"},
	{"grdmath", "core", "Reverse Polish Notation (RPN) calculator for grids (element by element)", "<GI,=GO,RG-"},
	{"grdpaste", "core", "Join two grids along their common edge", "<GI,GGO,RG-"},
	{"grdproject", "core", "Forward and inverse map transformation of grids", "<GI,GGO,RG-"},
	{"grdraster", "core", "Extract subregion from a binary raster and save as a GMT grid", "GGO,TDo,RG-"},
	{"grdsample", "core", "Resample a grid onto a new lattice", "<GI,GGO,RG-"},
	{"grdtrack", "core", "Sample grids at specified (x,y) locations", "<DI,DDo,GGi,>DO,RG-"},
	{"grdtrend", "core", "Fit trend surface to grids and compute residuals", "<GI,DGo,TGo,WGo,RG-"},
	{"grdvector", "core", "Plot vector field from two component grids", "<GI,CCi,>XO,RG-"},
	{"grdview", "core", "Create 3-D perspective image or surface mesh from a grid", "<GI,CCi,GGi,IGi,>XO,RG-"},
	{"grdvolume", "core", "Calculate grid volume and area constrained by a contour", "<GI,>DO,RG-"},
	{"greenspline", "core", "Interpolate using Green's functions for splines in 1-3 dimensions", "<DI,ADi,NDi,TGi,CDo,GGO,RG-"},
	{"kml2gmt", "core", "Extract GMT table data from Google Earth KML files", "<TI,>DO"},
	{"makecpt", "core", "Make GMT color palette tables", ">CO"},
	{"mapproject", "core", "Forward and inverse map transformations, datum conversions and geodesy", "<DI,LDi,>DO,W-i,RG-"},
	{"nearneighbor", "core", "Grid table data using a \"Nearest neighbor\" algorithm", "<DI,GGO,RG-"},
	{"project", "core", "Project table data onto lines or great circles, generate tracks, or translate coordinates", "<DI,>DO,G-i"},
	{"psbasemap", "core", "Plot base maps and frames", ">XO,RG-"},
	{"psclip", "core", "Initialize or terminate polygonal clip paths", "<DI,>XO"},
	{"pscoast", "core", "Plot continents, countries, shorelines, rivers, and borders on maps", ">XO,>DM,RG-"},
	{"pscontour", "core", "Contour table data by direct triangulation", "<DI,CCi,EDi,DDD,>XO,RG-"},
	{"psconvert", "core", "Convert [E]PS file(s) to other formats using GhostScript", "FIo"},
	{"pshistogram", "core", "Calculate and plot histograms", "<DI,>XO,>Do"},
	{"psimage", "core", "Place images or EPS files on maps", "<II,>XO,RG-"},
	{"pslegend", "core", "Plot legends on maps", "<TI,ACi,>XO,RG-"},
	{"psmask", "core", "Clip or mask map areas with no data table coverage", "<DI,DDD,C-i,>XO,RG-"},
	{"psrose", "core", "Plot a polar histogram (rose, sector, windrose diagrams)", "<DI,>XO"},
	{"psscale", "core", "Plot a gray-scale or color-scale on maps", "CCI,>XO"},
	{"pstext", "core", "Plot or typeset text on maps", "<TI,>XO,RG-"},
	{"pswiggle", "core", "Plot z = f(x,y) anomalies along tracks", "<DI,>XO,RG-"},
	{"psxy", "core", "Plot lines, polygons, and symbols on maps", "<DI,CCi,T-i,>XO,RG-"},
	{"psxyz", "core", "Plot lines, polygons, and symbols in 3-D", "<DI,CCi,T-i,>XO,RG-"},
	{"sample1d", "core", "Resample 1-D table data using splines", "<DI,NDi,>DO"},
	{"spectrum1d", "core", "Compute auto- [and cross-] spectra from one [or two] timeseries", "<DI,>DO,T-o"},
	{"sph2grd", "core", "Compute grid from spherical harmonic coefficients", "<DI,GGO,RG-"},
	{"sphdistance", "core", "Create Voronoi distance, node, or nearest-neighbor grid on a sphere", "<DI,NDi,QDi,GGO,RG-,Q-i"},
	{"sphinterpolate", "core", "Spherical gridding in tension of data on a sphere", "<DI,GGO,RG-"},
	{"sphtriangulate", "core", "Delaunay or Voronoi construction of spherical lon,lat data", "<DI,>DO,NDo,RG-"},
	{"splitxyz", "core", "Split xyz[dh] data tables into individual segments", "<DI,>DO"},
	{"surface", "core", "Grid table data using adjustable tension continuous curvature splines", "<DI,DDi,LGi,GGO,RG-"},
	{"trend1d", "core", "Fit a [weighted] [robust] polynomial [and/or Fourier] model for y = f(x) to xy[w] data", "<DI,>DO"},
	{"trend2d", "core", "Fit a [weighted] [robust] polynomial for z = f(x,y) to xyz[w] data", "<DI,>DO"},
	{"triangulate", "core", "Optimal (Delaunay) triangulation and gridding of Cartesian table data", "<DI,>DO,GGo,RG-"},
	{"xyz2grd", "core", "Convert data table to a grid file", "<DI,SDo,GGO,RG-"},
	{NULL, NULL, NULL, NULL} /* last element == NULL detects end of array */
#else
	{"blockmean", "core", "Block average (x,y,z) data tables by L2 norm", "<DI,>DO,RG-", &GMT_blockmean},
	{"blockmedian", "core", "Block average (x,y,z) data tables by L1 norm (spatial median)", "<DI,>DO,RG-", &GMT_blockmedian},
	{"blockmode", "core", "Block average (x,y,z) data tables by mode estimation", "<DI,>DO,RG-", &GMT_blockmode},
	{"filter1d", "core", "Time domain filtering of 1-D data tables", "<DI,>DO", &GMT_filter1d},
	{"fitcircle", "core", "Find mean position and great [or small] circle fit to points on sphere", "<DI,>TO,>DF", &GMT_fitcircle},
	{"gmt2kml", "core", "Convert GMT data tables to KML files for Google Earth", "<DI,>TO", &GMT_gmt2kml},
	{"gmtconnect", "core", "Connect individual lines whose end points match within tolerance", "<DI,>DO,CDo,LTo,QTo", &GMT_gmtconnect},
	{"gmtconvert", "core", "Convert, paste, or extract columns from data tables", "<DI,>DO", &GMT_gmtconvert},
	{"gmtdefaults", "core", "List current GMT default parameters", "", &GMT_gmtdefaults},
	{"gmtget", "core", "Get individual GMT default parameters", ">TO", &GMT_gmtget},
	{"gmtinfo", "core", "Get information about data tables", "<DI,>TO,>DC", &GMT_gmtinfo},
	{"gmtlogo", "core", "Plot the GMT logo on maps", "", &GMT_gmtlogo},
	{"gmtmath", "core", "Reverse Polish Notation (RPN) calculator for data tables", "<DI,ADi,>DO", &GMT_gmtmath},
	{"gmtread", "core", "Read GMT objects into external API", "<?I,>?O,-T-", &GMT_gmtread},
	{"gmtregress", "core", "Linear regression of 1-D data sets", "<DI,>DO", &GMT_gmtregress},
	{"gmtselect", "core", "Select data table subsets based on multiple spatial criteria", "<DI,CDi,FDi,LDi,>DO,RG-", &GMT_gmtselect},
	{"gmtset", "core", "Change individual GMT default parameters", "", &GMT_gmtset},
	{"gmtsimplify", "core", "Line reduction using the Douglas-Peucker algorithm", "<DI,>DO", &GMT_gmtsimplify},
	{"gmtspatial", "core", "Geospatial operations on lines and polygons", "<DI,DDi,NDi,TDi,>DO", &GMT_gmtspatial},
	{"gmtvector", "core", "Operations on Cartesian vectors in 2-D and 3-D", "<DI,ADi,>DO", &GMT_gmtvector},
	{"gmtwhich", "core", "Find full path to specified files", "<TI,>TO", &GMT_gmtwhich},
	{"gmtwrite", "core", "Write GMT objects from external API", "<?I,>?O,-T-", &GMT_gmtwrite},
	{"grd2cpt", "core", "Make linear or histogram-equalized color palette table from grid", "<GI,>CO,RG-", &GMT_grd2cpt},
	{"grd2rgb", "core", "Write r/g/b grid files from a grid file, a raw RGB file, or SUN rasterfile", "<GI,RG-", &GMT_grd2rgb},
	{"grd2xyz", "core", "Convert grid file to data table", "<GI,>DO,RG-", &GMT_grd2xyz},
	{"grdblend", "core", "Blend several partially over-lapping grids into one larger grid", "<TI,GGO,RG-", &GMT_grdblend},
	{"grdclip", "core", "Clip the range of grid values", "<GI,GGO,RG-", &GMT_grdclip},
	{"grdcontour", "core", "Make contour map using a grid", "<GI,CCi,DDD,>XO,RG-", &GMT_grdcontour},
	{"grdconvert", "core", "Convert between different grid formats", "<GI,>GO,RG-", &GMT_grdconvert},
	{"grdcut", "core", "Extract subregion from a grid", "<GI,GGO,RG-", &GMT_grdcut},
	{"grdedit", "core", "Modify header or content of a grid", "<GI,NDi,GGO,RG-", &GMT_grdedit},
	{"grdfft", "core", "Mathematical operations on grids in the wavenumber (or frequency) domain", "<GI,GGO,RG-,GDE", &GMT_grdfft},
	{"grdfilter", "core", "Filter a grid in the space (or time) domain", "<GI,GGO,RG-", &GMT_grdfilter},
	{"grdgradient", "core", "Compute directional gradients from a grid", "<GI,GGO,SGo,RG-", &GMT_grdgradient},
	{"grdhisteq", "core", "Perform histogram equalization for a grid", "<GI,GGO,DTo,RG-", &GMT_grdhisteq},
	{"grdimage", "core", "Project grids or images and plot them on maps", "<GI,CCi,IGi,>XO,RG-,AIA", &GMT_grdimage},
	{"grdinfo", "core", "Extract information from grids", "<GI,>TO,>DC,RG-", &GMT_grdinfo},
	{"grdlandmask", "core", "Create a \"wet-dry\" mask grid from shoreline data base", "GGO,RG-", &GMT_grdlandmask},
	{"grdmask", "core", "Create mask grid from polygons or point coverage", "<DI,GGORGi", &GMT_grdmask},
	{"grdmath", "core", "Reverse Polish Notation (RPN) calculator for grids (element by element)", "<GI,=GO,RG-", &GMT_grdmath},
	{"grdpaste", "core", "Join two grids along their common edge", "<GI,GGO,RG-", &GMT_grdpaste},
	{"grdproject", "core", "Forward and inverse map transformation of grids", "<GI,GGO,RG-", &GMT_grdproject},
	{"grdraster", "core", "Extract subregion from a binary raster and save as a GMT grid", "GGO,TDo,RG-", &GMT_grdraster},
	{"grdsample", "core", "Resample a grid onto a new lattice", "<GI,GGO,RG-", &GMT_grdsample},
	{"grdtrack", "core", "Sample grids at specified (x,y) locations", "<DI,DDo,GGi,>DO,RG-", &GMT_grdtrack},
	{"grdtrend", "core", "Fit trend surface to grids and compute residuals", "<GI,DGo,TGo,WGo,RG-", &GMT_grdtrend},
	{"grdvector", "core", "Plot vector field from two component grids", "<GI,CCi,>XO,RG-", &GMT_grdvector},
	{"grdview", "core", "Create 3-D perspective image or surface mesh from a grid", "<GI,CCi,GGi,IGi,>XO,RG-", &GMT_grdview},
	{"grdvolume", "core", "Calculate grid volume and area constrained by a contour", "<GI,>DO,RG-", &GMT_grdvolume},
	{"greenspline", "core", "Interpolate using Green's functions for splines in 1-3 dimensions", "<DI,ADi,NDi,TGi,CDo,GGO,RG-", &GMT_greenspline},
	{"kml2gmt", "core", "Extract GMT table data from Google Earth KML files", "<TI,>DO", &GMT_kml2gmt},
	{"makecpt", "core", "Make GMT color palette tables", ">CO", &GMT_makecpt},
	{"mapproject", "core", "Forward and inverse map transformations, datum conversions and geodesy", "<DI,LDi,>DO,W-i,RG-", &GMT_mapproject},
	{"nearneighbor", "core", "Grid table data using a \"Nearest neighbor\" algorithm", "<DI,GGO,RG-", &GMT_nearneighbor},
	{"project", "core", "Project table data onto lines or great circles, generate tracks, or translate coordinates", "<DI,>DO,G-i", &GMT_project},
	{"psbasemap", "core", "Plot base maps and frames", ">XO,RG-", &GMT_psbasemap},
	{"psclip", "core", "Initialize or terminate polygonal clip paths", "<DI,>XO", &GMT_psclip},
	{"pscoast", "core", "Plot continents, countries, shorelines, rivers, and borders on maps", ">XO,>DM,RG-", &GMT_pscoast},
	{"pscontour", "core", "Contour table data by direct triangulation", "<DI,CCi,EDi,DDD,>XO,RG-", &GMT_pscontour},
	{"psconvert", "core", "Convert [E]PS file(s) to other formats using GhostScript", "FIo", &GMT_psconvert},
	{"pshistogram", "core", "Calculate and plot histograms", "<DI,>XO,>Do", &GMT_pshistogram},
	{"psimage", "core", "Place images or EPS files on maps", "<II,>XO,RG-", &GMT_psimage},
	{"pslegend", "core", "Plot legends on maps", "<TI,ACi,>XO,RG-", &GMT_pslegend},
	{"psmask", "core", "Clip or mask map areas with no data table coverage", "<DI,DDD,C-i,>XO,RG-", &GMT_psmask},
	{"psrose", "core", "Plot a polar histogram (rose, sector, windrose diagrams)", "<DI,>XO", &GMT_psrose},
	{"psscale", "core", "Plot a gray-scale or color-scale on maps", "CCI,>XO", &GMT_psscale},
	{"pstext", "core", "Plot or typeset text on maps", "<TI,>XO,RG-", &GMT_pstext},
	{"pswiggle", "core", "Plot z = f(x,y) anomalies along tracks", "<DI,>XO,RG-", &GMT_pswiggle},
	{"psxy", "core", "Plot lines, polygons, and symbols on maps", "<DI,CCi,T-i,>XO,RG-", &GMT_psxy},
	{"psxyz", "core", "Plot lines, polygons, and symbols in 3-D", "<DI,CCi,T-i,>XO,RG-", &GMT_psxyz},
	{"sample1d", "core", "Resample 1-D table data using splines", "<DI,NDi,>DO", &GMT_sample1d},
	{"spectrum1d", "core", "Compute auto- [and cross-] spectra from one [or two] timeseries", "<DI,>DO,T-o", &GMT_spectrum1d},
	{"sph2grd", "core", "Compute grid from spherical harmonic coefficients", "<DI,GGO,RG-", &GMT_sph2grd},
	{"sphdistance", "core", "Create Voronoi distance, node, or nearest-neighbor grid on a sphere", "<DI,NDi,QDi,GGO,RG-,Q-i", &GMT_sphdistance},
	{"sphinterpolate", "core", "Spherical gridding in tension of data on a sphere", "<DI,GGO,RG-", &GMT_sphinterpolate},
	{"sphtriangulate", "core", "Delaunay or Voronoi construction of spherical lon,lat data", "<DI,>DO,NDo,RG-", &GMT_sphtriangulate},
	{"splitxyz", "core", "Split xyz[dh] data tables into individual segments", "<DI,>DO", &GMT_splitxyz},
	{"surface", "core", "Grid table data using adjustable tension continuous curvature splines", "<DI,DDi,LGi,GGO,RG-", &GMT_surface},
	{"trend1d", "core", "Fit a [weighted] [robust] polynomial [and/or Fourier] model for y = f(x) to xy[w] data", "<DI,>DO", &GMT_trend1d},
	{"trend2d", "core", "Fit a [weighted] [robust] polynomial for z = f(x,y) to xyz[w] data", "<DI,>DO", &GMT_trend2d},
	{"triangulate", "core", "Optimal (Delaunay) triangulation and gridding of Cartesian table data", "<DI,>DO,GGo,RG-", &GMT_triangulate},
	{"xyz2grd", "core", "Convert data table to a grid file", "<DI,SDo,GGO,RG-", &GMT_xyz2grd},
	{NULL, NULL, NULL, NULL, NULL} /* last element == NULL detects end of array */
#endif
};

/* Pretty print all GMT core module names and their purposes */
void gmt_core_module_show_all (void *V_API) {
	unsigned int module_id = 0;
	char message[256];
	struct GMTAPI_CTRL *API = gmt_get_api_ptr (V_API);
	GMT_Message (V_API, GMT_TIME_NONE, "\n===  GMT core: The main modules of the Generic Mapping Tools  ===\n");
	while (g_core_module[module_id].name != NULL) {
		if (module_id == 0 || strcmp (g_core_module[module_id-1].component, g_core_module[module_id].component)) {
			/* Start of new supplemental group */
			sprintf (message, "\nModule name:     Purpose of %s module:\n", g_core_module[module_id].component);
			GMT_Message (V_API, GMT_TIME_NONE, message);
			GMT_Message (V_API, GMT_TIME_NONE, "----------------------------------------------------------------\n");
		}
		if (API->mode || (strcmp (g_core_module[module_id].name, "gmtread") && strcmp (g_core_module[module_id].name, "gmtwrite"))) {
			sprintf (message, "%-16s %s\n",
				g_core_module[module_id].name, g_core_module[module_id].purpose);
				GMT_Message (V_API, GMT_TIME_NONE, message);
		}
		++module_id;
	}
}

/* Produce single list on stdout of all GMT core module names for gmt --show-modules */
void gmt_core_module_list_all (void *V_API) {
	unsigned int module_id = 0;
	struct GMTAPI_CTRL *API = gmt_get_api_ptr (V_API);
	while (g_core_module[module_id].name != NULL) {
		if (API->mode || (strcmp (g_core_module[module_id].name, "gmtread") && strcmp (g_core_module[module_id].name, "gmtwrite")))
			printf ("%s\n", g_core_module[module_id].name);
		++module_id;
	}
}

/* Lookup module id by name, return option keys pointer (for external API developers) */
const char *gmt_core_module_info (void *API, char *candidate) {
	int module_id = 0;
	GMT_UNUSED(API);

	/* Match actual_name against g_module[module_id].name */
	while (g_core_module[module_id].name != NULL &&
	       strcmp (candidate, g_core_module[module_id].name))
		++module_id;

	/* Return Module keys or NULL */
	return (g_core_module[module_id].keys);
}
	
#ifndef BUILD_SHARED_LIBS
/* Lookup module id by name, return function pointer */
void *gmt_core_module_lookup (void *API, const char *candidate) {
	int module_id = 0;
	GMT_UNUSED(API);

	/* Match actual_name against g_module[module_id].name */
	while (g_core_module[module_id].name != NULL &&
	       strcmp (candidate, g_core_module[module_id].name))
		++module_id;

	/* Return Module function or NULL */
	return (g_core_module[module_id].p_func);
}
#endif
