/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file ab_local.c
 * \brief local address book plugin
 */

#include <ffgtk.h>
#include <addressbook.h>

/** Local address book name */
static char *pnLocalName = NULL;

/**
 * \brief Set local address book name
 * \param nType 0 = csv, 1 = vcf
 */
static void setLocalName( gint nType ) {
	const gchar *pnHomeDir;
	size_t nLen;

	if ( pnLocalName != NULL ) {
		return;
	}

	pnHomeDir = g_get_home_dir();

	if ( nType == 0 ) {
		nLen = strlen( pnHomeDir ) + strlen( ".ffgtk/ab_local.csv" ) + 2;
		pnLocalName = g_malloc( nLen );
		if ( pnLocalName != NULL ) {
			snprintf( pnLocalName, nLen, "%s/%s", pnHomeDir, ".ffgtk/ab_local.csv" );
		}
	} else {
		nLen = strlen( pnHomeDir ) + strlen( ".ffgtk/ab_local.vcf" ) + 2;
		pnLocalName = g_malloc( nLen );
		if ( pnLocalName != NULL ) {
			snprintf( pnLocalName, nLen, "%s/%s", pnHomeDir, ".ffgtk/ab_local.vcf" );
		}
	}
}

/**
 * \brief Read address book
 * \return error code
 */
static int localReadBook( void ) {
	GList *psList = NULL;
	struct sPerson *psPerson = NULL;
	gint nRet = 0;

	/* First we try to load the old address book */
	if ( pnLocalName == NULL ) {
		setLocalName( 0 );
		if ( pnLocalName != NULL ) {
			/* read it */
			nRet = parseCsvFile( pnLocalName, getActiveProfile() -> pnName );

			/* If file exists, rename it */
			if ( nRet != -1 ) {
				gchar *pnTmpName = g_strdup_printf( "%s.OLD", pnLocalName );
				rename( pnLocalName, pnTmpName );
				g_free( pnTmpName );
			}

			/* Free local name */
			g_free( pnLocalName );
			pnLocalName = NULL;

			/* if we succeeded, return now */
			if ( nRet == 0 ) {
				for ( psList = psPersonsList; psList != NULL; psList = psList -> next ) {
					psPerson = psList -> data;

					if ( psPerson != NULL ) {
						Debug( KERN_DEBUG, "'%s'\n", psPerson -> pnDisplayName );
						psPerson -> nFlags |= PERSON_FLAG_NEW;
					}
				}
				SaveBook();

				return 0;
			}
		}
	}

	/* Old book not present, try new vcard */
	if ( pnLocalName == NULL ) {
		setLocalName( 1 );
	}

	if ( pnLocalName != NULL ) {
		loadCardFile( pnLocalName );
	}

	return 0;
}

/**
 * \brief Save address book
 * \return error code
 */
static int localSaveBook( void ) {
	//if ( pnLocalName == NULL ) {
		setLocalName( 1 );
	//}

	if ( pnLocalName != NULL ) {
		writeCardFile( pnLocalName );
	}
	return 0;
}


/** book definition */
struct sAddressBook sLocal = {
	localReadBook,
	localSaveBook,
	PERSON_FIRST_NAME | PERSON_LAST_NAME | PERSON_DISPLAY_NAME | PERSON_COMPANY |
	PERSON_TITLE | PERSON_IMAGE | PERSON_PRIVATE_PHONE | PERSON_PRIVATE_MOBILE |
	PERSON_PRIVATE_FAX | PERSON_PRIVATE_STREET | PERSON_PRIVATE_CITY |
	PERSON_PRIVATE_ZIPCODE | PERSON_PRIVATE_COUNTRY | PERSON_BUSINESS_PHONE |
	PERSON_BUSINESS_FAX | PERSON_BUSINESS_STREET | PERSON_BUSINESS_CITY |
	PERSON_BUSINESS_ZIPCODE | PERSON_BUSINESS_COUNTRY,
	PERSON_FIRST_NAME | PERSON_LAST_NAME | PERSON_DISPLAY_NAME | PERSON_COMPANY |
	PERSON_TITLE | PERSON_IMAGE | PERSON_PRIVATE_PHONE | PERSON_PRIVATE_MOBILE |
	PERSON_PRIVATE_FAX | PERSON_PRIVATE_STREET | PERSON_PRIVATE_CITY |
	PERSON_PRIVATE_ZIPCODE | PERSON_PRIVATE_COUNTRY | PERSON_BUSINESS_PHONE |
	PERSON_BUSINESS_FAX | PERSON_BUSINESS_STREET | PERSON_BUSINESS_CITY |
	PERSON_BUSINESS_ZIPCODE | PERSON_BUSINESS_COUNTRY,
	NULL
};

MODULE_INIT( PLUGIN_TYPE_BOOK, _( "Standard" ), &sLocal, NULL, NULL );
