/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file checklist.c
 * \brief Checklist widget
 */

#include <gtk/gtk.h>

/** Checklist states */
enum eCheckListStageState {
	STAGE_COMPLETED,
	STAGE_IN_PROGRESS,
	STAGE_PENDING
};

/** Check list details - num stages, current, progressbar */
struct sCheckListDetails {
	guint nNumStages;
	guint nStagesCompleted;
	GList *psListOfStages;
	GtkWidget *psBar;
};

/** Description of check list stage */
struct sCheckListStage {
	GtkLabel *psDescriptiveLabel;
	gchar *pnName;
};

/** Tick char */
static gunichar2 anTick[ 2 ] = { 0x2713, 0 };
/** Arrow char */
static gunichar2 anArrow[ 2 ] = { 0x25B8, 0 };

/**
 * \brief Setup text info for given stage
 * \param psStage stage pointer
 * \param nState describing state
 */
static void setupTextForStage( struct sCheckListStage *psStage, enum eCheckListStageState nState ) {
	g_return_if_fail( psStage );

	if ( nState == STAGE_IN_PROGRESS ) {
		gchar *pnArrowUtf8 = g_utf16_to_utf8( anArrow, -1, NULL, NULL, NULL );
		gchar *pnText = g_strdup_printf( "%s <b>%s</b>", pnArrowUtf8, psStage -> pnName );
		gtk_label_set_markup( psStage -> psDescriptiveLabel, pnText );
		g_free( pnText );
		g_free( pnArrowUtf8 );
	} else if ( nState == STAGE_COMPLETED ) {
		gchar *pnTickUtf8 = g_utf16_to_utf8( anTick, -1, NULL, NULL, NULL );
		gchar *pnText = g_strdup_printf( "%s %s", pnTickUtf8, psStage -> pnName );
		gtk_label_set_markup( psStage -> psDescriptiveLabel, pnText );
		g_free( pnText );
		g_free( pnTickUtf8 );
	} else {
		gtk_label_set_markup( psStage -> psDescriptiveLabel, psStage -> pnName );
	}
}

/**
 * \brief Get state of stage
 * \param nIndex state index
 * \param nNumStagesCompleted number of completed stages
 * \return stage status
 */
static enum eCheckListStageState getStateForStage( guint nIndex, guint nNumStagesCompleted ) {
	if ( nIndex < nNumStagesCompleted ) {
		return STAGE_COMPLETED;
	} else if ( nIndex == nNumStagesCompleted ) {
		return STAGE_IN_PROGRESS;
	} else {
		return STAGE_PENDING;
	}
}

/**
 * \brief Refresh stages details
 * \param psDetails check list details
 */
static void refreshStages( struct sCheckListDetails *psDetails ) {
	guint nIndex = 0;
	GList *psIter;

	g_return_if_fail( psDetails );

	for ( psIter = psDetails -> psListOfStages; psIter != NULL; psIter = psIter -> next ) {
		setupTextForStage( psIter -> data, getStateForStage( nIndex, psDetails -> nStagesCompleted ) );

		nIndex++;
	}
}

/**
 * \brief Add new stage to checklist
 * \param psCheckList checklist pointer
 * \param pnName new stage name
 */
void checklistAddStage( GtkWidget *psCheckList, const gchar *pnName ) {
	struct sCheckListDetails *psDetails;
	struct sCheckListStage *psStage;

	g_return_if_fail( psCheckList );
	g_return_if_fail( pnName );

	psDetails = g_object_get_data( G_OBJECT( psCheckList ), "details" );

	psStage = g_new0( struct sCheckListStage, 1 );

	psStage -> pnName = g_strdup( pnName );

	psStage -> psDescriptiveLabel = GTK_LABEL( gtk_label_new( NULL ) );
	gtk_misc_set_alignment( GTK_MISC( psStage -> psDescriptiveLabel ), 0, 0.5 );
	gtk_label_set_use_markup( psStage -> psDescriptiveLabel, TRUE) ;

	gtk_table_attach( GTK_TABLE( psCheckList ), GTK_WIDGET( psStage -> psDescriptiveLabel ), 1, 2, psDetails -> nNumStages, psDetails -> nNumStages + 1, GTK_FILL | GTK_EXPAND, 0, 0, 3 );

	psDetails -> nNumStages++;
	psDetails -> psListOfStages = g_list_append( psDetails -> psListOfStages, psStage );

	refreshStages( psDetails );

	gtk_widget_show( GTK_WIDGET( psStage -> psDescriptiveLabel ) );
}

/**
 * \brief Add progressbar to checklist
 * \param psCheckList checklist pointer
 */
void checklistAddProgressbar( GtkWidget *psCheckList ) {
	struct sCheckListDetails *psDetails;

	g_return_if_fail( psCheckList );

	psDetails = g_object_get_data( G_OBJECT( psCheckList ), "details" );

	psDetails -> psBar = gtk_progress_bar_new();
	gtk_table_attach( GTK_TABLE( psCheckList ), GTK_WIDGET( psDetails -> psBar ), 0, 3, psDetails -> nNumStages, psDetails -> nNumStages + 1, GTK_FILL, 0, 0, 5 );

	gtk_widget_show( GTK_WIDGET( psDetails -> psBar ) );
}

/**
 * \brief Complete current checklist stage
 * \param psCheckList checklist pointer
 * \return TRUE on success, else FALSE
 */
gboolean checklistCompleteStage( GtkWidget *psCheckList ) {
	struct sCheckListDetails *psDetails;

	g_return_val_if_fail( psCheckList, FALSE );

	psDetails = g_object_get_data( G_OBJECT( psCheckList ), "details" );
	g_assert( psDetails );

	psDetails -> nStagesCompleted++;
	refreshStages( psDetails );
	gtk_progress_bar_set_fraction( GTK_PROGRESS_BAR( psDetails -> psBar ), 1.0 * psDetails -> nStagesCompleted / psDetails -> nNumStages );

	if ( psDetails -> nStagesCompleted == psDetails -> nNumStages ) {
		return FALSE;
	}

	return TRUE;
}

/**
 * \brief Reset checklist structure
 * \param psCheckList checklist pointer
 */
void checklistReset( GtkWidget *psCheckList ) {
	struct sCheckListDetails *psDetails;

	g_return_if_fail( psCheckList );

	psDetails = g_object_get_data( G_OBJECT( psCheckList ), "details" );
	g_assert( psDetails );

	psDetails -> nStagesCompleted = 0;

	refreshStages( psDetails );
	gtk_progress_bar_set_fraction( GTK_PROGRESS_BAR( psDetails -> psBar ), 1.0 * psDetails -> nStagesCompleted / psDetails -> nNumStages );
	gtk_progress_bar_set_text( GTK_PROGRESS_BAR( psDetails -> psBar ), "" );
}

/**
 * \brief Set error in checklist
 * \param psCheckList checklist pointer
 * \param pnError errror text
 */
void checklistSetError( GtkWidget *psCheckList, const gchar *pnError ) {
	struct sCheckListDetails *psDetails;

	g_return_if_fail( psCheckList );

	psDetails = g_object_get_data( G_OBJECT( psCheckList ), "details" );
	g_assert( psDetails );

	if ( psDetails -> psBar != NULL ) {
		gtk_progress_bar_set_fraction( GTK_PROGRESS_BAR( psDetails -> psBar ), 1.0 );
		gtk_progress_bar_set_text( GTK_PROGRESS_BAR( psDetails -> psBar ), pnError );
	}
}

/**
 * \brief Create new checklist structure
 * \return table containing checklist structure
 */
GtkWidget *checklistNew( void ) {
	GtkTable *psTable = GTK_TABLE( gtk_table_new( 0, 3, FALSE ) );
	struct sCheckListDetails *psDetails = g_new0( struct sCheckListDetails, 1 );

	gtk_table_set_col_spacings( psTable, 6 );

	g_object_set_data( G_OBJECT( psTable ), "details", psDetails );

	return GTK_WIDGET( psTable );
}
