/*
 * Copyright (C) 2007-2012 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <inttypes.h>
#include <setjmp.h>
#include <stdbool.h>
#include <stdio.h>
#include <string.h>

#include "glue.h"

#include "chip_intel_80386_int.h"

#undef NAME
#undef NAME_

#define CHIP_(x)	chip_intel_80386_int_ ## x

struct cpssp {
	/* Config */

	/* Ports */
	struct sig_host_bus_main *host_bus_main;

	/* Signals */

	/* State */
	int state_interrupt_pending;
	int state_nmi_pending;
	int state_power;
	int state_n_reset;
	int state_n_init;

#define STATE
#define NAME		mmu
#define NAME_(x)	mmu_ ## x
#include "arch_gen_cpu_x86_mmu_int.c"
#undef NAME_
#undef NAME
#define NAME		core
#define NAME_(x)	core_ ## x
#include "arch_gen_cpu_x86_core_int.c"
#undef NAME_
#undef NAME
#undef STATE

	/* Processes */
	struct process process;
};

static struct cpssp *cpssp;

static uint32_t
CHIP_(ior)(struct cpssp *cpssp, uint16_t port, unsigned int bs)
{
	uint32_t val;

	val = -1;
	if (unlikely(sig_host_bus_ior(cpssp->host_bus_main, cpssp, port, bs, &val) != 0)) {
		sig_host_bus_type_addr(cpssp->host_bus_main, cpssp,
				0, SIG_HOST_BUS_IOR, port);
		/* delay... */
		sig_host_bus_read_data(cpssp->host_bus_main, cpssp,
				bs, &val);
	}
	return val;
}

static void
CHIP_(iow)(struct cpssp *cpssp, uint16_t port, unsigned int bs, uint32_t val)
{
	if (unlikely(sig_host_bus_iow(cpssp->host_bus_main, cpssp, port, bs, val) != 0)) {
		sig_host_bus_type_addr(cpssp->host_bus_main, cpssp,
				0, SIG_HOST_BUS_IOW, port);
		/* delay... */
		sig_host_bus_write_data(cpssp->host_bus_main, cpssp,
				bs, val);
	}
}

static void
CHIP_(mr)(unsigned long addr, unsigned int bs, uint32_t *valp)
{
	if (unlikely(sig_host_bus_mr(cpssp->host_bus_main, cpssp,
			0, addr, bs, valp) != 0)) {
		sig_host_bus_type_addr(cpssp->host_bus_main, cpssp,
				0, SIG_HOST_BUS_MR, addr);
		/* delay... */
		sig_host_bus_read_data(cpssp->host_bus_main, cpssp,
				bs, valp);
	}
}

static void
CHIP_(mw)(unsigned long addr, unsigned int bs, uint32_t val)
{
	if (unlikely(sig_host_bus_mw(cpssp->host_bus_main, cpssp,
			0, addr, bs, val) != 0)) {
		sig_host_bus_type_addr(cpssp->host_bus_main, cpssp,
				0, SIG_HOST_BUS_MW, addr);
		/* delay... */
		sig_host_bus_write_data(cpssp->host_bus_main, cpssp,
				bs, val);
	}
}

/*forward*/ static void
mmu_vmr(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	bool user,
	uint32_t *valp,
	uint8_t *errp);

static void
CHIP_(vmr)(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	bool user,
	uint32_t *valp,
	uint8_t *errp
)
{
	mmu_vmr(cpssp, addr, bs, user, valp, errp);
}

/*forward*/ static void
mmu_vmw(
	struct cpssp *cpssp,
	uint32_t addr, 
	unsigned int bs, 
	bool user,
	uint32_t val,
	uint8_t *errp);

static void
CHIP_(vmw)(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	bool user,
	uint32_t val,
	uint8_t *errp
)
{
	mmu_vmw(cpssp, addr, bs, user, val, errp);
}

static uint8_t
CHIP_(ack)(struct cpssp *cpssp)
{
	uint8_t vec;

	sig_host_bus_inta_addr(cpssp->host_bus_main, cpssp);
	sig_host_bus_inta_data(cpssp->host_bus_main, cpssp, &vec);

	return vec;
}

/*forward*/ static void
mmu_tlb_flush(struct cpssp *cpssp);

static void
CHIP_(tlb_flush)(struct cpssp *cpssp)
{
	mmu_tlb_flush(cpssp);
}

/*forward*/ static void
align_uiorb(struct cpssp *cpssp, uint32_t addr, uint8_t *valp);

static void
CHIP_(uiorb)(struct cpssp *cpssp, uint32_t addr, uint8_t *valp)
{
	align_uiorb(cpssp, addr, valp);
}

/*forward*/ static void
align_uiorw(struct cpssp *cpssp, uint32_t addr, uint16_t *valp);

static void
CHIP_(uiorw)(struct cpssp *cpssp, uint32_t addr, uint16_t *valp)
{
	align_uiorw(cpssp, addr, valp);
}

/*forward*/ static void
align_uiord(struct cpssp *cpssp, uint32_t addr, uint32_t *valp);

static void
CHIP_(uiord)(struct cpssp *cpssp, uint32_t addr, uint32_t *valp)
{
	align_uiord(cpssp, addr, valp);
}

/*forward*/ static void
align_uiowb(struct cpssp *cpssp, uint32_t addr, uint8_t value);

static void
CHIP_(uiowb)(struct cpssp *cpssp, uint32_t addr, uint8_t value)
{
	align_uiowb(cpssp, addr, value);
}

/*forward*/ static void
align_uioww(struct cpssp *cpssp, uint32_t addr, uint16_t value);

static void
CHIP_(uioww)(struct cpssp *cpssp, uint32_t addr, uint16_t value)
{
	align_uioww(cpssp, addr, value);
}

/*forward*/ static void
align_uiowd(struct cpssp *cpssp, uint32_t addr, uint32_t value);

static void
CHIP_(uiowd)(struct cpssp *cpssp, uint32_t addr, uint32_t value)
{
	align_uiowd(cpssp, addr, value);
}

/*forward*/ static void
align_umrb(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint8_t *valp,
	uint8_t *merrp);

static void
CHIP_(umrb)(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint8_t *valp,
	uint8_t *merrp
)
{
	align_umrb(cpssp, addr, user, valp, merrp);
}

/*forward*/ static void
align_umrw(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint16_t *valp,
	uint8_t *merrp);

static void
CHIP_(umrw)(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint16_t *valp,
	uint8_t *merrp
)
{
	align_umrw(cpssp, addr, user, valp, merrp);
}

/*forward*/ static void
align_umrd(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint32_t *valp,
	uint8_t *merrp);

static void
CHIP_(umrd)(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint32_t *valp,
	uint8_t *merrp
)
{
	align_umrd(cpssp, addr, user, valp, merrp);
}

/*forward*/ static void
align_umwb(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint8_t val,
	uint8_t *merrp);

static void
CHIP_(umwb)(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint8_t val,
	uint8_t *merrp
)
{
	align_umwb(cpssp, addr, user, val, merrp);
}

/*forward*/ static void
align_umww(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint16_t val,
	uint8_t *merrp);

static void
CHIP_(umww)(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint16_t val,
	uint8_t *merrp
)
{
	align_umww(cpssp, addr, user, val, merrp);
}

/*forward*/ static void
align_umwd(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint32_t val,
	uint8_t *merrp);

static void
CHIP_(umwd)(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint32_t val,
	uint8_t *merrp
)
{
	align_umwd(cpssp, addr, user, val, merrp);
}

/*forward*/ static void
seg_mrb(
	struct cpssp *cpssp,
	uint8_t seg,
	uint32_t addr,
	uint8_t *valp,
	bool *serrp,
	uint8_t *merrp);

static void
CHIP_(smrb)(
	struct cpssp *cpssp,
	int seg,
	uint32_t addr,
	uint8_t *valp,
	bool *serrp,
	uint8_t *merrp
)
{
	seg_mrb(cpssp, seg, addr, valp, serrp, merrp);
}

/*forward*/ static void
seg_mrw(
	struct cpssp *cpssp,
	uint8_t seg,
	uint32_t addr,
	uint16_t *valp,
	bool *serrp,
	uint8_t *merrp);

static void
CHIP_(smrw)(
	struct cpssp *cpssp,
	int seg,
	uint32_t addr,
	uint16_t *valp,
	bool *serrp,
	uint8_t *merrp
)
{
	seg_mrw(cpssp, seg, addr, valp, serrp, merrp);
}

/*forward*/ static void
seg_mrd(
	struct cpssp *cpssp,
	uint8_t seg,
	uint32_t addr,
	uint32_t *valp,
	bool *serrp,
	uint8_t *merrp);

static void
CHIP_(smrd)(
	struct cpssp *cpssp,
	int seg,
	uint32_t addr,
	uint32_t *valp,
	bool *serrp,
	uint8_t *merrp
)
{
	seg_mrd(cpssp, seg, addr, valp, serrp, merrp);
}

/*forward*/ static void
seg_mwb(
	struct cpssp *cpssp,
	uint8_t seg,
	uint32_t addr,
	uint8_t val,
	bool *serrp,
	uint8_t *merrp);

static void
CHIP_(smwb)(
	struct cpssp *cpssp,
	int seg,
	uint32_t addr,
	uint8_t val,
	bool *serrp,
	uint8_t *merrp
)
{
	seg_mwb(cpssp, seg, addr, val, serrp, merrp);
}

/*forward*/ static void
seg_mww(
	struct cpssp *cpssp,
	uint8_t seg,
	uint32_t addr,
	uint16_t val,
	bool *serrp,
	uint8_t *merrp);

static void
CHIP_(smww)(
	struct cpssp *cpssp,
	int seg,
	uint32_t addr,
	uint16_t val,
	bool *serrp,
	uint8_t *merrp
)
{
	seg_mww(cpssp, seg, addr, val, serrp, merrp);
}

/*forward*/ static void
seg_mwd(
	struct cpssp *cpssp,
	uint8_t seg,
	uint32_t addr,
	uint32_t val,
	bool *serrp,
	uint8_t *merrp);

static void
CHIP_(smwd)(
	struct cpssp *cpssp,
	int seg,
	uint32_t addr,
	uint32_t val,
	bool *serrp,
	uint8_t *merrp
)
{
	seg_mwd(cpssp, seg, addr, val, serrp, merrp);
}

#define BEHAVIOR
#define NAME		mmu
#define NAME_(x)	mmu_ ## x
#include "arch_gen_cpu_x86_mmu_int.c"
#undef NAME_
#undef NAME
#define NAME		align
#define NAME_(x)	align_ ## x
#include "arch_gen_cpu_x86_align_int.c"
#undef NAME_
#undef NAME
#define NAME		seg
#define NAME_(x)	seg_ ## x
#include "arch_gen_cpu_x86_seg_int.c"
#undef NAME_
#undef NAME
#define NAME		core
#define NAME_(x)	core_ ## x
#include "arch_gen_cpu_x86_core_int.c"
#undef NAME_
#undef NAME
#undef BEHAVIOR

void __attribute__((__noreturn__))
CHIP_(step)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

again:  ;
	sched_to_scheduler();

	while (cpssp->process.inst_cnt < cpssp->process.inst_limit) {
		if (! cpssp->state_power) {
			cpssp->process.inst_cnt = cpssp->process.inst_limit;

		} else {
			core_step(cpssp);
			cpssp->process.inst_cnt += 1;
		}
	}
	goto again;
}

static void
CHIP_(irq_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	cpssp->state_interrupt_pending = val;
}

static void
CHIP_(nmi_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	cpssp->state_nmi_pending = val;
}

static void
CHIP_(power_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	cpssp->state_power = val;
}

static void
CHIP_(n_reset_set)(void *_cpssp, unsigned int n_val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (n_val) {
		/* Reset gone. */
		cpssp->state_n_reset |= 1;
	} else {
		/* New reset. */
		cpssp->state_n_reset = 0;
	}
}

void *
CHIP_(create)(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_boolean *port_power,
	struct sig_boolean *port_n_reset,
	struct sig_boolean_or *port_irq,
	struct sig_boolean_or *port_nmi,
	struct sig_boolean *port_smi,
	struct sig_boolean_or *port_n_ferr,
	struct sig_boolean *port_n_ignne,
	struct sig_boolean *port_a20,
	struct sig_host_bus_main *port_bus
)
{
	static const struct sig_boolean_or_funcs irq_func = {
		.set = CHIP_(irq_set),
	};
	static const struct sig_boolean_or_funcs nmi_func = {
		.set = CHIP_(nmi_set),
	};
	static const struct sig_boolean_funcs power_funcs = {
		.set = CHIP_(power_set),
	};
	static const struct sig_boolean_funcs n_reset_funcs = {
		.set = CHIP_(n_reset_set),
	};

	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	memset(cpssp, 0, sizeof(*cpssp));

	/* Running a 20MHz CPU - FIXME */
	cpssp->process.inst_hz = 20*1000*1000;

	cpssp->host_bus_main = port_bus;

	sig_boolean_or_connect_in(port_irq, cpssp, &irq_func);
	sig_boolean_or_connect_in(port_nmi, cpssp, &nmi_func);
	sig_boolean_connect_in(port_power, cpssp, &power_funcs);
	sig_boolean_connect_in(port_n_reset, cpssp, &n_reset_funcs);

	sched_process_init(&cpssp->process, CHIP_(step), cpssp);

	return cpssp;
}

void
CHIP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}


void
CHIP_(suspend)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
CHIP_(resume)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fComp);
}
