/*
 * Copyright (C) 2008 Instituto Nokia de Tecnologia. All rights reserved.
 *
 * This file is part of QZion.
 *
 * QZion is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * QZion is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with QZion.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <QApplication>

#include "qzionmacros.h"
#include "qziontext.h"


class QZionTextPrivate
{
public:
    QZionText *owner;

    QZionTextPrivate(const QString &text, const QFont &font=QApplication::font(),
                     const QColor &color=Qt::black)
        : text(text), font(font), color(color), alignment(QPointF(0.5, 0.5))
    {
        multiline = false;
        style = QZionText::NoStyle;
        calcBoundingRect();
    }

    QZionText::TextStyle style;
    QString text;
    QFont font;
    QColor color;
    QColor color2;
    QColor color3;
    QRect boundingRect;
    QSize size;
    QSize maxSize;
    bool multiline;
    QPointF alignment;
    int lineAscent;
    int lineDescent;

    inline void calcBoundingRect();
    inline QRect realBoundingRect() const;
    inline void refresh(const bool calculateBound=false);
};

inline void QZionTextPrivate::refresh(const bool calculateBound)
{
    if (calculateBound)
        calcBoundingRect();
    if (owner->visible() && owner->canvas())
        owner->changed();
}

inline void QZionTextPrivate::calcBoundingRect()
{
    if (!multiline) {
        boundingRect = QFontMetrics(font).boundingRect(0, 0, size.width(),
                                                       size.height(),
                                                       Qt::TextSingleLine |
                                                       Qt::AlignHCenter, text);
    } else {
        QRect r = QRect(QPoint(0, 0), size);
        boundingRect = QFontMetrics(font).boundingRect(r, Qt::TextWordWrap, text);
    }
    lineAscent = QFontMetrics(font).ascent();
    lineDescent = QFontMetrics(font).descent();
}

inline QRect QZionTextPrivate::realBoundingRect() const
{
    int real_width;
    int real_height;
    int real_x;
    int real_y;

    // Does the bounding rect fit in the available width ?
    if (boundingRect.width() < size.width()) {
        real_width = boundingRect.width();
        real_x = alignment.x() * (size.width() - real_width);
    } else {
        real_width = size.width();
        real_x = 0;
    }

    // Does the bounding rect fit in the available height ?
    if (boundingRect.height() < size.height()) {
        real_height = boundingRect.height();
        real_y = alignment.y() * (size.height() - real_height);
    } else {
        real_height = size.height();
        real_y = 0;
    }

    // Move to absolute position.
    real_x += owner->pos().x();
    real_y += owner->pos().y();

    return QRect(real_x, real_y, real_width, real_height);
}
