/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

//*****************************************************************************
//
// Class:	 UserNotificationServiceLinux
// Description:	 The main class of the service.
//
//*****************************************************************************
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "UserNotificationServiceLinux.h"
#include "UNSCommonDefs.h"
#include "registryWrapper.h"
#include "Options.h"
#include "UNSConfig.h"

#define NO_ERROR 0
#define ERROR_INVALID_PARAMETER 1

//*****************************************************************************
// Name			: Constructor
// Description	: Constructor - initializes a new instance of the
//		  		  UserNotificationService class.
//*****************************************************************************
UserNotificationService::UserNotificationService()
{
	m_init = false;
}

//*****************************************************************************
// Name			: Destructor
// Description	: Destructor - finish soap.
//*****************************************************************************
UserNotificationService::~UserNotificationService(void)
{
	DeinitLogger();
}

//*****************************************************************************
// Name		: InitLogger
// Description	: initializes logger
//*****************************************************************************
void
UserNotificationService::InitLogger(void)
{
	m_Logger = new UNSEventLogger();
	m_UserNotificationThread.SetLogger(m_Logger);
	m_PostureRetreivalThread.SetLogger(m_Logger);
	m_AMTStatusThread.SetLogger(m_Logger);
}

//*****************************************************************************
// Name		: DeinitLogger
// Description	: initializes logger
//*****************************************************************************
void
UserNotificationService::DeinitLogger(void)
{
	m_AMTStatusThread.SetLogger(NULL);
	m_PostureRetreivalThread.SetLogger(NULL);
	m_UserNotificationThread.SetLogger(NULL);
	if (m_Logger) {
		delete m_Logger;
	}
}


//*****************************************************************************
// Name		: SetCredentialsMode
// Description	: This is a service set credentials function that gets called
//		  when the service gets created and have command line arguments
//*****************************************************************************
unsigned long
UserNotificationService::SetCredentialsMode(unsigned long argc,
					       const char **argv,
					       unsigned long *specificError)
{
	InitLogger();

	if (m_Logger) {
		m_Logger->DebugLog("Intel(R) AMT User Notification set credentials mode\n");
	}
#ifndef NO_STDOUT_DEBUG_LOG
	else
#endif
	{
		printf ("Intel(R) AMT User Notification set credentials mode\n");
	}
	unsigned long rc = UpdateRegistry(argc, const_cast<const char**>(argv));
	if (m_Logger) {
		m_Logger->DebugLog("Intel(R) AMT User Notification set credentials done\n");
	}
#ifndef NO_STDOUT_DEBUG_LOG
	else
#endif
	{
		printf ("Done.\n");
	}
	*specificError = rc;
	return UNS_STATUS_UPDATE_REG_MODE;
}

//*****************************************************************************
// Name		: ServiceInitialization
// Description	: This is a service initialization function that gets called
//		  when the service gets created
//*****************************************************************************
unsigned long
UserNotificationService::ServiceInitialization(unsigned long argc,
					       const char **argv,
					       unsigned long *specificError)
{
	UNSConfig Config;

	InitLogger();

	/* Initialize AMTStatus Configuration object */
	Config.Init();

	// Initialize the threads.
	m_PostureRetreivalThread.init(Config);
	m_UserNotificationThread.init();
	m_AMTStatusThread.init(Config);

	m_init = true;

	return UNS_STATUS_SUCCESS;
}

//*****************************************************************************
// Name		:	Stop
// Description	:	Initiates the stop service action.
//*****************************************************************************
void UserNotificationService::Stop()
{
	m_StopSemaphore.release();
}

//*****************************************************************************
// Name		:	InternalStop
// Description	:	Stop the service.
//*****************************************************************************
void UserNotificationService::InternalStop()
{
  m_PostureRetreivalThread.stop();
  m_UserNotificationThread.stop();
  m_AMTStatusThread.stop();
  if (m_Logger)
    m_Logger->ServiceStopped();	// Record stop in event log
}

//*****************************************************************************
// Name		:	Suspend
// Description	:	Put the service to suspend mode
//*****************************************************************************
void UserNotificationService::Suspend()
{
  m_PostureRetreivalThread.suspend();
  m_UserNotificationThread.suspend();
  m_AMTStatusThread.suspend();

  if (m_Logger)
    m_Logger->DebugLog("Intel(R) AMT User Notification service suspended.\n");
}

//*****************************************************************************
// Name		:	Resume
// Description	:	Resume the service.
//*****************************************************************************
void UserNotificationService::Resume()
{
  m_PostureRetreivalThread.resume();
  m_UserNotificationThread.resume();
  m_AMTStatusThread.resume();

  if (m_Logger)
    m_Logger->DebugLog("Intel(R) AMT User Notification service resumed.\n");
}

//*****************************************************************************
// Name		: start
// Description	: Listen of the master socket, if a data is received from the
//		  TCP network process the data: Invoke soap_serve(&soap):
//		  The soap_serve() function will invoke the __emc__SoapAlert.
//
//*****************************************************************************
unsigned long UserNotificationService::Start()
{
	if (m_Logger)
		m_Logger->ServiceStart();
	m_PostureRetreivalThread.start();
	m_UserNotificationThread.start();
	m_AMTStatusThread.start();
	
	m_StopSemaphore.acquire();
	//Check for releasing the semaphore
	while (!m_StopSemaphore.acquireTry()) {
		usleep(200);
	}

	InternalStop();

	return UNS_STATUS_SUCCESS;
}

//*****************************************************************************
// Name		:	UpdateRegistry
// Description	:	Run the programm in update registry mode.
//*****************************************************************************
unsigned long UserNotificationService::UpdateRegistry(int argc, const char **argv)
{
  Credentials *unsCreds = NULL;
  Credentials *eacCreds = NULL;
  unsigned long res;

  res = ParseArguments(argc, argv, &unsCreds, &eacCreds);
  if (res != NO_ERROR) {
    // since this is an error that aborts the service initialization.
    if (m_Logger)
      m_Logger->ErrorLog("Attempted to start the service with bad arguments.");
    return res;
  }

  if (eacCreds) {
    printf ("\nUpdating Endpoint Access Control credentials...\n");
    // store in registry
    if (!StoreCredentials(eacCreds, CRED_REG_INFO::EAC)) {
      if (m_Logger)
	m_Logger->WarningLog("Failed to store EAC credentials");
    }
    else
      {
	if (m_Logger)
	  m_Logger->DebugLog("Endpoint Access Control credentials were updated\n");
      }
    delete eacCreds;
  }

  if (unsCreds) {
    printf ("\nUpdating User Notification credentials...\n");

    // store in registry
    if (!StoreCredentials(unsCreds, CRED_REG_INFO::UNS)) {
      if (m_Logger)
	m_Logger->WarningLog("Failed to store UNS credentials\n");
    }
    else
      {
	if (m_Logger)
	  m_Logger->DebugLog("User Notification credentials were updated\n");
      }
    delete unsCreds;
  }

  return 0;
}

//*****************************************************************************
// Name			:	ParseArguments
// Description	:	Parse the command line arguments.
//*****************************************************************************
unsigned long UserNotificationService::ParseArguments(	int argc, const char **argv,
							Credentials **unsCreds,
							Credentials **eacCreds)
{
  Options::FormatNode RawFormat[] = {
    {"unsUser", false, true}, {"unsPass", false, true},
    {"eacUser", false, true}, {"eacPass", false, true},
    {"cainfo", true, false}, {"capath", true, false},
    {"cert", true, false}, {"key",true, false}};

  Options::Format format(RawFormat, sizeof(RawFormat)/sizeof(Options::FormatNode));

#ifdef _DEBUG
  format.addOption("clientip", true, true);
  format.addOption("serverip", true, true);
#endif
  Options options;
  if (!options.parse(argc, argv, format))
    {
      printf("%s\n",options.errorStr());
      return ERROR_INVALID_PARAMETER;
    }
  
  *unsCreds = *eacCreds = NULL;
  if (options.optionExists("cert") || options.optionExists("cainfo") ||
      options.optionExists("key"))
    {
      *unsCreds = new Credentials;
      *eacCreds = new Credentials;
      if (options.optionExists("cert")) {
	(*unsCreds)->certificate = options.getOption("cert");
	(*eacCreds)->certificate = options.getOption("cert");
      }
      if (options.optionExists("cainfo")) {
	(*unsCreds)->cainfo = options.getOption("cainfo");
	(*eacCreds)->cainfo = options.getOption("cainfo");
      }
      if (options.optionExists("key")) {
	(*unsCreds)->sslkey = options.getOption("key");
	(*eacCreds)->sslkey = options.getOption("key");
      }
    }

  if (options.optionExists("unsUser"))
    {
      if (*unsCreds == NULL)
	*unsCreds = new Credentials;
      (*unsCreds)->username = options.getOption("unsUser");
      if (options.optionExists("unsPass"))
	(*unsCreds)->password = options.getOption("unsPass");

    }

  if (options.optionExists("eacUser"))
    {
      if (*eacCreds == NULL)
	*eacCreds = new Credentials;
      (*eacCreds)->username = options.getOption("eacUser");
      if (options.optionExists("eacPass"))
	(*eacCreds)->password = options.getOption("eacPass");
    }
#ifdef _DEBUG
  if (options.optionExists("clientip"))
    {
      setClientIP(options.getOption("clientip"));
    }

  if (options.optionExists("serverip"))
    {
      setServerIP(options.getOption("serverip"));
    }
#endif
  return NO_ERROR;
}

#ifdef _DEBUG
//*****************************************************************************
// Name			: SetClientIP
// Description	: DEBUG ONLY - set remote AMT IP.
//*****************************************************************************
void UserNotificationService::setClientIP(const string &clip)
{
  m_UserNotificationThread.SetClientIP(clip);
  m_PostureRetreivalThread.SetIP(clip);
}

//*****************************************************************************
// Name			: SetserverIP
// Description	: DEBUG ONLY - set remote UNS server IP.
//*****************************************************************************
void UserNotificationService::setServerIP(const string &svip)
{
  m_UserNotificationThread.SetServerIP(svip);
}
#endif
