/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License version 3, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <QtTest/QtTest>

#include "application_manager.h"
#include "processcontroller.h"
#include "proc_info.h"
#include "taskcontroller.h"
#include "upstart/applicationcontroller.h"

#include "qmirserverapplication.h"
#include "qmirserver.h"

#include <signal.h>

using namespace unitymir;

static QString pidCommandLine(qint64 pid)
{
    QFile cmdline(QString("/proc/%1/cmdline").arg(pid));
    cmdline.open(QIODevice::ReadOnly | QIODevice::Text);
    QByteArray command = cmdline.readLine().replace('\0', ' ');
    cmdline.close();
    return command.simplified();
}

class ApplicationManagerTests: public QObject
{
    Q_OBJECT
private Q_SLOTS:
    void testStartStop();
};

void ApplicationManagerTests::testStartStop()
{
    ApplicationManager::Factory appFactory;
    QSharedPointer<ApplicationManager> manager{appFactory.create()};

    Application *app = manager->startApplication("unity-mir-test-helper-app", QStringList());
    QVERIFY(app);
    QCOMPARE(app->desktopFile(), QString("/usr/share/applications/unity-mir-test-helper-app.desktop"));
    QCOMPARE(app->name(), QString("My Fake App"));
    QCOMPARE(app->comment(), QString("My Fake App Comment"));
    QCOMPARE(app->icon(), QUrl("image://theme/fakeIcon"));
    QCOMPARE(app->exec(), QString("unity-mir-test-helper-app"));

    QTRY_VERIFY(!pidCommandLine(app->pid()).isEmpty());
    QString simplifiedCommand = pidCommandLine(app->pid());
    QCOMPARE(simplifiedCommand, QString("unity-mir-test-helper-app"));

    manager->stopApplication(app->appId());

    QDir d;
    QTRY_VERIFY(!d.exists(QString("/proc/%1").arg(app->pid())));
}

static int startShell(int argc, const char* argv[], ShellServerConfiguration* server)
{
    QMirServerApplication application(argc, (char**)argv, server);
    ApplicationManagerTests tc;
    int res = QTest::qExec(&tc, argc, (char**)argv);
    raise(SIGTERM);
    return res;
}

int main(int argc, const char *argv[])
{
    QMirServer mirServer(argc, argv);
    return mirServer.runWithClient(startShell);
}

#include "main.moc"
