// @HEADER
//
// ***********************************************************************
//
//        MueLu: A package for multigrid based preconditioning
//                  Copyright 2012 Sandia Corporation
//
// Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
// the U.S. Government retains certain rights in this software.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact
//                    Jonathan Hu       (jhu@sandia.gov)
//                    Andrey Prokopenko (aprokop@sandia.gov)
//                    Ray Tuminaro      (rstumin@sandia.gov)
//                    Tobias Wiesner    (tawiesn@sandai.gov)
//
// ***********************************************************************
//
// @HEADER
#ifndef PACKAGES_MUELU_SRC_REBALANCING_MUELU_REPARTITIONHEURISTICFACTORY_DECL_HPP_
#define PACKAGES_MUELU_SRC_REBALANCING_MUELU_REPARTITIONHEURISTICFACTORY_DECL_HPP_

#include "MueLu_ConfigDefs.hpp"
#ifdef HAVE_MPI

// Some classes are only used in the definition (_def.hpp) of this class
// but forward declarations are needed here to enable the UseShortNames mechanism.
#include <Xpetra_Map_fwd.hpp>
#include <Xpetra_MapFactory_fwd.hpp>
#include <Xpetra_Import_fwd.hpp>
#include <Xpetra_ImportFactory_fwd.hpp>
#include <Xpetra_Export_fwd.hpp>
#include <Xpetra_ExportFactory_fwd.hpp>
#include <Xpetra_Vector_fwd.hpp>
#include <Xpetra_Matrix_fwd.hpp>
#include <Xpetra_MatrixFactory_fwd.hpp>

#include "MueLu_SingleLevelFactoryBase.hpp"
#ifdef HAVE_MUELU_EXPERIMENTAL
#include "MueLu_BlockedRAPFactory_fwd.hpp"
#endif
#include "MueLu_RAPFactory_fwd.hpp"
#include "MueLu_SubBlockAFactory_fwd.hpp"
#include "MueLu_RepartitionFactory_fwd.hpp"
#include "MueLu_Utilities_fwd.hpp"

namespace MueLu {

  /*!
    @class RepartitionHeuristicFactory class.
    @brief Factory for determing the number of partitions for rebalancing

    Given the unrebalanced (coarse) matrix A and some user parameters we determine the number
    of partitions (processes) the (rebalanced) matrix should use. The information is used as input
    for the Interface classes which perform the actual Zoltan call to determine the partition placement
    on the processors.

    @note A must be a "non-rebalanced" matrix generated by RAPFactory, SubBlockAFactory or BlockedRAPFactory

    @note For blocked operators you have only one RepartitionHeuristicInterface instance. But you need an
    Interface class for each sub block on the diagonal of the input matrix.

    ## Input/output of RepartitionHeuristicInterface ##

    ### User parameters of RepartitionInterface ###
    Parameter | type | default | master.xml | validated | requested | description
    ----------|------|---------|:----------:|:---------:|:---------:|------------
    | A                                      | Factory | null  |   | * | * | Generating factory of the matrix A used during the prolongator smoothing process |
    | repartition: start level | int | 2 | * | * | | Minimum level to run partitioner. MueLu does not rebalance levels finer than this one.
    | repartition: min rows per proc | LO      | 800 | * | * |  | Minimum number of rows per processor. If the actual number if smaller, then rebalancing occurs.
    | repartition: max imbalance | double      | 1.2 | * | * |  | Maximum nonzero imbalance ratio. If the actual number is larger, the rebalancing occurs.

    The * in the @c master.xml column denotes that the parameter is defined in the @c master.xml file.<br>
    The * in the @c validated column means that the parameter is declared in the list of valid input parameters (see RepartitionHeuristicFactory::GetValidParameters).<br>
    The * in the @c requested column states that the data is requested as input with all dependencies (see RepartitionHeuristicFactory::DeclareInput).

    ### Variables provided by RepartitionHeuristicFactory ###

    After RepartitionHeuristicFactory::Build the following data is available (if requested)

    Parameter | generated by | description
    ----------|--------------|------------
    | number of partitions | RepartitionHeuristicFactory   | Integer containing the number of partitions to be built. See below for details

    The "number of partitions" variable contains the requested number of partitions for rebalancing. It is needed as input by the Interface classes
    (see, e.g., ZoltanInterface, Zoltan2Interface, IsorropiaInterface) and the RepartitionFactory. The Interface classes use it as input for the Zoltan
    call which considers the information when finding the rebalanced partition.

    If "number of partitions" is 1 (i.e., move all data to one processor), we can skip the call to the Interface classes (see RepartitionFactory).
    If "number of partitions" is -1 we don't perform rebalancing (just keep the partition unchanged).

  */
  template <class Scalar = double, class LocalOrdinal = int, class GlobalOrdinal = LocalOrdinal, class Node = KokkosClassic::DefaultNode::DefaultNodeType>
  class RepartitionHeuristicFactory : public SingleLevelFactoryBase {
#undef MUELU_REPARTITIONHEURISTICFACTORY_SHORT
#include "MueLu_UseShortNames.hpp"

  public:
    //! @name Constructors/Destructors.
    //@{

    //! Constructor.
    RepartitionHeuristicFactory() { }

    //! Destructor.
    virtual ~RepartitionHeuristicFactory() { }

    RCP<const ParameterList> GetValidParameterList() const;

    //@}

    //! @name Input
    //@{

    /*! @brief Determines the data that RepartitionHeuristicFactory needs, and the factories that generate that data.

        If this class requires some data, but the generating factory is not specified in DeclareInput, then this class
        will fall back to the settings in FactoryManager.
    */
    void DeclareInput(Level& currentLevel) const;

    //@}

    //! @name Build methods.
    //@{

    //! Build an object with this factory.
    void Build(Level& currentLevel) const;

    //@}
  }; // class RepartitionHeuristicFactory

} // namespace MueLu

#define MUELU_REPARTITIONHEURISTICFACTORY_SHORT

#endif //ifdef HAVE_MPI
#endif /* PACKAGES_MUELU_SRC_REBALANCING_MUELU_REPARTITIONHEURISTICFACTORY_DECL_HPP_ */
