/*
 * Test program for the BASIL INVENTORY-type "QUERY" method
 *
 * Perform a Basil INVENTORY and then reassemble the response to stdout.
 */
#include "basil_alps.h"

/* GLOBALS */
static enum basil_version version;	/* version-dependent format */
static const char	  *space;	/* version-dependent indenting */

/* DECLARATIONS */
static void print_proc(const struct basil_node_processor *proc)
{
	if (proc == NULL)
		return;

	print_proc(proc->next);
	printf("      %s<Processor ordinal=\"%u\" "
	       "architecture=\"%s\" clock_mhz=\"\%u\"", space,
	       proc->ordinal, nam_proc[proc->arch], proc->clock_mhz);

	if (proc->allocation == NULL) {
		printf("/>\n");
	} else {
		printf(">\n"
		       "       %s<ProcessorAllocation reservation_id=\"%u\"/>\n"
		       "      %s</Processor>\n", space,
		       proc->allocation->rsvn_id, space);
	}
}

static void print_mema(const struct basil_mem_alloc *mema)
{
	if (mema == NULL)
		return;
	print_mema(mema->next);
	printf("       %s<MemoryAllocation reservation_id=\"%u\" "
	       "page_count=\"%u\"/>\n", space, mema->rsvn_id, mema->page_count);
}

static void print_mem(const struct basil_node_memory *mem)
{
	if (mem == NULL)
		return;
	print_mem(mem->next);

	printf("      %s<Memory type=\"%s\" page_size_kb=\"%u\" "
			"page_count=\"%u\"", space, nam_memtype[mem->type],
			mem->page_size_kb, mem->page_count);

	if (mem->a_head == NULL) {
		printf("/>\n");
	} else {
		printf(">\n");
		print_mema(mem->a_head);
		printf("      %s</Memory>\n", space);
	}
}

static void print_label(const struct basil_label *label)
{
	if (label == NULL)
		return;
	print_label(label->next);
	printf("      %s<Label name=\"%s\" "
	       "type=\"%s\" disposition=\"%s\"/>\n", space, label->name,
	       nam_labeltype[label->type], nam_ldisp[label->disp]);
}

static void print_segment(struct basil_segment *seg)
{
	if (seg == NULL)
		return;

	print_segment(seg->next);
	printf("      <Segment ordinal=\"%u\">\n", seg->ordinal);
	if (seg->proc_head == NULL) {
		printf("       <ProcessorArray/>\n");
	} else {
		printf("       <ProcessorArray>\n");
		print_proc(seg->proc_head);
		printf("       </ProcessorArray>\n");
	}

	if (seg->mem_head == NULL) {
		printf("       <MemoryArray/>\n");
	} else {
		printf("       <MemoryArray>\n");
		print_mem(seg->mem_head);
		printf("       </MemoryArray>\n");
	}

	if (seg->lbl_head == NULL) {
		printf("       <LabelArray/>\n");
	} else {
		printf("       <LabelArray>\n");
		print_label(seg->lbl_head);
		printf("       </LabelArray>\n");
	}
	printf("      </Segment>\n");
}

static void print_accel(const struct basil_node_accelerator *accel)
{
	if (accel == NULL)
		return;
	print_accel(accel->next);
	printf("      <Accelerator ordinal=\"%u\" type=\"%s\" state=\"%s\" "
			           "family=\"%s\" memory_mb=\"%u\" "
				   "clock_mhz=\"%u\"", accel->ordinal,
		nam_acceltype[accel->type], nam_accelstate[accel->state],
		accel->family, accel->memory_mb, accel->clock_mhz);
	if (accel->allocation) {
		printf(">\n"
		       "       <AcceleratorAllocation reservation_id=\"%u\"/>\n"
		       "      </Accelerator>\n", accel->allocation->rsvn_id);
	} else {
		printf("/>\n");
	}
}

static void print_node(const struct basil_node *node)
{
	if (node == NULL)
		return;

	print_node(node->next);
	printf("    <Node node_id=\"%u\" name=\"%s\" architecture=\"%s\" "
						"role=\"%s\" state=\"%s\"",
		node->node_id, node->name, nam_arch[node->arch],
		nam_noderole[node->role], nam_nodestate[node->state]);
	if (version >= BV_3_1)
		printf(" router_id=\"%u\"", node->router_id);
	printf(">\n");

	if (version > BV_1_0) {
		printf("     <SegmentArray>\n");
		print_segment(node->seg_head);
		printf("     </SegmentArray>\n");
		if (node->accel_head) {
			printf("     <AcceleratorArray>\n");
			print_accel(node->accel_head);
			printf("     </AcceleratorArray>\n");
		}
		printf("    </Node>\n");
		return;
	}

	if (node->seg_head->proc_head == NULL) {
		printf("     <ProcessorArray/>\n");
	} else {
		printf("     <ProcessorArray>\n");
		print_proc(node->seg_head->proc_head);
		printf("     </ProcessorArray>\n");
	}

	if (node->seg_head->mem_head == NULL) {
		printf("     <MemoryArray/>\n");
	} else {
		printf("     <MemoryArray>\n");
		print_mem(node->seg_head->mem_head);
		printf("     </MemoryArray>\n");
	}

	if (node->seg_head->lbl_head == NULL) {
		printf("     <LabelArray/>\n");
	} else {
		printf("     <LabelArray>\n");
		print_label(node->seg_head->lbl_head);
		printf("     </LabelArray>\n");
	}

	printf("    </Node>\n");
}

static void print_node_array(const struct basil_inventory *inv)
{
	printf("   <NodeArray");
	if (version >= BV_4_0)
		printf(" changecount=\"%llu\"",
		(unsigned long long)inv->change_count);
	if (inv->f->node_head == NULL) {
		printf("/>\n");
	} else {
		printf(">\n");
		print_node(inv->f->node_head);
		printf("   </NodeArray>\n");
	}
}

static void print_command(const struct basil_rsvn_app_cmd *cmd)
{
	if (cmd == NULL)
		return;
	print_command(cmd->next);

	printf("        <Command width=\"%u\" depth=\"%u\" nppn=\"%u\" "
			 "memory=\"%u\" architecture=\"%s\" cmd=\"%s\"/>\n",
		cmd->width, cmd->depth, cmd->nppn, cmd->memory,
		nam_arch[cmd->arch], cmd->cmd);
}

static void print_rsvn_app(const struct basil_rsvn_app *app)
{
	if (app == NULL)
		return;
	print_rsvn_app(app->next);

	printf("      <Application application_id=\"%llu\" user_id=\"%u\" "
			"group_id=\"%u\" time_stamp=\"%u\">\n",
		(unsigned long long)app->apid,
		app->user_id, app->group_id, (unsigned)app->timestamp);

	printf("       <CommandArray>\n");
	print_command(app->cmd_head);
	printf("       </CommandArray>\n");

	printf("      </Application>\n");
}

static void print_rsvn(const struct basil_rsvn *rsvn)
{
	if (rsvn == NULL)
		return;
	print_rsvn(rsvn->next);

	printf("    <Reservation reservation_id=\"%u\" "
				"user_name=\"%s\" account_name=\"%s\"",
	       rsvn->rsvn_id, rsvn->user_name, rsvn->account_name);

	if (version == BV_1_0) {
		printf("/>\n");
	} else {
		/*
		 * Basil 1.1 and higher
		 */
		printf(" time_stamp=\"%u\" batch_id=\"%s\"",
			(unsigned)rsvn->timestamp, rsvn->batch_id);
		if (version >= BV_3_1)
			printf(" reservation_mode=\"%s\" gpc_mode=\"%s\"",
				nam_rsvn_mode[rsvn->rsvn_mode],
				nam_gpc_mode[rsvn->gpc_mode]);
		printf(">\n");

		printf("     <ApplicationArray>\n");
		print_rsvn_app(rsvn->app_head);
		printf("     </ApplicationArray>\n");

		printf("    </Reservation>\n");
	}
}

static void print_resv_array(const struct basil_full_inventory *f)
{
	if (f->rsvn_head == NULL) {
		printf("   <ReservationArray/>\n");
	} else {
		printf("   <ReservationArray>\n");
		print_rsvn(f->rsvn_head);
		printf("   </ReservationArray>\n");
	}
}

int main(int ac, char **av)
{
	struct basil_inventory *inv;
	const char *indent[] = { "" , "  " };

	version  = get_basil_version();
	space    = indent[version > BV_1_0];
	inv	 = get_full_inventory(version);
	if (inv == NULL)
		err(1, "allocation of inventory data failed");

	printf(	"<?xml version=\"1.0\"?>\n"
		"<BasilResponse protocol=\"%s\">\n"
		" <ResponseData status=\"SUCCESS\" method=\"QUERY\">\n",
	       	bv_names[version]);
	if (version >= BV_3_1)
	       	printf("  <Inventory mpp_host=\"%s\" timestamp=\"%u\">\n",
			inv->mpp_host, (unsigned)inv->timestamp);
	else
	       	printf("  <Inventory>\n");
	print_node_array(inv);
	print_resv_array(inv->f);
	printf( "  </Inventory>\n"
		" </ResponseData>\n"
		"</BasilResponse>\n");

	free_inv(inv);

	return EXIT_SUCCESS;
}
