## Usage

See the [unit tests](test/test.js) for many more examples.

```js
const get = require('get-value');
const obj = { a: { b: { c: { d: 'foo' } } } };

console.log(get(obj));            //=> { a: { b: { c: { d: 'foo' } } } };
console.log(get(obj, 'a'));       //=> { b: { c: { d: 'foo' } } }
console.log(get(obj, 'a.b'));     //=> { c: { d: 'foo' } }
console.log(get(obj, 'a.b.c'));   //=> { d: 'foo' }
console.log(get(obj, 'a.b.c.d')); //=> 'foo'
```


### Supports keys with dots

Unlike other dot-prop libraries, get-value works when keys have dots in them:

```js
console.log(get({ 'a.b': { c: 'd' } }, 'a.b.c'));
//=> 'd'

console.log(get({ 'a.b': { c: { 'd.e': 'f' } } }, 'a.b.c.d.e'));
//=> 'f'
```

### Supports arrays


```js
console.log(get({ a: { b: { c: { d: 'foo' } } }, e: [{ f: 'g' }, { f: 'h' }] }, 'e.1.f'));   
//=> 'h'

console.log(get({ a: { b: [{ c: 'd' }] } }, 'a.b.0.c')); 
//=> 'f'

console.log(get({ a: { b: [{ c: 'd' }, { e: 'f' }] } }, 'a.b.1.e'));
//=> 'f'
```

### Supports functions


```js
function foo() {}
foo.bar = { baz: 'qux' };

console.log(get(foo));            
//=> { [Function: foo] bar: { baz: 'qux' } }

console.log(get(foo, 'bar'));     
//=> { baz: 'qux' }

console.log(get(foo, 'bar.baz')); 
//=> qux
```

### Supports passing object path as an array

Slighly improve performance by passing an array of strings to use as object path segments (this is also useful when you need to dynamically build up the path segments):

```js
console.log(get({ a: { b: 'c' } }, ['a', 'b']));
//=> 'c'
```

## Options

### options.default

**Type**: `any`

**Default**: `undefined`

The default value to return when get-value cannot resolve a value from the given object.

```js
const obj = { foo: { a: { b: { c: { d: 'e' } } } } };
console.log(get(obj, 'foo.a.b.c.d', { default: true }));  //=> 'e'
console.log(get(obj, 'foo.bar.baz', { default: true }));  //=> true
console.log(get(obj, 'foo.bar.baz', { default: false })); //=> false
console.log(get(obj, 'foo.bar.baz', { default: null }));  //=> null

// you can also pass the default value as the last argument
// (this is necessary if the default value is an object)
console.log(get(obj, 'foo.a.b.c.d', true));  //=> 'e'
console.log(get(obj, 'foo.bar.baz', true));  //=> true
console.log(get(obj, 'foo.bar.baz', false)); //=> false
console.log(get(obj, 'foo.bar.baz', null));  //=> null
```


### options.isValid

**Type**: `function`

**Default**: `true`

If defined, this function is called on each resolved value. Useful if you want to do `.hasOwnProperty` or `Object.prototype.propertyIsEnumerable`.


```js
const isEnumerable = Object.prototype.propertyIsEnumerable;
const options = {
  isValid: (key, obj) => isEnumerable.call(obj, key)
};

const obj = {};
Object.defineProperty(obj, 'foo', { value: 'bar', enumerable: false });

console.log(get(obj, 'foo', options));           //=> undefined
console.log(get({}, 'hasOwnProperty', options)); //=> undefined
console.log(get({}, 'constructor', options));    //=> undefined

// without "isValid" check
console.log(get(obj, 'foo', options));           //=> bar
console.log(get({}, 'hasOwnProperty', options)); //=> [Function: hasOwnProperty]
console.log(get({}, 'constructor', options));    //=> [Function: Object]
```


### options.split

**Type**: `function`

**Default**: `String.split()`

Custom function to use for splitting the string into object path segments. 

```js
const obj = { 'a.b': { c: { d: 'e' } } };

// example of using a string to split the object path
const options = { split: path => path.split('/') };
console.log(get(obj, 'a.b/c/d', options)); //=> 'e'

// example of using a regex to split the object path
// (removing escaped dots is unnecessary, this is just an example)
const options = { split: path => path.split(/\\?\./) };
console.log(get(obj, 'a\\.b.c.d', options)); //=> 'e'
```


### options.separator

**Type**: `string|regex`

**Default**: `.`

The separator to use for spliting the string (this is probably not needed when `options.split` is used).

```js
const obj = { 'a.b': { c: { d: 'e' } } };

console.log(get(obj, 'a.b/c/d', { separator: '/' }));       
//=> 'e'

console.log(get(obj, 'a\\.b.c.d', { separator: /\\?\./ })); 
//=> 'e'
```


### options.join

**Type**: `function`

**Default**: `Array.join()`

Customize how the object path is created when iterating over path segments. 

```js
const obj = { 'a/b': { c: { d: 'e' } } };
const options = {
  // when segs === ['a', 'b'] use a "/" to join, otherwise use a "."
  join: segs => segs.join(segs[0] === 'a' ? '/' : '.')
};

console.log(get(obj, 'a.b.c.d', options));
//=> 'e'
```


### options.joinChar

**Type**: `string`

**Default**: `.`

The character to use when re-joining the string to check for keys with dots in them (this is probably not needed when `options.join` is used). This can be a different value than the separator, since the separator can be a string or regex.

```js
const target = { 'a-b': { c: { d: 'e' } } };
const options = { joinChar: '-' };
console.log(get(target, 'a.b.c.d', options)); 
//=> 'e'
```


## Benchmarks

_(benchmarks were run on a MacBook Pro 2.5 GHz Intel Core i7, 16 GB 1600 MHz DDR3)_.

get-value is more reliable and has more features than dot-prop, without sacrificing performance.

```
{%= include("./benchmark/stats.md") %}
```


### Running the benchmarks

Clone this library into a local directory:

```sh
$ git clone https://github.com/jonschlinkert/get-value.git
```

Then install devDependencies and run benchmarks:

```sh
$ npm install && node benchmark
```


## Release history

### v3.0.0

- Improved support for escaping. It's no longer necessary to use backslashes to escape keys.
- Adds `options.default` for defining a default value to return when no value is resolved.
- Adds `options.isValid` to allow the user to check the object after each iteration.
- Adds `options.separator` for customizing character to split on.
- Adds `options.split` for customizing how the object path is split.
- Adds `options.join` for customizing how the object path is joined when iterating over path segments.
- Adds `options.joinChar` for customizing the join character.

