/*
   libpbuild initialization code

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib.h>

#include "pbuild-priv.h"

#define ETH_P_IPX 0x8137 /* IPX over DIX */
#define ETH_P_VTP 0x2003 /* Cisco VTP */

static const struct pdu_dict ethertype_mappings[] = {
    {"ip", (uint16_t []) {ETH_TYPE_IP} },
    {"ip6", (uint16_t []) {ETH_TYPE_IPV6} },
    {"dot1q", (uint16_t []) {ETH_TYPE_8021Q} },
    {"mpls", (uint16_t []) {ETH_TYPE_MPLS} },
    {"ipx", (uint16_t []) {ETH_P_IPX} },
    {"arp", (uint16_t []) {ETH_TYPE_ARP} },
    {"arp-request", (uint16_t []) {ETH_TYPE_ARP} },
    {"arp-reply", (uint16_t []) {ETH_TYPE_ARP} },
    {"rarp", (uint16_t []) {ETH_TYPE_REVARP} },
    {"rarp-request", (uint16_t []) {ETH_TYPE_REVARP} },
    {"rarp-reply", (uint16_t []) {ETH_TYPE_REVARP} },
    {"vtp-summary", (uint16_t []) {ETH_P_VTP} },
    {"vtp-subset", (uint16_t []) {ETH_P_VTP} },
    {NULL, NULL}
};

GHashTable *ethertypes;

/* This holds all the registered PDUs */
static GHashTable *pdus;

/* This holds all the registered PDU options */
static GHashTable *pdu_options;

/*
 * Retrieves a PDU template by name.
 */
pdu_t *
_pb_getpdutemplate(const char *pdu_name)
{
    return g_hash_table_lookup(pdus, pdu_name);
}

/*
 * Retrieves a PDU option template by name.
 */
pdu_option_t *
_pb_getopttemplate(const char *opts_class, const char *opt_name)
{
    GHashTable *opt_class_hash;

    opt_class_hash = g_hash_table_lookup(pdu_options, opts_class);
    if (!opt_class_hash)
	return NULL;

    return g_hash_table_lookup(opt_class_hash, opt_name);
}

void
_pb_register_protocol_option(const char *option_class,
			     const pdu_option_t *pdu_option)
{
    GHashTable *opt_class_hash;

    if (pdu_option->name == NULL) {
	fprintf(stderr, "Error in PDU option definition: \"name\" field "
		"can't be NULL.\n");
	exit(EXIT_FAILURE);
    }

    if (pdu_option->len == 0 && pdu_option->get_len == NULL) {
	fprintf(stderr, "Error in \"%s\" PDU option definition: length is "
		"zero and there is no \"get_len\" method.\n", pdu_option->name);
	exit(EXIT_FAILURE);
    }

    if (!(opt_class_hash = g_hash_table_lookup(pdu_options, option_class) ) ) {
	opt_class_hash = g_hash_table_new_full(g_str_hash, g_str_equal, NULL, NULL);
	g_hash_table_insert(pdu_options, (gpointer) option_class, opt_class_hash);
    }

    if (g_hash_table_lookup(opt_class_hash, pdu_option->name) ) {
	fprintf(stderr, "Error in PDU option definition: PDU option "
		"\"%s\" already registered.\n", pdu_option->name);
	exit(EXIT_FAILURE);
    }

    g_hash_table_insert(opt_class_hash, pdu_option->name, (gpointer) pdu_option);
}

void
_pb_register_protocol(const pdu_t *pdu)
{
    if (pdu->name == NULL) {
	fprintf(stderr, "Error in PDU definition: \"name\" field can't be NULL\n");
	exit(EXIT_FAILURE);
    }

    if (pdu->len == 0 && pdu->get_len == NULL) {
	fprintf(stderr, "Error in \"%s\" PDU definition: length is zero "
		"and there is no \"get_len\" method\n", pdu->name);
	exit(EXIT_FAILURE);
    }

    if (g_hash_table_lookup(pdus, pdu->name) ) {
	fprintf(stderr, "Error in PDU definition: PDU \"%s\" already "
		"registered.\n", pdu->name);
	exit(EXIT_FAILURE);
    }

    g_hash_table_insert(pdus, pdu->name, (gpointer) pdu);
}

static void
_pb_register_pdus(void)
{
    _pb_register_ether();
    _pb_register_dot1q();
    _pb_register_dtp();
    _pb_register_vtp();
    _pb_register_gre();
    _pb_register_arp();
    _pb_register_mpls();
    _pb_register_ipv4();
    _pb_register_ipv4_generic();
    _pb_register_ipv6();
    _pb_register_ipx();
    _pb_register_icmp();
    _pb_register_icmp6();
    _pb_register_igmp();
    _pb_register_tcp();
    _pb_register_udp();
    _pb_register_ospf();
    _pb_register_bgp();
    _pb_register_dns();
    _pb_register_sntp();
    _pb_register_data();
}

int
pb_init(void)
{
    /*
     * Create hash table for protocol name to protocol value
     * mappings. This is used by the 1st pass PDU builder.
     */
    ethertypes = _pb_new_hash(ethertype_mappings);

    /*
     * In this dictionary, keys are strings. Hence, g_str_hash as the hash
     * function, and g_str_equal as the key_equal_func parameter.
     */
    pdus = g_hash_table_new_full(g_str_hash, g_str_equal, NULL, NULL);

    /*
     * In this dictionary, keys are strings. Hence, g_str_hash as the hash
     * function, and g_str_equal as the key_equal_func parameter.
     */
    pdu_options = g_hash_table_new_full(g_str_hash, g_str_equal, NULL, NULL);

    /* Register all PDUs */
    _pb_register_pdus();

    return 0;
}
