/* nexp_commands.c
 
   The bulk of the Network Expect commands

   Commands implemented here:

   nexp_version (NExp_VersionCmd)
   sleep (NExp_SleepCmd)
   txdelta (NExp_TxDeltaCmd)
   nexp_continue (NExp_ContinueCmd)
   system (NExp_SystemCmd)
   random (NExp_RandomCmd)
   host (NExp_HostCmd)
   help (NExp_HelpCmd)
   clear (NExp_ClearObjCmd)
   show (NExp_ShowObjCmd)
   islocal (NExp_IsLocalObjCmd)

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"
#include "nexp_cmd_parser.h"
#include "nexp_ghost.h"
#include "nexp_speakers.h"
#include "util-tcl.h"

static char nexp_version[] = PACKAGE_VERSION;

/********************************************************************
 *                            nexp_version                          *
 ********************************************************************/
static int
NExp_VersionCmd(ClientData clientData _U_, Tcl_Interp *interp,
		    int argc, const char **argv)
{
    int nmajor, umajor;
    const char *user_version; /* user-supplied version string */

    if (argc == 1) {
	Tcl_AppendResult(interp, "Network Expect version ", nexp_version, "\n",
		pcap_lib_version(), "\n",
		"Wireshark Packet Analyzer version ", epan_get_version(), "\n",
		NULL);
	return TCL_OK;
    }

    if (argc > 3 || (argc == 3 && strcmp(argv[1], "-exit") ) ) {
	nexp_error(interp, "usage: %s [[-exit] version]", argv[0]);
	return TCL_ERROR;
    }

    user_version = argv[argc == 2 ? 1 : 2];
    nmajor = atoi(nexp_version);
    umajor = atoi(user_version);

    /* first check major numbers */
    if (nmajor == umajor) {
	int u, n;
	char *dot;

	/* Now check minor numbers */
	dot = strchr(user_version, '.');
	if (!dot) {
	    nexp_error(interp, "version number must include a minor version number");
	    return TCL_ERROR;
	}
	u = atoi(dot + 1);

	dot = strchr(nexp_version, '.');
	n = atoi(dot + 1);

	if (n >= u)
	    return TCL_OK;
    }

    if (argc == 2) {
	nexp_error(interp, "%s requires Network Expect version %s but using %s",
		   program_name, user_version, nexp_version);
	return TCL_ERROR;
    }
    nexpErrorLog("%s requires Network Expect version %s but using %s\r\n",
		 program_name, user_version, nexp_version);

    Tcl_Eval(interp, "exit 1");

    /* Not reached */
    return TCL_OK;
}

/********************************************************************
 *                              sleep                               *
 ********************************************************************/
static int
NExp_SleepCmd(ClientData clientData _U_, Tcl_Interp *interp, int argc,
	      const char **argv)
{
    struct timeval t;
    double sec;

    if (argc != 2) {
	nexp_error(interp, "must have one arg: seconds");
	return TCL_ERROR;
    }

    sec = (double) atof(argv[1]);

    t.tv_sec = sec;
    t.tv_usec = (sec - t.tv_sec) * 1000000L;

restart:
    if (Tcl_AsyncReady()) {
	int rc = Tcl_AsyncInvoke(interp,TCL_OK);
	if (rc != TCL_OK) return rc;
    }

    if (select(1, NULL, NULL, NULL, &t) == -1 && errno == EINTR)
	goto restart;

    return TCL_OK;
}

/********************************************************************
 *			       txdelta                              *
 ********************************************************************/
static int
NExp_TxDeltaCmd(ClientData clientData _U_, Tcl_Interp *interp, int argc,
		const char **argv)
{
    struct nexp_speaker *s;
    struct timeval ts;
    double secs_previous, secs_current;
    Tcl_Obj *obj;

    if (argc != 2) {
	nexp_error(interp, "must have one arg: speaker ID");
	return TCL_ERROR;
    }

    s = lookup_speaker(argv[1]);
    if (!s) {
	nexp_error(interp, "No speaker named \"%s\". Use "
			   "\"spawn_network -info\" to find out existing "
			   "speakers.", argv[1]);
	return TCL_ERROR;
    }

    if (timerisset(&s->ts) ) {
	gettimeofday(&ts, NULL);
	secs_previous = s->ts.tv_sec + s->ts.tv_usec/1e6;
	secs_current = ts.tv_sec + ts.tv_usec/1e6;
	obj = Tcl_NewDoubleObj(secs_current - secs_previous);
    } else {
	obj = Tcl_NewDoubleObj(0.0);
    }

    Tcl_SetObjResult(interp, obj);

    return TCL_OK;
}

/********************************************************************
 *                          nexp_continue                           *
 ********************************************************************/

static int
NExp_ContinueCmd(ClientData clientData _U_, Tcl_Interp *interp, int argc,
		     const char **argv _U_)
{
    if (argc == 1)
	return NEXP_CONTINUE;

    nexp_error(interp, "usage: %s\n", argv[0]);
    return TCL_ERROR;
}

/********************************************************************
 *                              system                              *
 ********************************************************************/

/*
 * WARNING WARNING WARNING WARNING WARNING WARNING WARNING WARNING 
 * WARNING WARNING WARNING WARNING WARNING WARNING WARNING WARNING 
 * WARNING WARNING WARNING WARNING WARNING WARNING WARNING WARNING 
 *
 * This command does no input validation at all, which means that
 * it is very insecure. For example "system {ls;/bin/sh}" will give
 * you a shell. If nexp is run as root then the shell will be a root
 * shell.
 */
static int
NExp_SystemCmd(ClientData clientData _U_, Tcl_Interp *interp, int argc,
	       const char **argv)
{
    int i, retval;
    char *buf;
    Tcl_Obj *obj;
    size_t len;

    if (argc == 1) {
	nexp_error(interp, "usage: %s <syscmd> [<args>]", argv[0]);
	return TCL_ERROR;
    }

    for (len = 0, i = 1; i < argc; i++)
	len += strlen(argv[i]);

    len += argc;

    buf = xmalloc(len);

    for (strlcpy(buf, argv[1], len), i = 2; i < argc; i++) {
	strlcat(buf, " ", len);
	strlcat(buf, argv[i], len);
    }

    retval = system(buf);

    free(buf);

    obj = Tcl_NewIntObj(retval);

    Tcl_SetObjResult(interp, obj);

    return retval == -1 ? TCL_ERROR : TCL_OK;
}

/********************************************************************
 *                              random                              *
 ********************************************************************/

static int
NExp_RandomCmd(ClientData clientData _U_, Tcl_Interp *interp, int argc,
	       const char **argv)
{
    Tcl_Obj *obj;
    char *ip;
    char *s, *range = NULL;
    struct addr a;
    int mask;
    ip_addr_t x, y;
    eth_addr_t mac;

    if (argc == 1 || (argc == 2 && strstr("number", argv[1]) ) ) {
	/* Random number from the entire spectrum */
	obj = Tcl_NewIntObj(random() );
    } else if (argc == 2 && strstr("ip", argv[1]) ) {
	/* Random IPv4 address from the entire spectrum */
	ip = ipaddr2str(random() );
	obj = Tcl_NewStringObj(ip, strlen(ip) );
    } else if (argc == 3 && strstr("number", argv[1]) ) {
	/* Random number from within a range */
	range = xstrdup(argv[2]);

	s = strchr(range, ':');
	if (!s) {
	    nexp_error(interp, "invalid range. Must be in the form x:y");
	    free(range);
	    return TCL_ERROR;
	}
	*s++ = '\0';

	obj = Tcl_NewIntObj(random_int(atoi(range), atoi(s) ) );
    } else if (argc == 3 && strstr("ip", argv[1]) ) {
	/* Random IPv4 address from within a range */
	range = xstrdup(argv[2]);

	s = strchr(range, ':');
	if (s) {
	    /* Range specified as 1.2.3.4:5.6.7.8 */
	    *s++ = '\0';
	    if (addr_pton(range, &a) == -1 || a.addr_type != ADDR_TYPE_IP
		|| a.addr_bits != IP_ADDR_BITS) {
		nexp_error(interp, "invalid IP address range");
		free(range);
		return TCL_ERROR;
	    }
	    x = a.addr_ip;

	    if (addr_pton(s, &a) == -1 || a.addr_type != ADDR_TYPE_IP
		|| a.addr_bits != IP_ADDR_BITS) {
		nexp_error(interp, "invalid IP address range");
		free(range);
		return TCL_ERROR;
	    }
	    y = a.addr_ip;
	} else {
	    /* Range must have been specified as a network mask */
	    if (addr_pton(range, &a) == -1 || a.addr_type != ADDR_TYPE_IP
		|| a.addr_bits == IP_ADDR_BITS
		|| addr_btom(a.addr_bits, &mask, IP_ADDR_LEN) == -1) {
		nexp_error(interp, "invalid IP address range");
		free(range);
		return TCL_ERROR;
	    }

	    x = a.addr_ip & mask;
	    y = x + htonl( (1 << (IP_ADDR_BITS - a.addr_bits) ) - 1);
	}

	ip = ipaddr2str(random_ip(x, y) );
	obj = Tcl_NewStringObj(ip, strlen(ip) );
    } else if (argc == 2 && strstr("mac", argv[1]) ) {
	/*
	 * Random MAC address. We use IANA Ethernet address block for
	 * unicast use. See http://www.iana.org/assignments/ethernet-numbers.
	 */
	mac.data[0] = 0x00;
	mac.data[1] = 0x00;
	mac.data[2] = 0x5e;
	random_block(mac.data + 3, ETH_ADDR_LEN - 3);
	s = mac_to_ascii(&mac);
	obj = Tcl_NewStringObj(s, strlen(s) );
    } else if (argc == 2) {
	/* Treat the only argument as an integer range */
	range = xstrdup(argv[1]);

	s = strchr(range, ':');
	if (!s) {
	    nexp_error(interp, "invalid range. Must be in the form x:y");
	    free(range);
	    return TCL_ERROR;
	}
	*s++ = '\0';

	obj = Tcl_NewIntObj(random_int(atoi(range), atoi(s) ) );
    } else {
	nexp_error(interp, "usage: %s [x:y|number x:y|ip a.b.c.d/nn|"
			   "ip a.b.c.d:e.f.g.h|mac]", argv[0]);
	return TCL_ERROR;
    }

    free(range);

    Tcl_SetObjResult(interp, obj);

    return TCL_OK;
}

/********************************************************************
 *				 host				    *
 ********************************************************************/

static int
NExp_HostCmd(ClientData clientData _U_, Tcl_Interp *interp, int argc,
	     const char **argv)
{
    Tcl_Obj *obj;
    struct addr a;
    char *s;

    if (argc != 2) {
	nexp_error(interp, "%s: must have one arg: IP address", argv[0]);
	return TCL_ERROR;
    }

    if (addr_pton(argv[1], &a) == -1 || a.addr_type != ADDR_TYPE_IP
	|| a.addr_bits != IP_ADDR_BITS) {
	nexp_error(interp, "addr_pton(): %s", strerror(errno) );
	return TCL_ERROR;
    }

    s = addr_ntoa(&a);
    if (!s) {
	nexp_error(interp, "addr_ntoa(): %s", strerror(errno) );
	return TCL_ERROR;
    }

    obj = Tcl_NewStringObj(s, strlen(s) );
    Tcl_SetObjResult(interp, obj);

    return TCL_OK;
}

/********************************************************************
 *                               help                               *
 ********************************************************************/

static int
NExp_HelpCmd(ClientData clientData _U_, Tcl_Interp *interp _U_, int argc _U_,
	     const char **argv _U_)
{
    printf("\
Network Expect can be run in interactive mode or via scripts. In\n\
interactive mode commands are read from standard input. When running from\n\
a script, commands are read from a file.\n\
\n\
The most important Network Expect commands are:\n\
\n\
* spawn_network: spawns network listeners and speakers\n\
* send_network: creates and sends packets to a speaker\n\
* expect_network: expects for packets from a listener\n\
* send_expect: injects stimuli, receives answers, and matches stimuli an answers\n\
* pdu: manipulates PDU objects\n\
* packet: manipulates packet objects\n\
\n\
To exit from interactive mode type the End of File character (usually\n\
Control-D) or type \"exit\" at the \"netexpect>\" prompt.\n\
\n\
See nexp's man page for further information.\n");

    return TCL_OK;
}


/********************************************************************
 *				clear                               *
 ********************************************************************/

static int
NExp_ClearObjCmd(ClientData clientData _U_, Tcl_Interp *interp, int objc,
		 Tcl_Obj *CONST objv[])
{
    int index, retval;
    static const char *subcmds[] = {
	"dissection", NULL
    };
    enum subcmds {
	SUBCMD_DISSECTION
    };

    if (objc == 1) {
	nexp_error(interp, "wrong # args");
	return TCL_ERROR;
    }

    if (Tcl_GetIndexFromObj(interp, objv[1], subcmds, "subcmd", 0, &index)
	!= TCL_OK)
	return TCL_ERROR;

    switch (index) {
    case SUBCMD_DISSECTION:
	retval = pkt_clear_dissection(interp, objc - 1, &objv[1]);
	break;
    }

    return retval;
}

/********************************************************************
 *				 show                               *
 ********************************************************************/

static int
NExp_ShowObjCmd(ClientData clientData _U_, Tcl_Interp *interp, int objc,
		Tcl_Obj *CONST objv[])
{
    int retval;
    char *cmd;
    struct cdb cdb;

    cmd = copy_objv(objc, &objv[0]);

    if (cmd_parseargs(interp, cmd, &cdb) != 0) {
	free(cmd);
	return TCL_ERROR;
    }

    free(cmd);

    switch (cdb.code) {
    case CMD_SHOW_VERSION:
	Tcl_AppendResult(interp,
			 "Network Expect version ", nexp_version, "\n",
			 pcap_lib_version(), "\n",
			 "Wireshark Packet Analyzer version ",
			 epan_get_version(), "\n", NULL);
	retval = TCL_OK;
	break;
    case CMD_SHOW_DISSECTION_VARS:
	retval = pkt_show_dissection(interp);
	break;
    case CMD_SHOW_GHOSTS:
	ghosts_info();
	retval = TCL_OK;
	break;
    default:
	nexp_error(interp, "Invalid \"show\" subcommand.");
	retval = TCL_ERROR;
    }

    return retval;
}

static int
NExp_IsLocalObjCmd(ClientData clientData _U_, Tcl_Interp *interp, int objc,
		   Tcl_Obj *CONST objv[])
{
    char *netaddr_s, *ipaddr_s;
    struct addr netaddr, ipaddr, localnet;
    uint32_t netmask;
    Tcl_Obj *obj;

    if (objc != 3) {
	nexp_error(interp, "wrong # args");
	return TCL_ERROR;
    }

    netaddr_s = Tcl_GetStringFromObj(objv[1], NULL);
    if (addr_pton(netaddr_s, &netaddr) == -1
	|| netaddr.addr_type != ADDR_TYPE_IP
	|| netaddr.addr_bits == IP_ADDR_BITS
	|| addr_net(&netaddr, &localnet) == -1
	|| addr_btom(netaddr.addr_bits, &netmask, IP_ADDR_LEN) == -1) {
	nexp_error(interp, "invalid network address %s", netaddr_s);
	return TCL_ERROR;
    }

    ipaddr_s = Tcl_GetStringFromObj(objv[2], NULL);
    if (addr_pton(ipaddr_s, &ipaddr) == -1
	|| ipaddr.addr_type != ADDR_TYPE_IP
	|| ipaddr.addr_bits != IP_ADDR_BITS) {
	nexp_error(interp, "invalid IP address %s", ipaddr_s);
	return TCL_ERROR;
    }

    obj = Tcl_NewBooleanObj(localnet.addr_ip == (ipaddr.addr_ip & netmask) );
    Tcl_SetObjResult(interp, obj);

    return TCL_OK;
}

void
nexp_create_commands(Tcl_Interp *interp, struct nexp_cmd_data *c)
{
    for (; c->name; c++)
	if (c->objproc)
	    Tcl_CreateObjCommand(interp, c->name, c->objproc, c->data, NULL);
	else
	    Tcl_CreateCommand(interp, c->name, c->proc, c->data, NULL);
}

static struct nexp_cmd_data cmd_data[]  = {
    {"nexp_version", NULL, NExp_VersionCmd, 0, 0},
    {"sleep", NULL, NExp_SleepCmd, 0, 0},
    {"txdelta", NULL, NExp_TxDeltaCmd, 0, 0},
    {"nexp_continue", NULL, NExp_ContinueCmd, 0, 0},
    {"system", NULL, NExp_SystemCmd, 0, 0},
    {"random", NULL, NExp_RandomCmd, 0, 0},
    {"host", NULL, NExp_HostCmd, 0, 0},
    {"help", NULL, NExp_HelpCmd, 0, 0},
    {"?", NULL, NExp_HelpCmd, 0, 0},
    {"clear", NExp_ClearObjCmd, NULL, 0, 0},
    {"show", NExp_ShowObjCmd, NULL, 0, 0},
    {"islocal", NExp_IsLocalObjCmd, NULL, 0, 0},

    {NULL, NULL, NULL, 0, 0}
};

void
nexp_init_cmds(Tcl_Interp *interp)
{
    nexp_create_commands(interp, cmd_data);
}
