#include <assert.h>
#include <errno.h>
#include <poll.h>
#include <stddef.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <time.h>
#include <unistd.h>
#include <string>

#include "acceptor.h"
#include "log.h"
#include "mutexlock.h"
#include "serverpool.h"
#include "state.pb.h"
#include "stream.h"
#include "udpinput.h"
#include "util.h"
#include "version.h"

using namespace std;

extern ServerPool *servers;

UDPInput::UDPInput(const string &url)
	: url(url),
	  sock(-1)
{
	// Should be verified by the caller.
	string protocol;
	bool ok = parse_url(url, &protocol, &host, &port, &path);
	assert(ok);

	construct_header();

	pthread_mutex_init(&stats_mutex, NULL);
	stats.url = url;
	stats.bytes_received = 0;
	stats.data_bytes_received = 0;
	stats.connect_time = time(NULL);
}

UDPInput::UDPInput(const InputProto &serialized)
	: url(serialized.url()),
	  sock(serialized.sock())
{
	// Should be verified by the caller.
	string protocol;
	bool ok = parse_url(url, &protocol, &host, &port, &path);
	assert(ok);

	construct_header();

	pthread_mutex_init(&stats_mutex, NULL);
	stats.url = url;
	stats.bytes_received = serialized.bytes_received();
	stats.data_bytes_received = serialized.data_bytes_received();
	if (serialized.has_connect_time()) {
		stats.connect_time = serialized.connect_time();
	} else {
		stats.connect_time = time(NULL);
	}
}

InputProto UDPInput::serialize() const
{
	InputProto serialized;
	serialized.set_url(url);
	serialized.set_sock(sock);
	serialized.set_bytes_received(stats.bytes_received);
	serialized.set_data_bytes_received(stats.data_bytes_received);
	serialized.set_connect_time(stats.connect_time);
	return serialized;
}

void UDPInput::close_socket()
{
	safe_close(sock);
	sock = -1;
}
	
void UDPInput::construct_header()
{
	http_header =
		"HTTP/1.0 200 OK\r\n"
		"Content-type: application/octet-stream\r\n"
		"Cache-control: no-cache\r\n"
		"Server: " SERVER_IDENTIFICATION "\r\n"
	        "Connection: close\r\n";
}
	
void UDPInput::add_destination(int stream_index)
{
	stream_indices.push_back(stream_index);
	servers->set_header(stream_index, http_header, "");
}

void UDPInput::do_work()
{
	while (!should_stop()) {
		if (sock == -1) {
			int port_num = atoi(port.c_str());
			sockaddr_in6 addr = CreateAnyAddress(port_num);
			sock = create_server_socket(addr, UDP_SOCKET);
			if (sock == -1) {
				log(WARNING, "[%s] UDP socket creation failed. Waiting 0.2 seconds and trying again...",
				             url.c_str());
				usleep(200000);
				continue;
			}
		}

		// Wait for a packet, or a wakeup.
		bool activity = wait_for_activity(sock, POLLIN, NULL);
		if (!activity) {
			// Most likely, should_stop was set.
			continue;
		}

		int ret;
		do {
			ret = recv(sock, packet_buf, sizeof(packet_buf), 0);
		} while (ret == -1 && errno == EINTR);

		if (ret <= 0) {
			log_perror("recv");
			close_socket();
			continue;
		}

		{
			MutexLock lock(&stats_mutex);
			stats.bytes_received += ret;
			stats.data_bytes_received += ret;
		}
		
		for (size_t i = 0; i < stream_indices.size(); ++i) {
			servers->add_data(stream_indices[i], packet_buf, ret, SUITABLE_FOR_STREAM_START);
		}
	}
}

InputStats UDPInput::get_stats() const
{
	MutexLock lock(&stats_mutex);
	return stats;
}
