// Copyright (c) 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/webui/net_export_ui.h"

#include <string>

#include "base/bind.h"
#include "base/macros.h"
#include "base/memory/scoped_ptr.h"
#include "base/strings/string_util.h"
#include "base/strings/utf_string_conversions.h"
#include "base/values.h"
#include "chrome/browser/browser_process.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/common/features.h"
#include "chrome/common/url_constants.h"
#include "components/grit/components_resources.h"
#include "components/net_log/chrome_net_log.h"
#include "components/net_log/net_export_ui_constants.h"
#include "components/net_log/net_log_temp_file.h"
#include "content/public/browser/browser_thread.h"
#include "content/public/browser/url_data_source.h"
#include "content/public/browser/web_contents.h"
#include "content/public/browser/web_ui.h"
#include "content/public/browser/web_ui_data_source.h"
#include "content/public/browser/web_ui_message_handler.h"

#if BUILDFLAG(ANDROID_JAVA_UI)
#include "chrome/browser/android/intent_helper.h"
#endif

using content::BrowserThread;
using content::WebContents;
using content::WebUIMessageHandler;

namespace {

content::WebUIDataSource* CreateNetExportHTMLSource() {
  content::WebUIDataSource* source =
      content::WebUIDataSource::Create(chrome::kChromeUINetExportHost);

  source->SetJsonPath("strings.js");
  source->AddResourcePath(net_log::kNetExportUIJS, IDR_NET_LOG_NET_EXPORT_JS);
  source->SetDefaultResource(IDR_NET_LOG_NET_EXPORT_HTML);
  return source;
}

// This class receives javascript messages from the renderer.
// Note that the WebUI infrastructure runs on the UI thread, therefore all of
// this class's public methods are expected to run on the UI thread. All static
// functions except SendEmail run on FILE_USER_BLOCKING thread.
class NetExportMessageHandler
    : public WebUIMessageHandler,
      public base::SupportsWeakPtr<NetExportMessageHandler> {
 public:
  NetExportMessageHandler();
  ~NetExportMessageHandler() override;

  // WebUIMessageHandler implementation.
  void RegisterMessages() override;

  // Messages.
  void OnGetExportNetLogInfo(const base::ListValue* list);
  void OnStartNetLog(const base::ListValue* list);
  void OnStopNetLog(const base::ListValue* list);
  void OnSendNetLog(const base::ListValue* list);

 private:
  // Calls NetLogTempFile's ProcessCommand with DO_START and DO_STOP commands.
  static void ProcessNetLogCommand(
      base::WeakPtr<NetExportMessageHandler> net_export_message_handler,
      net_log::NetLogTempFile* net_log_temp_file,
      net_log::NetLogTempFile::Command command);

  // Returns the path to the file which has NetLog data.
  static base::FilePath GetNetLogFileName(
      net_log::NetLogTempFile* net_log_temp_file);

  // Send state/file information from NetLogTempFile.
  static void SendExportNetLogInfo(
      base::WeakPtr<NetExportMessageHandler> net_export_message_handler,
      net_log::NetLogTempFile* net_log_temp_file);

  // Send NetLog data via email. This runs on UI thread.
  static void SendEmail(const base::FilePath& file_to_send);

  // Call NetExportView.onExportNetLogInfoChanged JavsScript function in the
  // renderer, passing in |arg|. Takes ownership of |arg|.
  void OnExportNetLogInfoChanged(base::Value* arg);

  // Cache of g_browser_process->net_log()->net_log_temp_file().
  net_log::NetLogTempFile* net_log_temp_file_;

  base::WeakPtrFactory<NetExportMessageHandler> weak_ptr_factory_;

  DISALLOW_COPY_AND_ASSIGN(NetExportMessageHandler);
};

NetExportMessageHandler::NetExportMessageHandler()
    : net_log_temp_file_(g_browser_process->net_log()->net_log_temp_file()),
      weak_ptr_factory_(this) {
}

NetExportMessageHandler::~NetExportMessageHandler() {
  // Cancel any in-progress requests to collect net_log into temporary file.
  BrowserThread::PostTask(BrowserThread::FILE_USER_BLOCKING, FROM_HERE,
                          base::Bind(&net_log::NetLogTempFile::ProcessCommand,
                                     base::Unretained(net_log_temp_file_),
                                     net_log::NetLogTempFile::DO_STOP));
}

void NetExportMessageHandler::RegisterMessages() {
  DCHECK_CURRENTLY_ON(BrowserThread::UI);

  web_ui()->RegisterMessageCallback(
      net_log::kGetExportNetLogInfoHandler,
      base::Bind(&NetExportMessageHandler::OnGetExportNetLogInfo,
                 base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      net_log::kStartNetLogHandler,
      base::Bind(&NetExportMessageHandler::OnStartNetLog,
                 base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      net_log::kStopNetLogHandler,
      base::Bind(&NetExportMessageHandler::OnStopNetLog,
                 base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      net_log::kSendNetLogHandler,
      base::Bind(&NetExportMessageHandler::OnSendNetLog,
                 base::Unretained(this)));
}

void NetExportMessageHandler::OnGetExportNetLogInfo(
    const base::ListValue* list) {
  BrowserThread::PostTask(
      BrowserThread::FILE_USER_BLOCKING,
      FROM_HERE,
      base::Bind(&NetExportMessageHandler::SendExportNetLogInfo,
                 weak_ptr_factory_.GetWeakPtr(),
                 net_log_temp_file_));
}

void NetExportMessageHandler::OnStartNetLog(const base::ListValue* list) {
  std::string log_mode;
  bool result = list->GetString(0, &log_mode);
  DCHECK(result);

  net_log::NetLogTempFile::Command command;
  if (log_mode == "LOG_BYTES") {
    command = net_log::NetLogTempFile::DO_START_LOG_BYTES;
  } else if (log_mode == "NORMAL") {
    command = net_log::NetLogTempFile::DO_START;
  } else {
    DCHECK_EQ("STRIP_PRIVATE_DATA", log_mode);
    command = net_log::NetLogTempFile::DO_START_STRIP_PRIVATE_DATA;
  }

  ProcessNetLogCommand(weak_ptr_factory_.GetWeakPtr(), net_log_temp_file_,
                       command);
}

void NetExportMessageHandler::OnStopNetLog(const base::ListValue* list) {
  ProcessNetLogCommand(weak_ptr_factory_.GetWeakPtr(), net_log_temp_file_,
                       net_log::NetLogTempFile::DO_STOP);
}

void NetExportMessageHandler::OnSendNetLog(const base::ListValue* list) {
  content::BrowserThread::PostTaskAndReplyWithResult(
    content::BrowserThread::FILE_USER_BLOCKING,
        FROM_HERE,
        base::Bind(&NetExportMessageHandler::GetNetLogFileName,
                   base::Unretained(net_log_temp_file_)),
        base::Bind(&NetExportMessageHandler::SendEmail));
}

// static
void NetExportMessageHandler::ProcessNetLogCommand(
    base::WeakPtr<NetExportMessageHandler> net_export_message_handler,
    net_log::NetLogTempFile* net_log_temp_file,
    net_log::NetLogTempFile::Command command) {
  if (!BrowserThread::CurrentlyOn(BrowserThread::FILE_USER_BLOCKING)) {
    BrowserThread::PostTask(
        BrowserThread::FILE_USER_BLOCKING,
        FROM_HERE,
        base::Bind(&NetExportMessageHandler::ProcessNetLogCommand,
                   net_export_message_handler,
                   net_log_temp_file,
                   command));
    return;
  }

  DCHECK_CURRENTLY_ON(BrowserThread::FILE_USER_BLOCKING);
  net_log_temp_file->ProcessCommand(command);
  SendExportNetLogInfo(net_export_message_handler, net_log_temp_file);
}

// static
base::FilePath NetExportMessageHandler::GetNetLogFileName(
    net_log::NetLogTempFile* net_log_temp_file) {
  DCHECK_CURRENTLY_ON(BrowserThread::FILE_USER_BLOCKING);
  base::FilePath net_export_file_path;
  net_log_temp_file->GetFilePath(&net_export_file_path);
  return net_export_file_path;
}

// static
void NetExportMessageHandler::SendExportNetLogInfo(
    base::WeakPtr<NetExportMessageHandler> net_export_message_handler,
    net_log::NetLogTempFile* net_log_temp_file) {
  DCHECK_CURRENTLY_ON(BrowserThread::FILE_USER_BLOCKING);
  base::Value* value = net_log_temp_file->GetState();
  if (!BrowserThread::PostTask(
      BrowserThread::UI, FROM_HERE,
      base::Bind(&NetExportMessageHandler::OnExportNetLogInfoChanged,
                 net_export_message_handler,
                 value))) {
    // Failed posting the task, avoid leaking.
    delete value;
  }
}

// static
void NetExportMessageHandler::SendEmail(const base::FilePath& file_to_send) {
  if (file_to_send.empty())
    return;
  DCHECK_CURRENTLY_ON(BrowserThread::UI);

#if BUILDFLAG(ANDROID_JAVA_UI)
  std::string email;
  std::string subject = "net_internals_log";
  std::string title = "Issue number: ";
  std::string body =
      "Please add some informative text about the network issues.";
  base::FilePath::StringType file_to_attach(file_to_send.value());
  chrome::android::SendEmail(
      base::UTF8ToUTF16(email), base::UTF8ToUTF16(subject),
      base::UTF8ToUTF16(body), base::UTF8ToUTF16(title),
      base::UTF8ToUTF16(file_to_attach));
#endif
}

void NetExportMessageHandler::OnExportNetLogInfoChanged(base::Value* arg) {
  scoped_ptr<base::Value> value(arg);
  DCHECK_CURRENTLY_ON(BrowserThread::UI);
  web_ui()->CallJavascriptFunction(net_log::kOnExportNetLogInfoChanged, *arg);
}

}  // namespace

NetExportUI::NetExportUI(content::WebUI* web_ui) : WebUIController(web_ui) {
  web_ui->AddMessageHandler(new NetExportMessageHandler());

  // Set up the chrome://net-export/ source.
  Profile* profile = Profile::FromWebUI(web_ui);
  content::WebUIDataSource::Add(profile, CreateNetExportHTMLSource());
}
