// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __SimpleParser_h__
#define __SimpleParser_h__

#include <string>
#include <vector>
using namespace std;

namespace Puma {


/** 
 * Simple string parser. 
 * \class SimpleParser SimpleParser.h Puma/common/SimpleParser.h
 */
class SimpleParser {
public:
  /**
   * Remove enclosing double-quotes from a string.
   * \param str The string to strip.
   * \return String without the enclosing double-quotes.
   */
  static string strip (const string& str);
  /**
   * Splits a given string into tokens recognized by the given delimiters.
   * \param str The string to split.
   * \param tokens A vector for the tokens.
   * \param delimiters A string with the delimiters. Default is a space or tab.
   * \return Number of recognized tokens.
   */
  static int tokenize (const string& str, vector<string>& tokens, const string& delimiters = " ");

private:
  /**
   * Get the position of the next token.
   * \param line The string to tokenize.
   * \param delimiters A string with the token delimiters.
   * \param pos The current position in the string.
   * \param lastPos The last position in the string.
   * \return The position of the next token.
   */
  static string::size_type next (const string& line, const string& delimiters, string::size_type pos, string::size_type& lastPos);
  /**
   * Get the begin of the next token.
   * \param line The string to tokenize.
   * \param delimiters A string with the token delimiters.
   * \param pos The current position in the string.
   * \return The begin of the next token.
   */
  static string::size_type findBeginOfNextToken (const string& line, const string& delimiters, string::size_type pos);
  /**
   * Get the end of the next token.
   * \param line The string to tokenize.
   * \param delimiters A string with the token delimiters.
   * \param lastPos The last position in the string, i.e. the begin of the next token.
   * \return The end of the next token.
   */
  static string::size_type findEndOfNextToken (const string& line, const string& delimiters, string::size_type lastPos);
};


} // namespace Puma

#endif /* __SimpleParser_h__ */
