/* 	readwrite.c:
 *		stream channels routines
 *
 *	part of the martian_modem
 *
 *	description:
 *		stream implemented as pty
 *
 *
 *	Author: A. Chentsov
 * 	Copying: LGPL
 *
 */

#define _GNU_SOURCE


#include <stdio.h>
#include <unistd.h>		// close, unlink

// sigaction
#include <signal.h>

#include <termios.h>

#include <stdlib.h>

#include <sys/stat.h>

// fd sets
#include <sys/select.h>

// posix timers
#include "sysdep.h"
#include <time.h>		/*** (p)select */

// errno, EINTR
#include <errno.h>

#include <string.h>		/*** memset, memcmp */
#include <linux/major.h>	/*** pty major */

#include "mport.h"

#define PREFIX	"pty"
#include "log.h"
#include "common.h"

void pin_reset (int pin)
{
	if (pin != -1) 
		close (pin);
	unlink (config.symlink_name);
}

/* no file or broken link */
static int check_symbolic_link (char *symlink) {
	struct stat sl_stats;
	
	LOGDEBUG (Note, "checking file %s\n", symlink);
	if (stat (symlink, &sl_stats) == -1) {
		LOGSYSERR (Debug, "stat");
		if (errno == ENOENT)
			return 1;
	}
	else { /* file exists */
		if (! S_ISCHR (sl_stats.st_mode) || 
			major (sl_stats.st_rdev) !=  UNIX98_PTY_SLAVE_MAJOR) {
			LOGDEBUG (2, "%ld\n", sl_stats.st_dev);
			LOGERR ("File %s exists\n", config.symlink_name);
			return 0;
		}
	}

	if (lstat (symlink, &sl_stats) == 0)
		if ( ! S_ISLNK(sl_stats.st_mode) ) {
			LOGERR ("File %s exists and not a symlink\n", symlink);
			return 0;
		}

	return 1;
}

static int timer_pty, next_pty;
static char next_name[256];

static int pty_chown (int ptm) 
{
	char pts_name[256];

	if (ptsname_r (ptm, pts_name, sizeof pts_name - 1) == 0) {
		LOGDEBUG (Note, "chowning %s\n", pts_name);
				
		if (chown (pts_name, config.uid, config.gid) < 0) {
			LOGWARN ("failed to set tty owner\n");
			LOGSYSERR (Warning, "pty_chown");
			return -1;
		}
		else 
			return 0;
	} 
	else
		return -1;
}

static int pty_chmod (int ptm) 
{
	char pts_name[256];

	if (ptsname_r (ptm, pts_name, sizeof pts_name - 1) == 0) {
		LOGDEBUG (Note, "chmoding %s\n", pts_name);
				
		if (chmod (pts_name, config.mode) < 0) {
			LOGWARN ("failed to set tty permissions\n");
			LOGSYSERR (Warning, "pty_chmod");
			return -1;
		}
		else 
			return 0;
	} 
	else
		return -1;
}

static int setup_symbolic_link (char *path, int ptm) 
{
	char pts_name[256];
	char *name = (timer_pty == -1) ? pts_name : next_name;

	if (ptsname_r (ptm, name, sizeof pts_name - 1) == 0) {
		LOGDEBUG (2, "%s %s\n", 
				(timer_pty == -1) ? "using" : "exporting",
				name);
	} else
		return -1;

	unlink (path);
	
	return symlink (name, path);
}

extern mtimer_t ptytimer;
#define SIG_PTY_TIMER 	SIGRTMIN
		
int pin_check_dtr (void )
{
	struct termios term;
	tcgetattr (timer_pty, &term);
	return ! (cfgetospeed (&term) == B0 || cfgetispeed (&term) == B0);
}

static unsigned previous_dtr;

unsigned pin_check_mctrl (int changes)
{

	struct termios term;
	unsigned dtr, mask = 0;

	if (! changes) {
		struct termios term;

		tcgetattr (timer_pty, &term);
		previous_dtr = ! (cfgetospeed (&term) == B0 || cfgetispeed (&term) == B0);
		if (previous_dtr) 
			mask |= (TIOCM_DTR | TIOCM_RTS);

		return mask;
	}

	tcgetattr (timer_pty, &term);
	dtr = ! (cfgetospeed (&term) == B0 || cfgetispeed (&term) == B0);
	if (dtr ^ previous_dtr) {
		mask |= (TIOCM_DTR | TIOCM_RTS);
		previous_dtr = dtr;
		LOGDEBUG (Event, "hang-up %s\n", (!dtr) ? "on" : "off");
	}

	return mask;
}


/* pin_start_timer
 *	set to pulse every num ms and go
 */
void pin_start_timer (void )
{
	const struct timespec interval = {
		.tv_sec = 0,
		.tv_nsec = 200000000
	};
	const struct itimerspec pty_slice = {
		.it_value = interval,
		.it_interval = interval
	};

	int res = ptytimer.settime (ptytimer.id, 0, & pty_slice, NULL);
	if (res < 0) {
		LOGERR ("start timer error %d\n", res);
	}
}

void pin_stop_timer (void )
{
	struct itimerspec disarm; 
	memset (&disarm, 0, sizeof disarm);
	ptytimer.settime (ptytimer.id, 0, & disarm, NULL);
}

static int create_pty (void ) {
	int res;
	int ptm = getpt();
	if (ptm < 0) {
		LOGSYSERR (Error, "getpt");
		LOGERR ("devpts is possibly not mounted\n");
		return -1;
	}

	grantpt (ptm); unlockpt (ptm);

	if (config.uid != -1 || config.gid != -1) 
		pty_chown (ptm);
	

	if (config.permissions) 
		pty_chmod (ptm);
	

	// now set up symlink
	if (! check_symbolic_link (config.symlink_name))
		return -1;

	res = setup_symbolic_link (config.symlink_name, ptm);
	if (res != 0) {
		LOGSYSERR (Error, "symlink");
		return res;
	}

	return ptm;
}
/* pin_setup (pty)
 *	prepare master pty for connection
 *	pty - previous fd, 
 *	pty == -1 			  --> initialize
 *	! config.hide_pty || next_pty < 0 --> create pty, set link 
 *	config.hide_pty	&& next_pty 	  --> return prepared pty descriptor 
 */
int pin_setup (int pty) 
{
	int ptm = -1;

	// reset_pin (-1);
	if (pty == -1) {
		/* pre set */
		next_pty  =
		timer_pty = -1;
	}

	/* ASSERT (timer_pty == -1) */

	if (pty == -1 || !config.hide_pty || next_pty < 0) {
		ptm = create_pty ();
	}
	else { /* implies (config.hide_pty && next_pty >= 0) */
		ptm = next_pty;
		next_pty = -1;
		LOGDEBUG (Note, "attaching %s\n", next_name);
	}

	if (ptm < 0)
		return ptm;

	/* prepare the timer */
	else {
		struct termios term;
		timer_pty = ptm;

		/* not really necessary */
		/* timer will be started by mport anyway */
		/* and before this it prepares mctrl */
		tcgetattr (timer_pty, &term);
		previous_dtr = ! (cfgetospeed (&term) == B0 || cfgetispeed (&term) == B0);
	}

	return ptm;
}

int accept_client (int ptm) {
	struct termios startterm;
	struct timespec timeout;
	fd_set catch_set;
	int num;

	FD_ZERO (&catch_set);
	FD_SET (ptm, &catch_set);

	tcgetattr (ptm, &startterm);

	timeout.tv_sec	= 0;
	timeout.tv_nsec	= 200 * 1000000;

	/* return ptm; */
	while (select (ptm + 1, &catch_set, NULL, NULL, NULL) == -1) {
//	while ((num = pselect (ptm + 1, &catch_set, NULL, &catch_set, &timeout, NULL)) <= 0) {
		continue;
		if (num == 0 || errno == EINTR) {
			struct termios term;
			tcgetattr (ptm, &term);
			if (memcmp (&term, &startterm, sizeof term) == 0)
				continue;

			
			LOGDEBUG (Event, "termios changed\n");
			continue;

			break;
		}

		LOGSYSERR (Error, "select");
		return -1;
	}

	if (config.hide_pty) {
		next_pty = create_pty();
		if (next_pty < 0) 
			; /* we'll retry to create it later in pin_setup */
	}

	return ptm;
}	

/* pin_close
 *	client's off notification
 */
int pin_close (int pty) {
	/* pin_stop_timer(); */
	timer_pty = -1;
	return close (pty);
}

int pin_recv (int pin, char *buf, int length) {
	return read (pin, buf, length);
}

int pin_send (int pin, char *buf, int length) {
	return write (pin, buf, length);
}

int pin_get_send_size (int pty) {
	return 0x4000;
}

