/* { dg-do run { target { { {riscv_vector} && {rv64} } } } } */
/* { dg-options "-O2" } */

#include<cstdalign>
#include<cmath>
#include<cstddef>
#include<cstdint>
#include<iomanip>
#include<ios>
#include<iostream>
#include<memory>
#include<type_traits>
#include"riscv_vector.h"

using std::addressof;
using std::cerr;
using std::endl;
using std::int8_t;
using std::int16_t;
using std::int32_t;
using std::int64_t;
using std::uint8_t;
using std::uint16_t;
using std::uint32_t;
using std::uint64_t;
using std::ptrdiff_t;
using std::size_t;
using float16_t = _Float16;
using float32_t = float;
using float64_t = double;

template<class T, class T2>
constexpr T uint_to_float(T2 val) noexcept
{
  return *reinterpret_cast<T*>(&val);
}

constexpr const auto &f16(uint_to_float<float16_t, uint16_t>);
constexpr const auto &f32(uint_to_float<float32_t, uint32_t>);
constexpr const auto &f64(uint_to_float<float64_t, uint64_t>);

template<class T>
struct To_uint
{
  using type = std::conditional_t<
    sizeof(T) == 1, uint8_t, std::conditional_t<
      sizeof(T) == 2, uint16_t, std::conditional_t<
        sizeof(T) == 4, uint32_t, std::conditional_t<
          sizeof(T) == 8, uint64_t, void
        >
      >
    >
  >;
};

// isnan() does not support half type
template<class T>
struct To_float
{
  using type = std::conditional_t<
    std::is_same<T, float16_t>::value, float, std::conditional_t<
      std::is_same<T, float32_t>::value, float, std::conditional_t<
        std::is_same<T, float64_t>::value, double, float
      >
    >
  >;
};

template<class T>
using To_uint_t = typename To_uint<T>::type;

template<class T>
using To_isnan_float = typename To_float<T>::type;

template <class T>
void print_float(std::ostream &os, T val)
{
  using std::setw;
  os << std::hex << std::setfill('0') << setw(sizeof(T) * 2) << *reinterpret_cast<To_uint_t<T>*>(addressof(val)) << setw(0) << std::dec;
}

template <class T>
bool __attribute__((noinline))
check(const T *a, const T *b, size_t size)
{
  bool rv = true;
  for (size_t i = 0; i < (size / sizeof(T)); ++i) {
    if (reinterpret_cast<const To_uint_t<T> *>(a)[i] ==
        reinterpret_cast<const To_uint_t<T> *>(b)[i])
      continue;
    // floating negative zero == positive zero
    if ((std::is_floating_point_v<T> || std::is_same<T, float16_t>::value) &&
        (a[i] == b[i]))
      continue;
    // if both result are NaN, return true
    if ((std::is_same<T, float16_t>::value || std::is_floating_point_v<T>) &&
         std::isnan(static_cast<To_isnan_float<T>>(a[i])) &&
         std::isnan(static_cast<To_isnan_float<T>>(b[i])))
      continue;

    if (std::is_same<T, float16_t>::value) {
      cerr << std::hex << std::setfill('0') << std::setw(sizeof(T) * 2) << "["
           << i
           << "] result/golden:" << reinterpret_cast<const To_uint_t<T> *>(a)[i]
           << " != " << reinterpret_cast<const To_uint_t<T> *>(b)[i]
           << std::setw(0) << std::dec << endl;
    } else if constexpr (std::is_floating_point_v<T>) {
      cerr << "[" << i << "] result/golden:" << a[i] << "(";
      print_float(cerr, a[i]);
      cerr << ") != " << b[i] << "(";
      print_float(cerr, b[i]);
      cerr << ")" << endl;
    } else if constexpr (std::is_unsigned_v<T>) {
      cerr << "[" << i << "] result/golden: " << static_cast<uintmax_t>(a[i])
           << " != " << static_cast<uintmax_t>(b[i]) << endl;
    } else {
      cerr << "[" << i << "] result/golden:" << static_cast<intmax_t>(a[i])
           << " != " << static_cast<intmax_t>(b[i]) << endl;
    }
    rv = false;
  }
  return rv;
}

template <class T>
bool __attribute__((noinline))
check(const T a, const T golden)
{
  return check(addressof(a), addressof(golden), sizeof(T));
}



int main()
{
int return_value = 0;

size_t var_211 = 16u;
uint64_t var_210 [] = {3413955448782123771u, 5770915057685053022u};
size_t var_208 = 40u;
uint64_t var_207 [] = {11686681238268631633u, 17217089646841358934u, 4917608144621004608u, 2800347682848664045u, 13139172270140052590u};
size_t var_205 = 64u;
uint64_t var_204 [] = {2155812772479464600u, 3170612777771403051u, 1681134711641298423u, 8156897225395734044u, 17350593003963396555u, 15651252092822179957u, 8877172929823018199u, 10633166244594888701u};
size_t var_202 = 56u;
uint64_t var_201 [] = {7116734975071460602u, 12441625220002408740u, 2786654427366964938u, 13829764222335973050u, 4762436333200299360u, 11395394244577961796u, 8944665256461003018u};
size_t var_199 = 8u;
uint64_t var_198 [] = {5797432571859372123u};
size_t var_196 = 48u;
uint64_t var_195 [] = {12379964670789626889u, 13436296814432206489u, 4659013173841538396u, 6196494238874966981u, 6730617828857168898u, 8296127193014163030u};
size_t var_193 = 128u;
uint64_t var_192 [] = {1922947498486410857u, 16951704142057129556u, 5388344212325078882u, 5448258397625405342u, 1703194961276899101u, 9646419556518517367u, 16273654145082032307u, 3551281066422685306u, 10669730802081923878u, 8008712173776472922u, 5606540849143552077u, 17565367472452138214u, 2648738666890881870u, 2998195017278264134u, 2065211952219847096u, 18169511136733880014u};
size_t var_190 = 88u;
uint64_t var_189 [] = {15401374053725176026u, 15159971213859746434u, 15996044860437899159u, 10024478611893507882u, 17312537202804145326u, 13286484462280416129u, 5653446096408344014u, 1589575317326256715u, 1256266953356618611u, 5852474666329482281u, 8236254411035780100u};
size_t var_187 = 64u;
uint64_t var_186 [] = {1280769246380200460u, 14023690319710955253u, 13247119119507366408u, 7767358375133657872u, 10775238306218042028u, 6738525377594107821u, 1491839643263654600u, 17664969632620412524u};
size_t var_184 = 56u;
uint64_t var_183 [] = {11601634546274575695u, 18347035667798957494u, 13451989570814404378u, 456765431602104240u, 10052606093390301498u, 2437914183458767421u, 1539225365727642394u};
size_t var_181 = 256u;
float64_t var_180 [] = {f64(6187698773816159056u), f64(9776498580032912749u), f64(18269226306426886945u), f64(12870129469455680428u), f64(8111286762322373945u), f64(9289281457567414133u), f64(10639002934180078457u), f64(6195007212266041285u), f64(10865104194439280749u), f64(12224740054729171255u), f64(9698718863891634989u), f64(16464479761714716521u), f64(10670676975682736140u), f64(8962998596501157150u), f64(8766120491176435417u), f64(2717371620507294816u), f64(3837943005303818774u), f64(5129903549361771946u), f64(17642969427757166520u), f64(4007765606054665226u), f64(1647027738278082783u), f64(10690219188164819365u), f64(15228053371569790034u), f64(10344531484724988184u), f64(710760864140949770u), f64(15615459515851778470u), f64(9927951426822621308u), f64(4957518843951601594u), f64(7641934781991551769u), f64(14252201341143490410u), f64(10241407868670681836u), f64(8264550004758408330u)};
// 4.4611521076973743e+105, -1.0758468088979187e-271, -5.176835491667206e+296, -6.619601004503453e-65, 1.6970322567792563e+234, -2.979938591390489e-304, -4.950276524818289e-214, 1.438417155610527e+106, -6.428709007665846e-199, -4.896911894513031e-108, -6.986997664109763e-277, -1.1706628228396777e+176, -6.493431793743611e-212, 1.4787708549897394e+291, 1.0421560948913072e+278, 5.088139069960686e-127, 3.990899891565144e-52, 8.867669768871947e+34, -7.161121366560219e+254, 8.736910566334191e-41, 1.432775837239442e-198, -1.2969217753816681e-210, -2.726501234875066e+93, -9.799637510894672e-234, 3.7003426124468857e-261, -2.1441371653352167e+119, -1.4715600945527723e-261, 2.705444685329441e+23, 7.251873910049452e+202, -1.6087437906676368e+28, -1.2589823680213738e-240, 3.000901131149856e+244

size_t var_178 = 256u;
float64_t var_177 [] = {f64(12268982402006723053u), f64(1322662639198263372u), f64(10892092016535164825u), f64(14850734564898422239u), f64(4523747090804693057u), f64(15313422636296236341u), f64(6682154344242841459u), f64(8061073346680340919u), f64(6867894128954076650u), f64(2628301318073533989u), f64(10642317549290804120u), f64(16892802778425156682u), f64(16296392560269218515u), f64(4745368322886126618u), f64(2775903153273905952u), f64(4227008168332205515u), f64(1428962108255738703u), f64(15685550439166428737u), f64(8721174667886690300u), f64(6389093016280766630u), f64(915371217100750559u), f64(13102858936469556209u), f64(14325435142686572326u), f64(13100434734449401387u), f64(16751290947827411650u), f64(3724510073455636207u), f64(13176800956873038965u), f64(17108147774706966430u), f64(10272006816657832197u), f64(2166840198934152003u), f64(3126865544766700112u), f64(3984812247613312207u)};
// -4.3997420428351414e-105, 2.992377506963031e-220, -4.094307940460808e-197, -1.6450844168843733e+68, 2.8107367677455903e-06, -1.381116861022872e+99, 5.1692570320776656e+138, 7.443728610914345e+230, 1.2955022920890574e+151, 5.635813034929725e-133, -7.93999595358361e-214, -4.905389684561529e+204, -7.030330639871949e+164, 1807574008.0749574, 4.157950442133119e-123, 4.09547138485164e-26, 3.8417697419442373e-213, -1.0004951443736317e+124, 1.0315950134023327e+275, 1.3424750996227978e+119, 1.7924122107403712e-247, -2.427530528447156e-49, -1.2238156964734415e+33, -1.608666481217323e-49, -1.7914323013217252e+195, 1.0030184333291902e-59, -2.0599789696372077e-44, -1.2509772089128461e+219, -1.4156718618632884e-238, 7.88506931791861e-164, 1.1920211310425338e-99, 2.5915707560738407e-42

size_t var_175 = 256u;
float64_t var_174 [] = {f64(2300673783722258919u), f64(3467781142773234771u), f64(2345233581152263551u), f64(1868587953186126122u), f64(14774452287169263094u), f64(13922652697835041731u), f64(2386878503535424367u), f64(16547495967924420906u), f64(8942958103762160449u), f64(3567426388470419897u), f64(8511350044499123665u), f64(15485424648914767536u), f64(18041792644663864377u), f64(16469534312049021029u), f64(14429124106383846452u), f64(4626547782719094966u), f64(16292164099935306522u), f64(6083025255475019503u), f64(3059779533977631997u), f64(18445158560006719557u), f64(319680150795867099u), f64(4740419680481906389u), f64(1941324043466159959u), f64(9897452317974437047u), f64(9477233337600713314u), f64(13752614432306922614u), f64(5813536777454517309u), f64(5288450412570259281u), f64(1967956071157440815u), f64(480961478938269997u), f64(8707019076078059829u), f64(11673730106455029722u)};
// 6.9071744526170175e-155, 6.923400865513078e-77, 6.6691704148362785e-152, 8.989536893970336e-184, -1.3060543175303795e+63, -1520349.006217227, 3.8975989077786173e-149, -4.256422679919585e+181, 6.765582943934791e+289, 3.26810917231141e-70, 9.34508116044192e+260, -4.373920489375915e+110, -3.1445758471145765e+281, -2.4962431291505274e+176, -1.0394464537775981e+40, 20.799593290256517, -3.6558900666657e+164, 4.6559448455176147e+98, 3.927729333541973e-104, nan, 2.604906894646101e-287, 850733770.7895151, 6.542829947198155e-179, -1.3305439476683419e-263, -1.0971075979162187e-291, -6.4614803614591834e-06, 4.4214616749469996e+80, 3.630582797776415e+45, 3.8971392710020073e-177, 1.6200731142167154e-276, 1.1655694468313672e+274, -6.975510498054295e-145

size_t var_172 = 256u;
float64_t var_171 [] = {f64(13145979413242798804u), f64(18249049212272875542u), f64(5686275838205320814u), f64(12870065405008121229u), f64(15805617088064363265u), f64(3117616025036960168u), f64(7870459193100051406u), f64(3315587624886762670u), f64(14216093991953321985u), f64(6167009503567322368u), f64(5975854712965768936u), f64(1416387832014661521u), f64(11346221027479399797u), f64(4353915784447293972u), f64(6406040773904701275u), f64(17372403961692722173u), f64(13460688990661137053u), f64(5732641488031677947u), f64(8302093860329706908u), f64(8215692482325664371u), f64(12755166353681822613u), f64(7343163571706921621u), f64(6870297400155601231u), f64(17333100453460500437u), f64(4003950932397511804u), f64(14059799586156460318u), f64(9475401553823698285u), f64(4490512161850491093u), f64(14784584775073250232u), f64(16914986396484564094u), f64(17691693453950978772u), f64(6279794993919638703u)};
// -1.7461975733798845e-46, -2.254087553597392e+295, 1.4195485286653068e+72, -6.56557059752003e-65, -1.1016163011924826e+132, 2.8570537732987263e-100, 1.405831311328661e+218, 4.859391936026342e-87, -6.216679631024869e+25, 1.9368435102568067e+104, 3.1065594384724844e+91, 5.574344229526038e-214, -9.273922549336407e-167, 1.223681041535281e-17, 1.8421677350513206e+120, -5.895455415968474e+236, -1.937018094571547e-25, 1.7207015532833868e+75, 1.0034462709138033e+247, 1.6672588912213468e+241, -1.3763203172142895e-72, 8.019409183696546e+182, 1.9803645707206568e+151, -1.3683560131281643e+234, 4.847550088368645e-41, -2142187190128199.5, -7.863533030929549e-292, 1.6301853792973474e-08, -6.046522871793079e+63, -1.5101313690964458e+206, -1.2925742773419124e+258, 6.707760764200356e+111

size_t var_169 = 256u;
float64_t var_168 [] = {f64(13224541104105447859u), f64(15997926192552457056u), f64(4094401123071894779u), f64(1545273865987701927u), f64(15659828517197660337u), f64(6915099360988292381u), f64(8958927779396419652u), f64(8685124436318166792u), f64(11977899365095451790u), f64(7576084845947023205u), f64(11316065961843452656u), f64(14752197883205700933u), f64(17170357365308316348u), f64(16822887611204495095u), f64(929241872652159986u), f64(2416525204936783495u), f64(15618689276009020277u), f64(3390556701043326625u), f64(8619163760572831983u), f64(10045360014138332961u), f64(6473997592331735548u), f64(6092137062352193108u), f64(3920024461294893156u), f64(5925543705618177486u), f64(3040463253665551873u), f64(7832216569604445537u), f64(3260652848984743697u), f64(5264387754046825011u), f64(12437208545964732727u), f64(18032103368874551954u), f64(10928493007737974437u), f64(8736180249753791438u)};
// -3.301497051880758e-41, -7.824357091719868e+144, 5.48735852433986e-35, 2.232103621266717e-205, -1.950836146289716e+122, 1.9582178940580678e+154, 7.99321150819077e+290, 4.0167944017799746e+272, -1.5391400570269561e-124, 2.939249845694959e+198, -8.857511628261053e-169, -4.231905412538173e+61, -1.833646749363425e+223, -1.0956913814650382e+200, 1.525354422288766e-246, 3.9450657524151244e-147, -3.37529024887866e+119, 4.888478223185148e-82, 1.5178661307387018e+268, -1.0352474730035452e-253, 6.4148349766401e+124, 1.8877801849557373e+99, 1.2439000706934742e-46, 1.3805781208153196e+88, 1.9506651391806318e-105, 3.796953938141276e+215, 9.920037370309857e-91, 8.603177109513668e+43, -7.742646945720953e-94, -7.011301093277985e+280, -1.104748850672612e-194, 1.0091390349968425e+276

size_t var_166 = 256u;
float64_t var_165 [] = {f64(13033630116291160682u), f64(13772017025171702634u), f64(16452570950244089285u), f64(10209873756352972180u), f64(17119853242907834613u), f64(14245066108076568688u), f64(17699335364636616925u), f64(3116209871420214325u), f64(14463765876127797749u), f64(18031450949543799456u), f64(9210760816764462022u), f64(12120025697351591259u), f64(16995241348632377744u), f64(7915899406135338251u), f64(406377938911823835u), f64(16065803484180075706u), f64(14968592841381640912u), f64(1043738279784612710u), f64(15175473985883537247u), f64(13199875443726906855u), f64(15788495357954902769u), f64(17993851087565454968u), f64(12586360954608982297u), f64(4415008118551710253u), f64(17150087706896440098u), f64(11615746700657174614u), f64(6938776541782514361u), f64(3490026499357013992u), f64(10928968634653961674u), f64(10625417857303851178u), f64(8065318126631907784u), f64(11133154175594107343u)};
// -5.466899617959982e-54, -0.00012232414246285, -1.9068052426878134e+175, -9.817245583224469e-243, -7.743726959095287e+219, -5.150207386507462e+27, -4.005015332238227e+258, 2.214640668568439e-100, -2.2317187718408813e+42, -6.485377075266914e+280, 5.391940644291987e+307, -4.817288312673622e-115, -3.5803659177737863e+211, 1.5206806950059204e+221, 1.6995712452525845e-281, -2.7107509173254475e+149, -1.2268777941161214e+76, 6.743419999793945e-239, -8.311747154223234e+89, -7.0352015396959275e-43, -7.764425028177321e+130, -2.0386407935305358e+278, -7.138820689120118e-84, 1.5105950122009423e-13, -7.839999953325312e+221, -9.493370139050926e-149, 7.370629774472962e+155, 2.1462525702061535e-75, -1.177085676167517e-194, -6.11129595835773e-215, 1.4441344931021933e+231, -5.09370538181253e-181

size_t var_163 = 256u;
float64_t var_162 [] = {f64(2632921265178571059u), f64(7739823830606308283u), f64(17697032702367856286u), f64(5624473324698136339u), f64(1188717030892108108u), f64(10606512620327013719u), f64(978729597025812214u), f64(8118890521685184687u), f64(11007430839376624032u), f64(2348795760367716774u), f64(1666052550638850530u), f64(14766276574517822248u), f64(10218395580884964891u), f64(12864696557087944463u), f64(3179091509485438472u), f64(12860044981525174075u), f64(7151919826670670809u), f64(14826929065008683114u), f64(13933918488568616870u), f64(15897872117805872548u), f64(8685067125261860160u), f64(11984743900469710868u), f64(1136675127470815317u), f64(969291925713753836u), f64(2070265963710882159u), f64(7885445542263917712u), f64(13056849031755908653u), f64(13101601188901844550u), f64(10901757296049235944u), f64(4705400189456856572u), f64(7109449830182884890u), f64(13160060282619420098u)};
// 1.1453608349705888e-132, 2.6072597653252422e+209, -2.9417641187959127e+258, 1.0158277633751194e+68, 3.2124591426633125e-229, -3.24575766418193e-216, 3.0969364862449104e-243, 5.58278810533562e+234, -2.048039168742427e-189, 1.1741851527965889e-151, 2.5925802850748534e-197, -3.6448917413702844e+62, -3.6360331565189224e-242, -2.917920547847776e-65, 3.558295462025385e-96, -1.4277604418874809e-65, 1.2622695056025323e+170, -4.177465611769624e+66, -8184873.255941307, -1.5432460055462645e+138, 3.9823733042040855e+272, -4.3404967266843524e-124, 1.1212265530695812e-232, 7.182391930036128e-244, 2.8010811508046684e-170, 1.3559786058413101e+219, -2.009392564122529e-52, -1.949809438996329e-49, -1.7941070300540528e-196, 3793173.5205600243, 1.904646631992245e+167, -1.569995522922617e-45

size_t var_160 = 256u;
float64_t var_159 [] = {f64(14399718874005915098u), f64(15571815439736755123u), f64(15184880873889889051u), f64(9847979870979827366u), f64(12421407484372908519u), f64(7713300078375383008u), f64(2092709178248925316u), f64(6013523761173124620u), f64(8690015471402520246u), f64(7665390200081996829u), f64(12474830707341252676u), f64(2806689855448223324u), f64(2930519840717419624u), f64(7557401689543362471u), f64(6194550792424245657u), f64(10363031979926300333u), f64(210521861724201706u), f64(7426682387493218810u), f64(10159668005810775079u), f64(7834135109053839824u), f64(10035779077385053057u), f64(10929870836272427382u), f64(11180733598665910010u), f64(6988926481853656685u), f64(10302766308584164910u), f64(2362866735761578619u), f64(16566436948637593932u), f64(7657618613119722783u), f64(2704140616799128185u), f64(14022246625121577556u), f64(16360055489489409415u), f64(5848231571965353358u)};
// -1.173879235740122e+38, -2.5810004906693034e+116, -3.5054832194770695e+90, -6.554597307406097e-267, -6.630856606830876e-95, 4.3577672186230927e+207, 8.875460424589357e-169, 1.0600734757558945e+94, 8.498977317699981e+272, 2.655056358903447e+204, -2.4163244305828842e-91, 4.687153285841857e-121, 8.867909590829656e-113, 1.6150294500628063e+197, 1.345442791805468e+106, -1.6994033973485e-232, 1.3663132941226661e-294, 2.93717316375549e+188, -4.346659678864265e-246, 5.2655009521218076e+215, -2.3709150338905216e-254, -1.3142991072045571e-194, -8.003307628270562e-178, 1.6285887920683165e+159, -1.6449565503335684e-236, 1.0153452684011588e-150, -7.903053294195814e+182, 8.358041199622042e+203, 6.64698250963969e-128, -6879477289642.582, -1.2545086730594271e+169, 9.520710064215625e+82

size_t var_157 = 256u;
float64_t var_156 [] = {f64(7255389652619498159u), f64(11863592198503758248u), f64(1969861952934041841u), f64(14465802635759213596u), f64(11987915719740930989u), f64(8348926455447142360u), f64(18125216623108691390u), f64(3871463617952730428u), f64(13438223262881941206u), f64(4221934333025834245u), f64(4996973116443597634u), f64(17474422668216801327u), f64(11725728403266292315u), f64(7035920488256748319u), f64(14861010728333006889u), f64(3059446796589634216u), f64(16594761466888570085u), f64(12940937831007012101u), f64(16899747385326341061u), f64(10970432581561643055u), f64(15997942716186654365u), f64(9188923438452309560u), f64(10714463679512932715u), f64(10214480283686026984u), f64(2157908360834425177u), f64(16195744088095360215u), f64(6298184597360631864u), f64(2384636251274771142u), f64(768186602526962554u), f64(11394792936284522431u), f64(10786182531884216994u), f64(7168677360139691009u)};
// 1.0334573093152597e+177, -3.512642050519687e-132, 5.516797633442999e-177, -2.93653754293034e+42, -7.003982839140666e-124, 1.3131484076683424e+250, -1.2231619672889087e+287, 7.003111936882136e-50, -6.0906836879560264e-27, 1.8840860783382415e-26, 1.1999673595322263e+26, -3.7597876802005495e+243, -2.1449060009502597e-141, 2.318173484551994e+162, -7.795763884493084e+68, 3.721560354577447e-104, -6.041388700192658e+184, -3.6477524459116975e-60, -1.502660994358613e+205, -6.752064838664454e-192, -7.847264381607677e+144, 1.8972327765919883e+306, -5.299231972045231e-209, -2.0062927209415055e-242, 2.0003256246037705e-164, -1.293434978849255e+158, 1.1373563518337742e+113, 2.924164175177123e-149, 2.6168617111373365e-257, -1.6007971485693217e-163, -3.233003820801041e-204, 1.7063211043344023e+171

size_t var_154 = 256u;
float64_t var_153 [] = {f64(3105541477770240205u), f64(15227122452644821730u), f64(11678207403249918316u), f64(11128540721469280860u), f64(18275261683157975961u), f64(6461767542735320167u), f64(3109276659177311872u), f64(4266815001250444883u), f64(1612238301678626405u), f64(14093772356554162507u), f64(4201917729312711379u), f64(2860219621414878541u), f64(150937946329558483u), f64(14210177997135595118u), f64(16665578249559877062u), f64(8155633507205672198u), f64(14839034989466928520u), f64(6138980857711264887u), f64(15729868420075129818u), f64(10582523735765810008u), f64(12889391048551104296u), f64(18433971604506014850u), f64(8676317621761077002u), f64(9342788639089810943u), f64(1891945788602585138u), f64(10298412558729337383u), f64(12420181664158716352u), f64(15074325653973558305u), f64(1221095909190347100u), f64(6952866046159325456u), f64(6747173569448055456u), f64(3770367332817146332u)};
// 4.482597982526992e-101, -2.2953290459445205e+93, -1.3876185986783294e-144, -2.488068483251321e-181, -1.2578064004084964e+297, 9.522490838457363e+123, 7.990095590127968e-101, 1.8836573331597036e-23, 6.495575795920485e-201, -4.168142169120652e+17, 8.184258530893848e-28, 1.7390228660324606e-117, 1.4477864297130512e-298, -2.5017207003516037e+25, -3.3455037516407536e+189, 1.5582739660028077e+237, -2.598482837820986e+67, 2.5254895685248244e+102, -9.381689972569665e+126, -8.123708368435597e-218, -1.2394290799474622e-63, -5.231026904268411e+307, 1.0342864302420205e+272, -1.1317219513511448e-300, 3.302660128750505e-182, -8.388294784296539e-237, -5.496155351184002e-95, -1.423835876822299e+83, 4.80278823243729e-227, 6.337561690461767e+156, 1.1447935694863294e+143, 1.2130182497903373e-56

size_t var_151 = 256u;
float64_t var_150 [] = {f64(12874656926157541625u), f64(10984361015792627889u), f64(4749994239954255872u), f64(2507944306858608414u), f64(2540008438893585319u), f64(3503256139372151119u), f64(15438029745595274859u), f64(16289407516372582417u), f64(11439186572662330235u), f64(2275939344695701572u), f64(6325322638742505882u), f64(12773319639513182300u), f64(12091917200326777123u), f64(14901662909839393349u), f64(14362263934758655200u), f64(10322547615606009407u), f64(12020404286899312514u), f64(11715727083193098381u), f64(7789781619114644950u), f64(4420926019493481983u), f64(6961562765584694572u), f64(9447434243951189849u), f64(9773179847478454830u), f64(5113924339932864446u), f64(11836669285237257358u), f64(1714811448298014619u), f64(2974566882611829175u), f64(12171651892852916674u), f64(1942383630757053263u), f64(14427654890492813348u), f64(7949545421220510370u), f64(10842857940942377413u)};
// -1.3279442969267285e-64, -5.7126034259879e-191, 3673473517.0, 4.9216388223873334e-141, 6.7016535128759384e-139, 1.6617926499019489e-74, -2.8347139154509724e+107, -2.2747158128529418e+164, -1.4363121221015763e-160, 1.584958734257381e-156, 7.406426345287954e+114, -2.2579664519032115e-71, -6.170299816256117e-117, -4.050173433306196e+71, -3.533158536686456e+35, -3.3526906713116416e-235, -1.031969187757535e-121, -4.638193538312987e-142, 5.6522712684828104e+212, 3.735229714087532e-13, 2.4403723366100964e+157, -1.0971595804690102e-293, -6.399472919878548e-272, 7.888706862480327e+33, -5.584975159330578e-134, 4.834667193907169e-194, 7.911972107345559e-110, -1.3718912362953205e-111, 7.994338278617173e-179, -8.618291512517948e+39, 2.6689428921546715e+223, -2.087800215285169e-200

uint64_t var_148 [] = {10110301610812824219u, 18440176424075637850u};
float64_t var_147 [] = {f64(12874656926157541625u), f64(10984361015792627889u), f64(14503716710727919861u), f64(2507944306858608414u), f64(2540008438893585319u), f64(3503256139372151119u), f64(15438029745595274859u), f64(16289407516372582417u), f64(11439186572662330235u), f64(2275939344695701572u), f64(6325322638742505882u), f64(12773319639513182300u), f64(12091917200326777123u), f64(14901662909839393349u), f64(14362263934758655200u), f64(10322547615606009407u), f64(12020404286899312514u), f64(11715727083193098381u), f64(7789781619114644950u), f64(4420926019493481983u), f64(6961562765584694572u), f64(9447434243951189849u), f64(9773179847478454830u), f64(5113924339932864446u), f64(11836669285237257358u), f64(1714811448298014619u), f64(2974566882611829175u), f64(12171651892852916674u), f64(1942383630757053263u), f64(14427654890492813348u), f64(7949545421220510370u), f64(10842857940942377413u)};
// -1.3279442969267285e-64, -5.7126034259879e-191, -1.0504878265636506e+45, 4.9216388223873334e-141, 6.7016535128759384e-139, 1.6617926499019489e-74, -2.8347139154509724e+107, -2.2747158128529418e+164, -1.4363121221015763e-160, 1.584958734257381e-156, 7.406426345287954e+114, -2.2579664519032115e-71, -6.170299816256117e-117, -4.050173433306196e+71, -3.533158536686456e+35, -3.3526906713116416e-235, -1.031969187757535e-121, -4.638193538312987e-142, 5.6522712684828104e+212, 3.735229714087532e-13, 2.4403723366100964e+157, -1.0971595804690102e-293, -6.399472919878548e-272, 7.888706862480327e+33, -5.584975159330578e-134, 4.834667193907169e-194, 7.911972107345559e-110, -1.3718912362953205e-111, 7.994338278617173e-179, -8.618291512517948e+39, 2.6689428921546715e+223, -2.087800215285169e-200

uint64_t var_145 [] = {16u, 16u};
float64_t var_143 [] = {f64(3105541477770240205u), f64(11645929167617686076u), f64(11678207403249918316u), f64(11128540721469280860u), f64(18275261683157975961u), f64(6461767542735320167u), f64(3109276659177311872u), f64(4266815001250444883u), f64(1612238301678626405u), f64(14093772356554162507u), f64(4201917729312711379u), f64(2860219621414878541u), f64(150937946329558483u), f64(14210177997135595118u), f64(16665578249559877062u), f64(8155633507205672198u), f64(14839034989466928520u), f64(6138980857711264887u), f64(15729868420075129818u), f64(10582523735765810008u), f64(12889391048551104296u), f64(18433971604506014850u), f64(8676317621761077002u), f64(9342788639089810943u), f64(1891945788602585138u), f64(10298412558729337383u), f64(12420181664158716352u), f64(15074325653973558305u), f64(1221095909190347100u), f64(6952866046159325456u), f64(6747173569448055456u), f64(3770367332817146332u)};
// 4.482597982526992e-101, -9.588681720006641e-147, -1.3876185986783294e-144, -2.488068483251321e-181, -1.2578064004084964e+297, 9.522490838457363e+123, 7.990095590127968e-101, 1.8836573331597036e-23, 6.495575795920485e-201, -4.168142169120652e+17, 8.184258530893848e-28, 1.7390228660324606e-117, 1.4477864297130512e-298, -2.5017207003516037e+25, -3.3455037516407536e+189, 1.5582739660028077e+237, -2.598482837820986e+67, 2.5254895685248244e+102, -9.381689972569665e+126, -8.123708368435597e-218, -1.2394290799474622e-63, -5.231026904268411e+307, 1.0342864302420205e+272, -1.1317219513511448e-300, 3.302660128750505e-182, -8.388294784296539e-237, -5.496155351184002e-95, -1.423835876822299e+83, 4.80278823243729e-227, 6.337561690461767e+156, 1.1447935694863294e+143, 1.2130182497903373e-56

uint64_t var_142 [] = {8u};
uint64_t var_140 [] = {2174953555673768977u, 2225785007977329183u, 17185955649875337423u, 15256053411745002566u, 1493105653971291160u};
float64_t var_139 [] = {f64(13879594102898563438u), f64(11863592198503758248u), f64(13200161071657400096u), f64(14465802635759213596u), f64(11987915719740930989u), f64(14727513166716404377u), f64(18125216623108691390u), f64(3871463617952730428u), f64(13438223262881941206u), f64(15120750088449560219u), f64(4996973116443597634u), f64(17474422668216801327u), f64(11725728403266292315u), f64(7035920488256748319u), f64(14861010728333006889u), f64(3059446796589634216u), f64(16594761466888570085u), f64(12940937831007012101u), f64(16899747385326341061u), f64(10970432581561643055u), f64(15997942716186654365u), f64(9188923438452309560u), f64(10714463679512932715u), f64(10214480283686026984u), f64(2157908360834425177u), f64(16195744088095360215u), f64(6298184597360631864u), f64(2384636251274771142u), f64(768186602526962554u), f64(11394792936284522431u), f64(10786182531884216994u), f64(7168677360139691009u)};
// -1934.3248362556, -3.512642050519687e-132, -7.350786487711076e-43, -2.93653754293034e+42, -7.003982839140666e-124, -9.359149702594958e+59, -1.2231619672889087e+287, 7.003111936882136e-50, -6.0906836879560264e-27, -1.8413138694839325e+86, 1.1999673595322263e+26, -3.7597876802005495e+243, -2.1449060009502597e-141, 2.318173484551994e+162, -7.795763884493084e+68, 3.721560354577447e-104, -6.041388700192658e+184, -3.6477524459116975e-60, -1.502660994358613e+205, -6.752064838664454e-192, -7.847264381607677e+144, 1.8972327765919883e+306, -5.299231972045231e-209, -2.0062927209415055e-242, 2.0003256246037705e-164, -1.293434978849255e+158, 1.1373563518337742e+113, 2.924164175177123e-149, 2.6168617111373365e-257, -1.6007971485693217e-163, -3.233003820801041e-204, 1.7063211043344023e+171

uint64_t var_137 [] = {72u, 16u, 0u, 0u, 40u};
uint64_t var_135 [] = {13247355471621574986u, 11624887258427325317u, 9942614915966752235u, 4139325451928433498u, 4665401268889591332u, 9660725369276401929u, 693164432746007111u, 14002700909243782976u};
float64_t var_134 [] = {f64(14399718874005915098u), f64(15571815439736755123u), f64(15184880873889889051u), f64(9847979870979827366u), f64(1228142684408972675u), f64(3371124014568150452u), f64(8087692759457401387u), f64(9609384580782505387u), f64(1477597197320753420u), f64(15104241445223582113u), f64(4381946958418744703u), f64(2806689855448223324u), f64(2930519840717419624u), f64(7557401689543362471u), f64(6194550792424245657u), f64(10363031979926300333u), f64(210521861724201706u), f64(7426682387493218810u), f64(10159668005810775079u), f64(7834135109053839824u), f64(10035779077385053057u), f64(10929870836272427382u), f64(11180733598665910010u), f64(6988926481853656685u), f64(10302766308584164910u), f64(2362866735761578619u), f64(16566436948637593932u), f64(7657618613119722783u), f64(2704140616799128185u), f64(14022246625121577556u), f64(16360055489489409415u), f64(5848231571965353358u)};
// -1.173879235740122e+38, -2.5810004906693034e+116, -3.5054832194770695e+90, -6.554597307406097e-267, 1.4373066879102555e-226, 2.5365516056520468e-83, 4.5421550575223275e+232, -7.368321057939974e-283, 6.646130444492209e-210, -1.4106296941045448e+85, 8.827077666729902e-16, 4.687153285841857e-121, 8.867909590829656e-113, 1.6150294500628063e+197, 1.345442791805468e+106, -1.6994033973485e-232, 1.3663132941226661e-294, 2.93717316375549e+188, -4.346659678864265e-246, 5.2655009521218076e+215, -2.3709150338905216e-254, -1.3142991072045571e-194, -8.003307628270562e-178, 1.6285887920683165e+159, -1.6449565503335684e-236, 1.0153452684011588e-150, -7.903053294195814e+182, 8.358041199622042e+203, 6.64698250963969e-128, -6879477289642.582, -1.2545086730594271e+169, 9.520710064215625e+82

uint64_t var_132 [] = {80u, 32u, 72u, 48u, 64u, 56u, 48u, 40u};
uint64_t var_130 [] = {17990819689217368475u, 14267941834120012433u, 3465934470447936347u, 9314078761803266462u, 11918246364261742983u, 9337666089742224978u, 15262252020620409617u};
float64_t var_129 [] = {f64(2632921265178571059u), f64(16122410467215816819u), f64(17697032702367856286u), f64(5624473324698136339u), f64(5392093733751496085u), f64(10606512620327013719u), f64(978729597025812214u), f64(1563413232623796983u), f64(11007430839376624032u), f64(2348795760367716774u), f64(1666052550638850530u), f64(14766276574517822248u), f64(10218395580884964891u), f64(6974100765172843093u), f64(3179091509485438472u), f64(12860044981525174075u), f64(7151919826670670809u), f64(14826929065008683114u), f64(13933918488568616870u), f64(15897872117805872548u), f64(8685067125261860160u), f64(11984743900469710868u), f64(1136675127470815317u), f64(969291925713753836u), f64(2070265963710882159u), f64(7885445542263917712u), f64(13056849031755908653u), f64(13101601188901844550u), f64(10901757296049235944u), f64(4705400189456856572u), f64(7109449830182884890u), f64(13160060282619420098u)};
// 1.1453608349705888e-132, -1.587369346275365e+153, -2.9417641187959127e+258, 1.0158277633751194e+68, 3.0777367264321676e+52, -3.24575766418193e-216, 3.0969364862449104e-243, 3.659869695523208e-204, -2.048039168742427e-189, 1.1741851527965889e-151, 2.5925802850748534e-197, -3.6448917413702844e+62, -3.6360331565189224e-242, 1.715045151657037e+158, 3.558295462025385e-96, -1.4277604418874809e-65, 1.2622695056025323e+170, -4.177465611769624e+66, -8184873.255941307, -1.5432460055462645e+138, 3.9823733042040855e+272, -4.3404967266843524e-124, 1.1212265530695812e-232, 7.182391930036128e-244, 2.8010811508046684e-170, 1.3559786058413101e+219, -2.009392564122529e-52, -1.949809438996329e-49, -1.7941070300540528e-196, 3793173.5205600243, 1.904646631992245e+167, -1.569995522922617e-45

uint64_t var_127 [] = {8u, 104u, 32u, 8u, 32u, 56u, 32u};
uint64_t var_125 [] = {7749970125292175198u};
float64_t var_124 [] = {f64(6717432649256650565u), f64(13772017025171702634u), f64(16452570950244089285u), f64(10209873756352972180u), f64(17119853242907834613u), f64(14245066108076568688u), f64(17699335364636616925u), f64(3116209871420214325u), f64(14463765876127797749u), f64(18031450949543799456u), f64(9210760816764462022u), f64(12120025697351591259u), f64(16995241348632377744u), f64(7915899406135338251u), f64(406377938911823835u), f64(16065803484180075706u), f64(14968592841381640912u), f64(1043738279784612710u), f64(15175473985883537247u), f64(13199875443726906855u), f64(15788495357954902769u), f64(17993851087565454968u), f64(12586360954608982297u), f64(4415008118551710253u), f64(17150087706896440098u), f64(11615746700657174614u), f64(6938776541782514361u), f64(3490026499357013992u), f64(10928968634653961674u), f64(10625417857303851178u), f64(8065318126631907784u), f64(11133154175594107343u)};
// 1.1963237920104146e+141, -0.00012232414246285, -1.9068052426878134e+175, -9.817245583224469e-243, -7.743726959095287e+219, -5.150207386507462e+27, -4.005015332238227e+258, 2.214640668568439e-100, -2.2317187718408813e+42, -6.485377075266914e+280, 5.391940644291987e+307, -4.817288312673622e-115, -3.5803659177737863e+211, 1.5206806950059204e+221, 1.6995712452525845e-281, -2.7107509173254475e+149, -1.2268777941161214e+76, 6.743419999793945e-239, -8.311747154223234e+89, -7.0352015396959275e-43, -7.764425028177321e+130, -2.0386407935305358e+278, -7.138820689120118e-84, 1.5105950122009423e-13, -7.839999953325312e+221, -9.493370139050926e-149, 7.370629774472962e+155, 2.1462525702061535e-75, -1.177085676167517e-194, -6.11129595835773e-215, 1.4441344931021933e+231, -5.09370538181253e-181

uint64_t var_122 [] = {0u};
uint64_t var_120 [] = {8448416982140549461u, 3258302210728262219u, 8199730978408770741u, 9128214378827162799u, 4536927345912178180u, 2991083242117142323u};
float64_t var_119 [] = {f64(13224541104105447859u), f64(15997926192552457056u), f64(3195903267007773770u), f64(1545273865987701927u), f64(6215025560157673690u), f64(6915099360988292381u), f64(2632921648119447519u), f64(17895236051778914459u), f64(11977899365095451790u), f64(7576084845947023205u), f64(9278818693610826823u), f64(14752197883205700933u), f64(17170357365308316348u), f64(16822887611204495095u), f64(929241872652159986u), f64(2416525204936783495u), f64(15618689276009020277u), f64(3390556701043326625u), f64(8619163760572831983u), f64(10045360014138332961u), f64(6473997592331735548u), f64(6092137062352193108u), f64(3920024461294893156u), f64(5925543705618177486u), f64(3040463253665551873u), f64(7832216569604445537u), f64(3260652848984743697u), f64(5264387754046825011u), f64(12437208545964732727u), f64(18032103368874551954u), f64(10928493007737974437u), f64(8736180249753791438u)};
// -3.301497051880758e-41, -7.824357091719868e+144, 4.893148204050249e-95, 2.232103621266717e-205, 2.97353410506837e+107, 1.9582178940580678e+154, 1.1454207318030275e-132, -5.208964322758926e+271, -1.5391400570269561e-124, 2.939249845694959e+198, -5.977038512818269e-305, -4.231905412538173e+61, -1.833646749363425e+223, -1.0956913814650382e+200, 1.525354422288766e-246, 3.9450657524151244e-147, -3.37529024887866e+119, 4.888478223185148e-82, 1.5178661307387018e+268, -1.0352474730035452e-253, 6.4148349766401e+124, 1.8877801849557373e+99, 1.2439000706934742e-46, 1.3805781208153196e+88, 1.9506651391806318e-105, 3.796953938141276e+215, 9.920037370309857e-91, 8.603177109513668e+43, -7.742646945720953e-94, -7.011301093277985e+280, -1.104748850672612e-194, 1.0091390349968425e+276

uint64_t var_117 [] = {48u, 48u, 56u, 80u, 32u, 16u};
uint64_t var_115 [] = {12453453393991684959u, 1655499287040947693u, 6189547968460419452u, 4629587434453153441u, 16888425463350897627u, 17105020277016971372u, 7164930518073510035u, 10387124655452247732u, 9432696966287778066u, 9703442708627101798u, 10649378859654593450u, 4434227414598513652u, 16404836041053194768u, 13937807261870445194u, 2491338157831118301u, 3743397714460008323u};
float64_t var_114 [] = {f64(13145979413242798804u), f64(18249049212272875542u), f64(11866001986412525350u), f64(11677155855818879728u), f64(15805617088064363265u), f64(3117616025036960168u), f64(295759759671089251u), f64(3315587624886762670u), f64(14216093991953321985u), f64(6167009503567322368u), f64(5975854712965768936u), f64(1416387832014661521u), f64(11346221027479399797u), f64(4353915784447293972u), f64(10092973457820298185u), f64(4793549766813476845u), f64(18041407716158862495u), f64(5732641488031677947u), f64(1877837052694714372u), f64(17406371823426831493u), f64(12755166353681822613u), f64(7343163571706921621u), f64(3099293917688714607u), f64(17333100453460500437u), f64(9758115797673631714u), f64(14059799586156460318u), f64(9475401553823698285u), f64(5222253444020763116u), f64(14784584775073250232u), f64(16914986396484564094u), f64(17691693453950978772u), f64(6279794993919638703u)};
// -1.7461975733798845e-46, -2.254087553597392e+295, -5.0203281864944534e-132, -1.184886369616024e-144, -1.1016163011924826e+132, 2.8570537732987263e-100, 6.862206746477298e-289, 4.859391936026342e-87, -6.216679631024869e+25, 1.9368435102568067e+104, 3.1065594384724844e+91, 5.574344229526038e-214, -9.273922549336407e-167, 1.223681041535281e-17, -1.5228291748021803e-250, 3038751425116.9907, -2.9003344503478956e+281, 1.7207015532833868e+75, 3.696942236742707e-183, -1.0371652147038235e+239, -1.3763203172142895e-72, 8.019409183696546e+182, 1.6880402163604667e-101, -1.3683560131281643e+234, -6.421633541911156e-273, -2142187190128199.5, -7.863533030929549e-292, 1.3704529179157082e+41, -6.046522871793079e+63, -1.5101313690964458e+206, -1.2925742773419124e+258, 6.707760764200356e+111

uint64_t var_112 [] = {48u, 144u, 144u, 216u, 24u, 120u, 128u, 16u, 128u, 112u, 192u, 216u, 176u, 176u, 152u, 48u};
uint64_t var_110 [] = {1974574619820150201u, 6828673976832995218u, 4469486898657243538u, 12171083690363424775u, 4733747186279628566u, 804531996737134521u, 17932648977680029171u, 7199266062394303577u, 11206859691607683769u, 2456110949156168632u, 4233775760192098211u};
float64_t var_109 [] = {f64(14644260122741405446u), f64(3467781142773234771u), f64(7831633878900570457u), f64(1868587953186126122u), f64(14774452287169263094u), f64(13922652697835041731u), f64(2386878503535424367u), f64(16547495967924420906u), f64(8942958103762160449u), f64(3567426388470419897u), f64(13457489635873293098u), f64(15485424648914767536u), f64(197721995978144155u), f64(8306488134467655190u), f64(8909600536278509230u), f64(4626547782719094966u), f64(18249362645861906414u), f64(6083025255475019503u), f64(3587950130224573213u), f64(13544553895317881614u), f64(12150708944791209477u), f64(4740419680481906389u), f64(1941324043466159959u), f64(9897452317974437047u), f64(9477233337600713314u), f64(13752614432306922614u), f64(5813536777454517309u), f64(5288450412570259281u), f64(1967956071157440815u), f64(480961478938269997u), f64(8707019076078059829u), f64(11673730106455029722u)};
// -2.5729885582194777e+54, 6.923400865513078e-77, 3.399107850939321e+215, 8.989536893970336e-184, -1.3060543175303795e+63, -1520349.006217227, 3.8975989077786173e-149, -4.256422679919585e+181, 6.765582943934791e+289, 3.26810917231141e-70, -1.2024822587538113e-25, -4.373920489375915e+110, 1.8623719629834263e-295, 1.9729465817720833e+247, 4.0465318686762067e+287, 20.799593290256517, -2.396324201111364e+295, 4.6559448455176147e+98, 7.812372448435905e-69, -8.105054017547407e-20, -5.196060342532765e-113, 850733770.7895151, 6.542829947198155e-179, -1.3305439476683419e-263, -1.0971075979162187e-291, -6.4614803614591834e-06, 4.4214616749469996e+80, 3.630582797776415e+45, 3.8971392710020073e-177, 1.6200731142167154e-276, 1.1655694468313672e+274, -6.975510498054295e-145

uint64_t var_107 [] = {0u, 16u, 112u, 144u, 80u, 128u, 160u, 96u, 104u, 144u, 152u};
uint64_t var_105 [] = {17986676480636375055u, 16561927364456256080u, 9763399153731215526u, 17011184081394977003u, 5515476774723883814u, 9457918034744210939u, 10603173954453111005u, 9727232406901432990u};
float64_t var_104 [] = {f64(12268982402006723053u), f64(11633317314059709852u), f64(2355759684042883150u), f64(14916586682027897541u), f64(4523747090804693057u), f64(15313422636296236341u), f64(17852314057458726189u), f64(8061073346680340919u), f64(6867894128954076650u), f64(2628301318073533989u), f64(12362862840435370860u), f64(16892802778425156682u), f64(16296392560269218515u), f64(4745368322886126618u), f64(17610928996722171208u), f64(4227008168332205515u), f64(1428962108255738703u), f64(15685550439166428737u), f64(8721174667886690300u), f64(6389093016280766630u), f64(915371217100750559u), f64(13102858936469556209u), f64(14325435142686572326u), f64(13100434734449401387u), f64(16751290947827411650u), f64(3724510073455636207u), f64(13176800956873038965u), f64(17108147774706966430u), f64(10272006816657832197u), f64(2166840198934152003u), f64(3126865544766700112u), f64(3984812247613312207u)};
// -4.3997420428351414e-105, -1.3956358185785168e-147, 3.310721118965692e-151, -4.055255512934733e+72, 2.8107367677455903e-06, -1.381116861022872e+99, -6.669862983800634e+268, 7.443728610914345e+230, 1.2955022920890574e+151, 5.635813034929725e-133, -8.097801568182746e-99, -4.905389684561529e+204, -7.030330639871949e+164, 1807574008.0749574, -5.175243087826741e+252, 4.09547138485164e-26, 3.8417697419442373e-213, -1.0004951443736317e+124, 1.0315950134023327e+275, 1.3424750996227978e+119, 1.7924122107403712e-247, -2.427530528447156e-49, -1.2238156964734415e+33, -1.608666481217323e-49, -1.7914323013217252e+195, 1.0030184333291902e-59, -2.0599789696372077e-44, -1.2509772089128461e+219, -1.4156718618632884e-238, 7.88506931791861e-164, 1.1920211310425338e-99, 2.5915707560738407e-42

uint64_t var_102 [] = {112u, 48u, 48u, 112u, 16u, 80u, 8u, 24u};
uint64_t var_100 [] = {10626802471431058568u, 313001281678515730u, 16466312065171707704u, 13381090709553971058u, 7068539325592326040u, 15412762098109586724u, 14400127101981903638u};
float64_t var_99 [] = {f64(6187698773816159056u), f64(9776498580032912749u), f64(18269226306426886945u), f64(12870129469455680428u), f64(13778738789904889113u), f64(9289281457567414133u), f64(10639002934180078457u), f64(6195007212266041285u), f64(10865104194439280749u), f64(16359521363106835779u), f64(9698718863891634989u), f64(230294556184204338u), f64(10670676975682736140u), f64(11465988120190307059u), f64(8766120491176435417u), f64(2717371620507294816u), f64(3837943005303818774u), f64(5129903549361771946u), f64(17642969427757166520u), f64(4007765606054665226u), f64(1647027738278082783u), f64(10690219188164819365u), f64(15228053371569790034u), f64(10344531484724988184u), f64(710760864140949770u), f64(15615459515851778470u), f64(9927951426822621308u), f64(4957518843951601594u), f64(7641934781991551769u), f64(14252201341143490410u), f64(10241407868670681836u), f64(8264550004758408330u)};
// 4.4611521076973743e+105, -1.0758468088979187e-271, -5.176835491667206e+296, -6.619601004503453e-65, -0.00036489525625856136, -2.979938591390489e-304, -4.950276524818289e-214, 1.438417155610527e+106, -6.428709007665846e-199, -1.1649903737111477e+169, -6.986997664109763e-277, 2.845076679936957e-293, -6.493431793743611e-212, -8.925983327534352e-159, 1.0421560948913072e+278, 5.088139069960686e-127, 3.990899891565144e-52, 8.867669768871947e+34, -7.161121366560219e+254, 8.736910566334191e-41, 1.432775837239442e-198, -1.2969217753816681e-210, -2.726501234875066e+93, -9.799637510894672e-234, 3.7003426124468857e-261, -2.1441371653352167e+119, -1.4715600945527723e-261, 2.705444685329441e+23, 7.251873910049452e+202, -1.6087437906676368e+28, -1.2589823680213738e-240, 3.000901131149856e+244

uint64_t var_97 [] = {104u, 72u, 88u, 32u, 88u, 32u, 88u};
uint32_t var_95 [] = {348891937u, 929472872u};
uint32_t var_94 [] = {421885700u, 716864914u};
uint8_t var_93 [] = {113u, 117u};
uint8_t var_92 [] = {252u, 80u};
uint32_t var_91 [] = {264712479u, 258897714u};
uint64_t var_90 [] = {11601634546274575695u, 18347035667798957494u, 13451989570814404378u, 456765431602104240u, 10052606093390301498u, 2437914183458767421u, 1539225365727642394u};
float64_t var_89 [] = {f64(8962998596501157150u), f64(12224740054729171255u), f64(7400247037932311376u), f64(6032466557879469896u), f64(5839367888926184591u), f64(8111286762322373945u), f64(16464479761714716521u)};
// 1.4787708549897394e+291, -4.896911894513031e-108, 5.155727613246544e+186, 1.971320548368363e+95, 2.4285423850251372e+82, 1.6970322567792563e+234, -1.1706628228396777e+176

uint64_t var_88 [] = {1280769246380200460u, 14023690319710955253u, 13247119119507366408u, 7767358375133657872u, 10775238306218042028u, 6738525377594107821u, 1491839643263654600u, 17664969632620412524u};
float64_t var_87 [] = {f64(16304373337755592897u), f64(11714700170807880927u), f64(6682154344242841459u), f64(2775903153273905952u), f64(10892092016535164825u), f64(10642317549290804120u), f64(1322662639198263372u), f64(14850734564898422239u)};
// -2.392147333909853e+165, -3.8902382189730166e-142, 5.1692570320776656e+138, 4.157950442133119e-123, -4.094307940460808e-197, -7.93999595358361e-214, 2.992377506963031e-220, -1.6450844168843733e+68

uint64_t var_86 [] = {15401374053725176026u, 15159971213859746434u, 15996044860437899159u, 10024478611893507882u, 17312537202804145326u, 13286484462280416129u, 5653446096408344014u, 1589575317326256715u, 1256266953356618611u, 5852474666329482281u, 8236254411035780100u};
float64_t var_85 [] = {f64(2300673783722258919u), f64(2345233581152263551u), f64(14429124106383846452u), f64(3279288055166527134u), f64(8511350044499123665u), f64(16292164099935306522u), f64(319680150795867099u), f64(18041792644663864377u), f64(16469534312049021029u), f64(3059779533977631997u), f64(18445158560006719557u)};
// 6.9071744526170175e-155, 6.6691704148362785e-152, -1.0394464537775981e+40, 1.803751188554954e-89, 9.34508116044192e+260, -3.6558900666657e+164, 2.604906894646101e-287, -3.1445758471145765e+281, -2.4962431291505274e+176, 3.927729333541973e-104, nan

uint64_t var_84 [] = {1922947498486410857u, 16951704142057129556u, 5388344212325078882u, 5448258397625405342u, 1703194961276899101u, 9646419556518517367u, 16273654145082032307u, 3551281066422685306u, 10669730802081923878u, 8008712173776472922u, 5606540849143552077u, 17565367472452138214u, 2648738666890881870u, 2998195017278264134u, 2065211952219847096u, 18169511136733880014u};
float64_t var_83 [] = {f64(17659590213590998702u), f64(15964857636350179584u), f64(8302093860329706908u), f64(10398331577184171968u), f64(12870065405008121229u), f64(17372403961692722173u), f64(8998427184828039058u), f64(5686275838205320814u), f64(13460688990661137053u), f64(6406040773904701275u), f64(4003950932397511804u), f64(4490512161850491093u), f64(4134453016479536686u), f64(6870297400155601231u), f64(8215692482325664371u), f64(7870459193100051406u)};
// -9.134636827727053e+255, -4.659469567024577e+142, 1.0034462709138033e+247, -3.902670263958087e-230, -6.56557059752003e-65, -5.895455415968474e+236, 3.3600664624367966e+293, 1.4195485286653068e+72, -1.937018094571547e-25, 1.8421677350513206e+120, 4.847550088368645e-41, 1.6301853792973474e-08, 2.5465086384920936e-32, 1.9803645707206568e+151, 1.6672588912213468e+241, 1.405831311328661e+218

uint64_t var_82 [] = {12379964670789626889u, 13436296814432206489u, 4659013173841538396u, 6196494238874966981u, 6730617828857168898u, 8296127193014163030u};
float64_t var_81 [] = {f64(5926993659265029067u), f64(8958927779396419652u), f64(8685124436318166792u), f64(11316065961843452656u), f64(15659828517197660337u), f64(4094401123071894779u)};
// 1.682091154120208e+88, 7.99321150819077e+290, 4.0167944017799746e+272, -8.857511628261053e-169, -1.950836146289716e+122, 5.48735852433986e-35

uint64_t var_80 [] = {5797432571859372123u};
float64_t var_79 [] = {f64(13033630116291160682u)};
// -5.466899617959982e-54

int64_t var_78 [] = {2849001835972176893, -2276289870088551842};
int8_t var_77 [] = {-108, -85};
int64_t var_76 [] = {-3392684395881840317, -432674178100072972};
int64_t var_75 [] = {-2471443589070945772, -1223946683744760999};
uint64_t var_74 [] = {7116734975071460602u, 12441625220002408740u, 2786654427366964938u, 13829764222335973050u, 4762436333200299360u, 11395394244577961796u, 8944665256461003018u};
float64_t var_73 [] = {f64(15952028776600326978u), f64(12864696557087944463u), f64(13582857386678690992u), f64(7739823830606308283u), f64(13001311485958403778u), f64(8118890521685184687u), f64(1188717030892108108u)};
// -6.47474704450738e+141, -2.917920547847776e-65, -2.776118718553634e-17, 2.6072597653252422e+209, -3.815411605227875e-56, 5.58278810533562e+234, 3.2124591426633125e-229

uint64_t var_72 [] = {2155812772479464600u, 3170612777771403051u, 1681134711641298423u, 8156897225395734044u, 17350593003963396555u, 15651252092822179957u, 8877172929823018199u, 10633166244594888701u};
float64_t var_71 [] = {f64(12474830707341252676u), f64(12421407484372908519u), f64(7665390200081996829u), f64(14912023387491630955u), f64(8690015471402520246u), f64(6013523761173124620u), f64(2092709178248925316u), f64(7713300078375383008u)};
// -2.4163244305828842e-91, -6.630856606830876e-95, 2.655056358903447e+204, -2.004208317953279e+72, 8.498977317699981e+272, 1.0600734757558945e+94, 8.875460424589357e-169, 4.3577672186230927e+207

uint32_t var_70 [] = {2502819275u, 3673473517u};
uint32_t var_69 [] = {1873404000u, 3357537829u};
uint32_t var_68 [] = {2249156032u, 2604756913u};
uint64_t var_67 [] = {11686681238268631633u, 17217089646841358934u, 4917608144621004608u, 2800347682848664045u, 13139172270140052590u};
float64_t var_66 [] = {f64(4221934333025834245u), f64(1969861952934041841u), f64(4767317979525211814u), f64(7255389652619498159u), f64(8348926455447142360u)};
// 1.8840860783382415e-26, 5.516797633442999e-177, 53506266743.450485, 1.0334573093152597e+177, 1.3131484076683424e+250

uint64_t var_65 [] = {3413955448782123771u, 5770915057685053022u};
float32_t var_64 [] = {f32(4282101878u), f32(2381220644u)};
// -2.4948187278321463e+38, -1.470137934347422e-30

float64_t var_63 [] = {f64(830612788143373984u), f64(2638216761943270681u)};
// 3.9093685536053063e-253, 2.5384472749053234e-132

float64_t var_62 [] = {f64(15227122452644821730u)};
// -2.2953290459445205e+93

int64_t var_61 = 203570955881815601;
uint32_t var_60 = 2733979527u;
int8_t var_59 = -26;
uint32_t var_58 = 1897186263u;
__riscv_vsetvl_e64m8(2);
vuint64m8_t var_144 = __riscv_vle64_v_u64m8(var_145, 2);
// 16, 16

__riscv_vsetvl_e64m8(1);
vuint64m8_t var_141 = __riscv_vle64_v_u64m8(var_142, 1);
// 8

__riscv_vsetvl_e64m8(5);
vuint64m8_t var_136 = __riscv_vle64_v_u64m8(var_137, 5);
// 72, 16, 0, 0, 40

__riscv_vsetvl_e64m8(8);
vuint64m8_t var_131 = __riscv_vle64_v_u64m8(var_132, 8);
// 80, 32, 72, 48, 64, 56, 48, 40

__riscv_vsetvl_e64m8(7);
vuint64m8_t var_126 = __riscv_vle64_v_u64m8(var_127, 7);
// 8, 104, 32, 8, 32, 56, 32

__riscv_vsetvl_e64m8(1);
vuint64m8_t var_121 = __riscv_vle64_v_u64m8(var_122, 1);
// 0

__riscv_vsetvl_e64m8(6);
vuint64m8_t var_116 = __riscv_vle64_v_u64m8(var_117, 6);
// 48, 48, 56, 80, 32, 16

__riscv_vsetvl_e64m8(16);
vuint64m8_t var_111 = __riscv_vle64_v_u64m8(var_112, 16);
// 48, 144, 144, 216, 24, 120, 128, 16, 128, 112, 192, 216, 176, 176, 152, 48

__riscv_vsetvl_e64m8(11);
vuint64m8_t var_106 = __riscv_vle64_v_u64m8(var_107, 11);
// 0, 16, 112, 144, 80, 128, 160, 96, 104, 144, 152

__riscv_vsetvl_e64m8(8);
vuint64m8_t var_101 = __riscv_vle64_v_u64m8(var_102, 8);
// 112, 48, 48, 112, 16, 80, 8, 24

__riscv_vsetvl_e64m8(7);
vuint64m8_t var_96 = __riscv_vle64_v_u64m8(var_97, 7);
// 104, 72, 88, 32, 88, 32, 88

__riscv_vsetvl_e32m2(2);
vuint32m2_t var_21 = __riscv_vle32_v_u32m2(var_95, 2);
// 348891937, 929472872

vuint32m2_t var_22 = __riscv_vle32_v_u32m2(var_94, 2);
// 421885700, 716864914

vuint8mf2_t var_23 = __riscv_vle8_v_u8mf2(var_93, 2);
// 113, 117

vuint8mf2_t var_24 = __riscv_vle8_v_u8mf2(var_92, 2);
// 252, 80

vuint32m2_t var_25 = __riscv_vle32_v_u32m2(var_91, 2);
// 264712479, 258897714

__riscv_vsetvl_e64m8(7);
vuint64m8_t var_26 = __riscv_vle64_v_u64m8(var_90, 7);
// 11601634546274575695, 18347035667798957494, 13451989570814404378, 456765431602104240, 10052606093390301498, 2437914183458767421, 1539225365727642394

vfloat64m8_t var_27 = __riscv_vle64_v_f64m8(var_89, 7);
// 1.4787708549897394e+291, -4.896911894513031e-108, 5.155727613246544e+186, 1.971320548368363e+95, 2.4285423850251372e+82, 1.6970322567792563e+234, -1.1706628228396777e+176

__riscv_vsetvl_e64m8(8);
vuint64m8_t var_28 = __riscv_vle64_v_u64m8(var_88, 8);
// 1280769246380200460, 14023690319710955253, 13247119119507366408, 7767358375133657872, 10775238306218042028, 6738525377594107821, 1491839643263654600, 17664969632620412524

vfloat64m8_t var_29 = __riscv_vle64_v_f64m8(var_87, 8);
// -2.392147333909853e+165, -3.8902382189730166e-142, 5.1692570320776656e+138, 4.157950442133119e-123, -4.094307940460808e-197, -7.93999595358361e-214, 2.992377506963031e-220, -1.6450844168843733e+68

__riscv_vsetvl_e64m8(11);
vuint64m8_t var_30 = __riscv_vle64_v_u64m8(var_86, 11);
// 15401374053725176026, 15159971213859746434, 15996044860437899159, 10024478611893507882, 17312537202804145326, 13286484462280416129, 5653446096408344014, 1589575317326256715, 1256266953356618611, 5852474666329482281, 8236254411035780100

vfloat64m8_t var_31 = __riscv_vle64_v_f64m8(var_85, 11);
// 6.9071744526170175e-155, 6.6691704148362785e-152, -1.0394464537775981e+40, 1.803751188554954e-89, 9.34508116044192e+260, -3.6558900666657e+164, 2.604906894646101e-287, -3.1445758471145765e+281, -2.4962431291505274e+176, 3.927729333541973e-104, nan

__riscv_vsetvl_e64m8(16);
vuint64m8_t var_32 = __riscv_vle64_v_u64m8(var_84, 16);
// 1922947498486410857, 16951704142057129556, 5388344212325078882, 5448258397625405342, 1703194961276899101, 9646419556518517367, 16273654145082032307, 3551281066422685306, 10669730802081923878, 8008712173776472922, 5606540849143552077, 17565367472452138214, 2648738666890881870, 2998195017278264134, 2065211952219847096, 18169511136733880014

vfloat64m8_t var_33 = __riscv_vle64_v_f64m8(var_83, 16);
// -9.134636827727053e+255, -4.659469567024577e+142, 1.0034462709138033e+247, -3.902670263958087e-230, -6.56557059752003e-65, -5.895455415968474e+236, 3.3600664624367966e+293, 1.4195485286653068e+72, -1.937018094571547e-25, 1.8421677350513206e+120, 4.847550088368645e-41, 1.6301853792973474e-08, 2.5465086384920936e-32, 1.9803645707206568e+151, 1.6672588912213468e+241, 1.405831311328661e+218

__riscv_vsetvl_e64m8(6);
vuint64m8_t var_34 = __riscv_vle64_v_u64m8(var_82, 6);
// 12379964670789626889, 13436296814432206489, 4659013173841538396, 6196494238874966981, 6730617828857168898, 8296127193014163030

vfloat64m8_t var_35 = __riscv_vle64_v_f64m8(var_81, 6);
// 1.682091154120208e+88, 7.99321150819077e+290, 4.0167944017799746e+272, -8.857511628261053e-169, -1.950836146289716e+122, 5.48735852433986e-35

__riscv_vsetvl_e64m8(1);
vuint64m8_t var_36 = __riscv_vle64_v_u64m8(var_80, 1);
// 5797432571859372123

vfloat64m8_t var_37 = __riscv_vle64_v_f64m8(var_79, 1);
// -5.466899617959982e-54

__riscv_vsetvl_e64m8(2);
vint64m8_t var_40 = __riscv_vle64_v_i64m8(var_78, 2);
// 2849001835972176893, -2276289870088551842

vint8m1_t var_41 = __riscv_vle8_v_i8m1(var_77, 2);
// -108, -85

vint64m8_t var_42 = __riscv_vle64_v_i64m8(var_76, 2);
// -3392684395881840317, -432674178100072972

vint64m8_t var_43 = __riscv_vle64_v_i64m8(var_75, 2);
// -2471443589070945772, -1223946683744760999

__riscv_vsetvl_e64m8(7);
vuint64m8_t var_44 = __riscv_vle64_v_u64m8(var_74, 7);
// 7116734975071460602, 12441625220002408740, 2786654427366964938, 13829764222335973050, 4762436333200299360, 11395394244577961796, 8944665256461003018

vfloat64m8_t var_45 = __riscv_vle64_v_f64m8(var_73, 7);
// -6.47474704450738e+141, -2.917920547847776e-65, -2.776118718553634e-17, 2.6072597653252422e+209, -3.815411605227875e-56, 5.58278810533562e+234, 3.2124591426633125e-229

__riscv_vsetvl_e64m8(8);
vuint64m8_t var_46 = __riscv_vle64_v_u64m8(var_72, 8);
// 2155812772479464600, 3170612777771403051, 1681134711641298423, 8156897225395734044, 17350593003963396555, 15651252092822179957, 8877172929823018199, 10633166244594888701

vfloat64m8_t var_47 = __riscv_vle64_v_f64m8(var_71, 8);
// -2.4163244305828842e-91, -6.630856606830876e-95, 2.655056358903447e+204, -2.004208317953279e+72, 8.498977317699981e+272, 1.0600734757558945e+94, 8.875460424589357e-169, 4.3577672186230927e+207

__riscv_vsetvl_e32m4(2);
vuint32m4_t var_48 = __riscv_vle32_v_u32m4(var_70, 2);
// 2502819275, 3673473517

vuint32m4_t var_50 = __riscv_vle32_v_u32m4(var_69, 2);
// 1873404000, 3357537829

vuint32m4_t var_51 = __riscv_vle32_v_u32m4(var_68, 2);
// 2249156032, 2604756913

__riscv_vsetvl_e64m8(5);
vuint64m8_t var_52 = __riscv_vle64_v_u64m8(var_67, 5);
// 11686681238268631633, 17217089646841358934, 4917608144621004608, 2800347682848664045, 13139172270140052590

vfloat64m8_t var_53 = __riscv_vle64_v_f64m8(var_66, 5);
// 1.8840860783382415e-26, 5.516797633442999e-177, 53506266743.450485, 1.0334573093152597e+177, 1.3131484076683424e+250

__riscv_vsetvl_e64m8(2);
vuint64m8_t var_54 = __riscv_vle64_v_u64m8(var_65, 2);
// 3413955448782123771, 5770915057685053022

vfloat32m4_t var_55 = __riscv_vle32_v_f32m4(var_64, 2);
// -2.4948187278321463e+38, -1.470137934347422e-30

vfloat64m8_t var_56 = __riscv_vle64_v_f64m8(var_63, 2);
// 3.9093685536053063e-253, 2.5384472749053234e-132

__riscv_vsetvl_e64m8(1);
vfloat64m8_t var_57 = __riscv_vle64_v_f64m8(var_62, 1);
// -2.2953290459445205e+93

__riscv_vsetvl_e8mf2(2);
vbool16_t var_20 = __riscv_vmsleu_vv_u8mf2_b16(var_23, var_24, 2);
// 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0

uint32_t var_18 = __riscv_vmv_x_s_u32m2_u32(var_25);
// 264712479

__riscv_vsetvl_e64m8(7);
__riscv_vse64_v_u64m8(var_100, var_26, 7);
__riscv_vsoxei64_v_f64m8(var_99, var_96, var_27, 7);
__riscv_vsetvl_e64m8(8);
__riscv_vse64_v_u64m8(var_105, var_28, 8);
__riscv_vsoxei64_v_f64m8(var_104, var_101, var_29, 8);
__riscv_vsetvl_e64m8(11);
__riscv_vse64_v_u64m8(var_110, var_30, 11);
__riscv_vsoxei64_v_f64m8(var_109, var_106, var_31, 11);
__riscv_vsetvl_e64m8(16);
__riscv_vse64_v_u64m8(var_115, var_32, 16);
__riscv_vsoxei64_v_f64m8(var_114, var_111, var_33, 16);
__riscv_vsetvl_e64m8(6);
__riscv_vse64_v_u64m8(var_120, var_34, 6);
__riscv_vsoxei64_v_f64m8(var_119, var_116, var_35, 6);
__riscv_vsetvl_e64m8(1);
__riscv_vse64_v_u64m8(var_125, var_36, 1);
__riscv_vsoxei64_v_f64m8(var_124, var_121, var_37, 1);
__riscv_vsetvl_e8m1(2);
vbool8_t var_39 = __riscv_vmsgt_vx_i8m1_b8(var_41, var_59, 2);
// 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0

vbool8_t var_38 = __riscv_vmsge_vv_i64m8_b8(var_42, var_43, 2);
// 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0

__riscv_vsetvl_e64m8(7);
__riscv_vse64_v_u64m8(var_130, var_44, 7);
__riscv_vsoxei64_v_f64m8(var_129, var_126, var_45, 7);
__riscv_vsetvl_e64m8(8);
__riscv_vse64_v_u64m8(var_135, var_46, 8);
__riscv_vsoxei64_v_f64m8(var_134, var_131, var_47, 8);
__riscv_vsetvl_e32m4(2);
vfloat64m8_t var_6 = __riscv_vfwcvt_f_xu_v_f64m8(var_48, 2);
// 2502819275.0, 3673473517.0

vbool8_t var_49 = __riscv_vmsbc_vx_u32m4_b8(var_51, var_58, 2);
// 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0

__riscv_vsetvl_e64m8(5);
__riscv_vse64_v_u64m8(var_140, var_52, 5);
__riscv_vsoxei64_v_f64m8(var_139, var_136, var_53, 5);
__riscv_vsetvl_e64m8(1);
__riscv_vsoxei64_v_f64m8(var_143, var_141, var_57, 1);
__riscv_vsetvl_e32m2(2);
vuint32m2_t var_19 = __riscv_vdivu_vx_u32m2_mu(var_20, var_21, var_22, var_60, 2);
// 0, 929472872

if(!check(var_100, var_183, var_184)) {cerr << "check 182 fails" << endl; return_value = 1;}
if(!check(var_99, var_180, var_181)) {cerr << "check 179 fails" << endl; return_value = 1;}
if(!check(var_105, var_186, var_187)) {cerr << "check 185 fails" << endl; return_value = 1;}
if(!check(var_104, var_177, var_178)) {cerr << "check 176 fails" << endl; return_value = 1;}
if(!check(var_110, var_189, var_190)) {cerr << "check 188 fails" << endl; return_value = 1;}
if(!check(var_109, var_174, var_175)) {cerr << "check 173 fails" << endl; return_value = 1;}
if(!check(var_115, var_192, var_193)) {cerr << "check 191 fails" << endl; return_value = 1;}
if(!check(var_114, var_171, var_172)) {cerr << "check 170 fails" << endl; return_value = 1;}
if(!check(var_120, var_195, var_196)) {cerr << "check 194 fails" << endl; return_value = 1;}
if(!check(var_119, var_168, var_169)) {cerr << "check 167 fails" << endl; return_value = 1;}
if(!check(var_125, var_198, var_199)) {cerr << "check 197 fails" << endl; return_value = 1;}
if(!check(var_124, var_165, var_166)) {cerr << "check 164 fails" << endl; return_value = 1;}
__riscv_vsetvl_e64m8(2);
vbool8_t var_10 = __riscv_vmslt_vx_i64m8_b8_mu(var_38, var_39, var_40, var_61, 2);
// 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0

if(!check(var_130, var_201, var_202)) {cerr << "check 200 fails" << endl; return_value = 1;}
if(!check(var_129, var_162, var_163)) {cerr << "check 161 fails" << endl; return_value = 1;}
if(!check(var_135, var_204, var_205)) {cerr << "check 203 fails" << endl; return_value = 1;}
if(!check(var_134, var_159, var_160)) {cerr << "check 158 fails" << endl; return_value = 1;}
if(!check(var_140, var_207, var_208)) {cerr << "check 206 fails" << endl; return_value = 1;}
if(!check(var_139, var_156, var_157)) {cerr << "check 155 fails" << endl; return_value = 1;}
if(!check(var_143, var_153, var_154)) {cerr << "check 152 fails" << endl; return_value = 1;}
__riscv_vsetvl_e32m2(2);
vuint32m2_t var_13 = __riscv_vmaxu_vx_u32m2(var_19, var_18, 2);
// 264712479, 929472872

uint32_t var_7 = __riscv_vmv_x_s_u32m2_u32(var_13);
// 264712479

__riscv_vsetvl_e32m4(2);
vbool8_t var_5 = __riscv_vmsleu_vx_u32m4_b8_mu(var_10, var_49, var_50, var_7, 2);
// 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0

vuint64m8_t var_3 = __riscv_vfwcvt_xu_f_v_u64m8_mu(var_5, var_54, var_55, 2);
// 3413955448782123771, 5770915057685053022

vfloat64m8_t var_2 = __riscv_vfmul_vv_f64m8_mu(var_5, var_6, var_6, var_56, 2);
// 2502819275.0, 3673473517.0

__riscv_vse64_v_u64m8(var_148, var_3, 2);
__riscv_vsoxei64_v_f64m8(var_147, var_144, var_2, 2);
if(!check(var_148, var_210, var_211)) {cerr << "check 209 fails" << endl; return_value = 1;}
if(!check(var_147, var_150, var_151)) {cerr << "check 149 fails" << endl; return_value = 1;}
if (return_value)
  __builtin_abort ();
return return_value;
}
